% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/mtlr_cv.R
\name{mtlr_cv}
\alias{mtlr_cv}
\title{MTLR Internal Cross-Validation for Selecting C1.}
\usage{
mtlr_cv(formula, data, time_points = NULL, nintervals = NULL,
  normalize = T, C1_vec = c(0.001, 0.01, 0.1, 1, 10, 100, 1000),
  train_biases = T, seed_weights = NULL, previous_weights = T,
  loss = c("ll"), nfolds = 5, foldtype = c("fullstrat",
  "censorstrat", "random"), verbose = FALSE, threshold = 1e-05,
  maxit = 5000, lower = -15, upper = 15)
}
\arguments{
\item{formula}{a formula object with the response to the left of the "~" operator. The response must be a survival object returned
by the \code{\link[survival]{Surv}} function.}

\item{data}{a data.frame containing the features for survival prediction. These must be variables corresponding to the formula object.}

\item{time_points}{the time points for MTLR to create weights. If left as NULL, the time_points chosen will be based on equally spaced quantiles
of the survival times. In the case of interval censored data note that only the start time is considered and not the end time for selecting time points.
It is strongly recommended to specify time points if your data is heavily interval censored. If time_points is not NULL then nintervals is ignored.}

\item{nintervals}{Number of time intervals to use for MTLR. Note the number of time points will be nintervals + 1. If left as NULL
a default of sqrt(N) is used where N is the number of observations in the supplied dataset. This parameter is ignored if time_points is specified.}

\item{normalize}{if TRUE, variables will be normalized (mean 0, standard deviation of 1). This is STRONGLY suggested. If normalization
does not occur it is much more likely that MTLR will fail to converge. Additionally, if FALSE consider adjusting "lower" and "upper"
used for L-BFGS-B optimization.}

\item{C1_vec}{a vector of regularization parameters to test. All values must be non-negative. For large datasets you may want to reduce the number
of value tried to increase efficiency. Similarly for nfolds.}

\item{train_biases}{if TRUE, biases will be trained before feature weights (and again trained while training feature weights). This
has shown to speed up total training time.}

\item{seed_weights}{the initialization weights for the biases and the features. If left as NULL all weights are initialized to zero. If seed_weights are
specified then either nintervals or time_points must also be specified. The length of seed_weights should correspond to (number of features + 1)*(length of
time_points) = (number of features + 1)*(nintervals + 1).}

\item{previous_weights}{a boolean specifying if sequential folds should use the previous fold's parameters as seed_weights. Doing this will likely
speed up the computation time for cross-validation as we are providing weights which are (likely) close to the optimal weights. Note that this is
done separately for each value of C1 so there is no parameter sharing between different values of C1, and instead only across the same value of C1.}

\item{loss}{a string indicating the loss to optimize for which to choose the regularization parameter. Currently only the log-likelihood (ll)
is supported.}

\item{nfolds}{the number of internal cross validation folds, default is 5.}

\item{foldtype}{type of cross validation folds. Full stratification, "fullstrat", sorts observations by their event time and their event indicators
and numbers them off into folds. This effectively give each fold approximately the same number of uncensored observations as well as keeps the range
of time points as equivalent as possible across folds. This type of cross-validation is completely deterministic.
Censored stratification, "censorstrat", will put approximately the same number of uncensored observations in each fold but not pay any attention to
event time. This is partially stochastic. The totally random cross-validation, "random", randomly assigns observations to folds without considering
event time nor event status.}

\item{verbose}{if TRUE the progress will be printed for every completed value of C1.}

\item{threshold}{The threshold for the convergence tolerance (in the objective function) when training the feature weights.
This threshold will be passed to \link[stats]{optim}.}

\item{maxit}{The maximum iterations to run for MTLR. This parameter will be passed to \link[stats]{optim}.}

\item{lower}{The lower bound for L-BFGS-B optimization. This parameter will be passed to \link[stats]{optim}.}

\item{upper}{The upper bound for L-BFGS-B optimization. This parameter will be passed to \link[stats]{optim}.}
}
\value{
Performing mtlr_cv will return the following:
\itemize{
  \item best_C1: The value of C1 which achieved the best (lowest) loss.
  \item avg_loss: The averaged value of loss across the five folds for each value of C1 tested.
  }
}
\description{
MTLR Internal Cross-Validation for Selecting C1.
}
\details{
Currently only the log-likelihood loss is supported for optimizing C1. Here the loss considers censored and uncensored observations differently.
For uncensored observations, we assign a loss of the negative log probability assigned to the interval in which the observation had their event, \emph{e.g.}
if an observation had a 20% chance of having its event between timepoint1 and timepoint2 and it did have it's event in that interval then the loss
is -log(0.2). We want these probabilities to be large so we would normally want to maximize this value (since logs of probabilities are negative)
but we take the negative and instead minimize the value, thus we want the lowest loss. For censored observations we take the log of the probability
of survival at the time of censoring, \emph{e.g.} if an observation is censored at time = 42 we take the negative log of the survival probability assigned
to time 42 as the loss.
}
\examples{
library(survival)
cv_mod <- mtlr_cv(Surv(time,status)~., data = lung)
#Note the best C1 also corresponds to the lost average loss:
cv_mod
}
\seealso{
\code{\link[MTLR]{mtlr}}
}
