\name{mc.irf}
\alias{mc.irf}
\alias{mc.irf.VAR}
\alias{mc.irf.BVAR}
\alias{mc.irf.BSVAR}
%- Also NEED an '\alias' for EACH other topic documented here.
\title{ Monte Carlo Integration / Simulation of Impulse Response
  Functions }
\description{
  Simulates a posterior of impulse response functions (IRF) by
  Monte Carlo integration.  This can handle Bayesian and frequentist
  VARs and Bayesian (structural) VARs estimated with the \code{szbvar},
  \code{szbsvar} or \code{reduced.form.var} functions.  The decomposition of the
  contemporaneous innovations is handled by a Cholesky decomposition of
  the error covariance matrix in reduced form (B)VAR object, or for the
  contemporaneous structure in S-VAR models.  Simulations of IRFs
  from the Bayesian model utilize the posterior estimates for that model.
}
\usage{
mc.irf(varobj, nsteps, draws=0, A0.posterior=NULL,
       sign.list=rep(1, ncol(varobj$Y)))

mc.irf.VAR(varobj, nsteps, draws)

mc.irf.BVAR(varobj, nsteps, draws)

mc.irf.BSVAR(varobj, nsteps, A0.posterior, sign.list)
}
%- maybe also 'usage' for other objects documented here.
\arguments{
  \item{varobj}{ VAR objects for a fitted VAR model from either
    \code{reduced.form.var}, \code{szbvar} or \code{szbsvar}.}
  \item{nsteps}{ Number of periods over which to compute the impulse responses}
  \item{draws}{ Number of draws for the simulation of the posterior
    distribution of the IRFs (if not a \code{szbsvar} object}
  \item{A0.posterior}{Posterior sample objects generated by
    \code{gibbs.A0()} for B-SVAR models, based on the structural identification in
    \code{varobj$ident}. }
  \item{sign.list}{ A list of signs (length = number of variables) for
    normalization given as either 1 or -1. }
}
\details{
  VAR/BVAR:
  
  Draws a set of posterior samples from the VAR coefficients and
  computes impulse responses for each sample.  These samples can then be
  summarized to compute MCMC-based estimates of the responses using the
  error band methods described in Sims and Zha (1999).

  B-SVAR:
  Generates a set of \code{N2} draws from the impulse responses for the
  Bayesian SVAR model in \code{varobj}.  The function takes as its
  arguments the posterior moments of the B-SVAR model in \code{varobj},
  the draws of the contemporaneous structural coefficients
  \eqn{A_0}{A(0)} from \code{gibbs.A0}, and a list of signs for
  normalization. This function then computes a posterior sample of the
  impulse responses based on the Schur product of the sign list and the
  draws of \eqn{A_0}{A(0)} and draws from the normal posterior pdf for
  the other coefficients in the model. 

  The computations are done using compiled C++ code as of version
  0.3.0.  See the package source code for details about the
  implementation.
  
}
\value{
  VAR/BVAR:
  
  An mc.irf.VAR/mc.irf.BVAR class object object that is the array of
  impulse response samples for the Monte Carlo samples

  \item{impulse}{\eqn{draws \times nsteps \times m^2}{ draws X nsteps X
      (m*m)} array of the impulse responses }

  B-SVAR:
  mc.irf.BSVAR object which is an \eqn{(N2, nsteps, m^2)} array of the impulse
  responses for the associated B-SVAR model in \code{varobj} and the posterior
  \eqn{A_0}{A(0)}.
  
}

\references{
  Brandt, Patrick T. and John R. Freeman. 2006. "Advances in Bayesian
  Time Series Modeling and the Study of Politics: Theory Testing,
  Forecasting, and Policy Analysis"
   \emph{Political Analysis} 14(1):1-36.
   
  Sims, C.A. and Tao Zha. 1999. "Error Bands for Impulse
  Responses." \emph{Econometrica} 67(5): 1113-1156.

  Hamilton, James. 1994.  Time Series Analysis.  Chapter 11.

  Waggoner, Daniel F. and Tao A. Zha.  2003. "A Gibbs sampler for
  structural vector autoregressions" \emph{Journal of Economic Dynamics
  \& Control}. 28:349--366.

}
\author{ Patrick T. Brandt }
\note{ Users need to think carefully about the number of steps and the
  size of the posterior sample in \eqn{A_0}{A(0)}, since enough memory
  needs to be available to store and process the posterior of the
  impulse responses.   The number of bytes consumed by the impulse
  responses will be approximately \eqn{m^2 \times nsteps \times N2
    \times 16}{m^2 x nsteps x N2 x 16} where \code{N2} is the number of
  draws of \eqn{A_0}{A(0)} from the \code{gibbs.A0}.  Be sure
  you have enough memory available to store the object you create! }

\seealso{ See also as \code{\link{plot.mc.irf}} for plotting methods
  and error band construction for the posterior of the impulse response
  functions, \code{\link{szbsvar}} for estimation of the
  posterior moments of the B-SVAR model,  \code{\link{gibbs.A0}}
  for drawing posterior samples of \eqn{A_0}{A(0)} for the B-SVAR model
  before the IRF computations, and \code{\link{plot.mc.irf}} for a
  plotting method for the posterior of the impulse responses.
}
\examples{
# Example 1
data(IsraelPalestineConflict)
varnames <- colnames(IsraelPalestineConflict)

fit.BVAR <- szbvar(Y=IsraelPalestineConflict, p=6, z=NULL,
                   lambda0=0.6, lambda1=0.1,
                   lambda3=2, lambda4=0.25, lambda5=0, mu5=0,
                   mu6=0, nu=3, qm=4,
                   prior=0, posterior.fit=FALSE)

# Draw from the posterior pdf of the impulse responses.
posterior.impulses <- mc.irf(fit.BVAR, nsteps=10, draws=5000)

# Plot the responses
plot(posterior.impulses, method=c("Sims-Zha2"), component=1,
                 probs=c(0.16,0.84), varnames=varnames)

# Example 2
ident <- diag(2)
varobj <- szbsvar(Y=IsraelPalestineConflict, p=6, z = NULL,
                  lambda=0.6, lambda1=0.1, lambda3=2, lambda4=0.25,
                  lambda5=0, mu5=0, mu6=0, ident, qm = 4)

A0.posterior <- gibbs.A0(varobj, N1=1000, N2=1000)

# Note you need to explcitly reference the sampled A0.posterior object
# in the following call for R to find it in the namespace!

impulse.sample <- mc.irf(varobj, nsteps=12, A0.posterior=A0.posterior)
}
\keyword{ ts}
\keyword{ models}
\keyword{ regression }
