% Generated by roxygen2 (4.1.1): do not edit by hand
% Please edit documentation in R/univDecomp.R
\name{fcptpaBasis}
\alias{fcptpaBasis}
\title{Calculate a smooth PCA representation for functional data on two-dimensional
domains}
\usage{
fcptpaBasis(funDataObject, npc, smoothingDegree = rep(2, 2), alphaRange,
  orderValues = TRUE, normalize = FALSE)
}
\arguments{
\item{funDataObject}{An object of class \code{\link[funData]{funData}}
containing the observed functional data samples (here: images) for which the
smooth PCA is to be calculated.}

\item{npc}{An integer, giving the number of principal components to be
calculated.}

\item{smoothingDegree}{A numeric vector of length 2, specifying the degree of
the difference penalties inducing smoothness in both directions of the
image. Defaults to \code{2} for each direction (2nd differences).}

\item{alphaRange}{A list of length 2 with entries \code{v} and \code{w}
containing the range of smoothness parameters to test for each direction.}

\item{orderValues}{Logical. If \code{TRUE}, the eigenvalues are ordered
decreasingly, together with their associated eigenimages and scores.
Defaults to \code{TRUE}.}

\item{normalize}{Logical. If \code{TRUE} the eigenfunctions are normalized to
 have norm 1. Defaults to \code{FALSE}.}
}
\value{
\item{scores}{A matrix of scores (coefficients) with dimension \code{N
 x npc}, reflecting the weights for principal component in each observation.}
 \item{B}{A matrix containing the scalar product of all pairs of basis
 functions.} \item{ortho}{Logical, indicating whether the eigenfunctions are
 orthonormal. Set to \code{normalize}, as this influences whether a
 normalization is done or not.} \item{functions}{A functional data object,
 representing the functional principal component basis functions.}
 \item{values}{A vector of length \code{npc}, containing the eigenvalues in
 decreasing order.}
}
\description{
This function calculates a smooth PCA representation based on the FCP_TPA
algorithm (see References) for functional data on two-dimensional domains. In
this case, the data can be interpreted as images with \code{S1 x S2} pixels
(assuming \code{nObsPoints(funDataObject) = (S1, S2)}), i.e. the total data
for \code{N} observations can be represented as third order tensor of
dimension \code{N x S1 x S2}.
}
\details{
The smooth PCA of the tensor data is calculated via the \code{\link{FCP_TPA}}
function. Smoothness is induced by difference penalty matrices for both
directions of the images, weighted by smoothing parameters \eqn{\alpha_v,
\alpha_w}. The resulting eigenvectors can be interpreted in terms of
eigenfunctions and individual scores for each observation. See
\code{\link{FCP_TPA}} for details.
}
\examples{
# simulate image data for N = 100 observations
N <- 100
b1 <- eFun(seq(0,1,0.01), M = 7, type = "Poly")
b2 <- eFun(seq(-pi, pi, 0.03), M = 8, type = "Fourier")
b <- tensorProduct(b1,b2) # 2D basis functions
scores <- matrix(rnorm(N*56), nrow = N)

# calculate observations (= linear combination of basis functions)
f <- expandBasisFunction(scores = scores, functions = b)

# calculate basis functions based on FCP_TPA algorithm (needs some time)
\donttest{
fcptpa <- fcptpaBasis(f, npc = 5, alphaRange = list(v = c(1e-5, 1e5), w = c(1e-5, 1e5)))

oldpar <- par(no.readonly = TRUE)

for(i in 1:5) # plot all 5 basis functions
plot(fcptpa$functions, obs = i, main = paste("Basis function", i)) # plot first basis function

par(oldpar)}
}
\references{
G. I. Allen, "Multi-way Functional Principal Components Analysis",
 In IEEE International Workshop on Computational Advances in Multi-Sensor
 Adaptive Processing, 2013.
}
\seealso{
\code{\link{univDecomp}}, \code{\link{FCP_TPA}}
}

