% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/plot.functions.R
\name{binplot}
\alias{binplot}
\title{Plot relative effects from NMAs performed at multiple time-bins}
\usage{
binplot(
  network,
  overlay.nma = c(0, stats::quantile(network$data.ab$time)),
  method = "common",
  link = "identity",
  lim = "cred",
  plot.bins = TRUE,
  legend = TRUE,
  ...
)
}
\arguments{
\item{network}{An object of class \code{"mb.network"}.}

\item{overlay.nma}{Numeric vector used to overlay the results from a standard NMA model that
"lumps" time-points together within the time bin ranges specified in \code{overlay.nma}.
The numbers in \code{overlay.nma} define the boundaries of the time bins within which to perform
a standard NMA. Length must be >=2, or can be left as \code{NULL} (the default) to indicate that no NMA
should be performed. \code{overlay.nma} can only be specified if \code{overlay.ref==TRUE}.
See Details for further information.}

\item{method}{Can take \code{"common"} or \code{"random"} to indicate the type of NMA model used to synthesise data points
given in \code{overlay.nma}. The default is \code{"random"} since this assumes different
time-points in \code{overlay.nma} have been lumped together to estimate the NMA.}

\item{link}{Can take either \code{"identity"} (the default),
\code{"log"} (for modelling Ratios of Means \insertCite{friedrich2011}{MBNMAtime}) or
\code{"smd"} (for modelling Standardised Mean Differences - although this also corresponds to an identity link function).}

\item{lim}{Specifies calculation of either 95\% credible intervals (\code{lim="cred"}) or 95\% prediction intervals (\code{lim="pred"}).}

\item{plot.bins}{Plot time bin boundaries as vertical dashed lines. Setting \code{plot.bins=TRUE} if \code{overlay.nma}
is specified also sets x-axis ticks to time bin boundaries automatically.}

\item{legend}{\code{TRUE}/\code{FALSE} to indicate whether a legend should be plotted.}

\item{...}{Arguments to be sent to R2jags.}
}
\value{
Plots treatment effects from NMAs performed within discrete time bins. The
object returned is a list containing the plot and a sublist of NMA results and
predictions from each time bin specified in \code{overlay.nma}.
}
\description{
Plot relative effects from NMAs performed at multiple time-bins
}
\details{
Performs several standard NMAs at different time "bins", time periods within
which treatment effects are assumed to be constant over time. Separate NMAs
are then performed within each time bin on data points from studies that fall
within the time bin (only a single follow-up time is taken from each study
to avoid double counting).

Note that the wider the time bin boundaries specified by the user, the
larger the potential range of included follow-up times and this can
introduce heterogeneity or inconsistency.

Results are plotted versus the network reference and are plotted on the
specified link scale. Each time bin window is marked on the plot by
vertical dashed lines. The NMA estimates within each time bin are plotted
as a horizontal solid black line (the posterior median) with a shaded region
indicating the 95\% credible interval (prediction intervals can instead
be plotted). The width of these shaded regions is equal to the range of study
time-points included in the NMA performed within that timebin, which
may therefore be more narrow than the time bin specified in the \code{binplot()}
command due to the follow-up times at which data is available in included
studies.
}
\section{Overlaying NMA results}{


\code{overlay.nma} indicates regions of the data (defined as "time bins") over which it may be reasonable to "lump" different
follow-up times from different studies together and assume a standard NMA model. For example:
\itemize{
\item \code{overlay.nma=c(5,10)} indicates a single NMA of studies with follow-up times \verb{>5} and \verb{<=10}
\item \code{overlay.nma=c(5,10,15)} indicates two NMAs should be performed of studies with follow-up times \verb{>5} and \verb{<=10}
of studies with follow-up times \verb{>10} and \verb{<=15}
}

When used with MBNMA (via \code{predict.mbnma()}) this allows comparison to MBNMA results over a specific range of time within each time bin.
It can be useful to assess which time-course function might be suitable when using \code{binplot()}, or to
to assess if the MBNMA predictions are in agreement with predictions from an NMA model when using \code{plot.mb.predict()}
for a specific range of time-points. This can be a general indicator of the fit of the time-course model.

However, it is important to note that the wider the range specified in \code{overlay.nma}, the more likely it is that different time-points
are included, and therefore that there is greater heterogeneity/inconsistency in the NMA model. If \code{overlay.nma} includes
several follow-up times for any study then only a single time-point will be taken (the one closest to \code{mean(overlay.nma)}).
The NMA predictions are plotted over the range specified in \code{overlay.nma} as a horizontal line, with the 95\%CrI shown by a grey
rectangle. The NMA predictions represent those for \emph{any time-points within this range} since they lump together data at
all these time-points. Predictions for treatments that are disconnected from
the network reference treatment at data points specified within \code{overlay.nma} cannot be estimated so are not included.

It is important to note that the NMA model is not necessarily the "correct" model, since it "lumps" different time-points
together and ignores potential differences in treatment effects that may arise from this. The wider the range specified in
\code{overlay.nma}, the greater the effect of "lumping" and the stronger the assumption of similarity between studies.

For an NMA model to be estimated and a corresponding prediction to be made from it, \strong{each} time bin
must include the network reference treatment (treatment=1) evaluated in at least 1 connected study in the time bin.
If a given time bin does not meet this criteria then an NMA will not be calculated for it.
}

\examples{
\donttest{
# Create an mb.network object from a dataset
alognet <- mb.network(alog_pcfb)

# Plot relative effects from NMAs calculated for a single time-bins
# Do not plot time-bin boundaries
binplot(alognet, overlay.nma=c(0,5), plot.bins=FALSE)

# Plot relative effects from NMAs at multiple time-bins
# With random treatment effects
binplot(alognet, overlay.nma=c(5,10,15,20),
  method="random")
}

}
