% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/mbnma-class.R
\name{predict.mbnma}
\alias{predict.mbnma}
\title{Predict responses for different doses of agents in a given population based on MBNMA
dose-response models}
\usage{
\method{predict}{mbnma}(
  object,
  n.doses = 30,
  exact.doses = NULL,
  E0 = 0.2,
  synth = "fixed",
  lim = "cred",
  regress.vals = NULL,
  ...
)
}
\arguments{
\item{object}{An S3 object of class \code{"mbnma"} generated by running
a dose-response MBNMA model}

\item{n.doses}{A number indicating the number of doses at which to make predictions
within each agent. The default is \code{30}.}

\item{exact.doses}{A list of numeric vectors. Each named element in the list corresponds to an
agent (either named similarly to agent names given in the data, or named
correspondingly to the codes for agents given in \code{mbnma}) and each number within the vector
for that element corresponds to a dose of the agent for which to predict responses.
Doses can only take positive values. For models fitted using \code{dspline()} making predictions at only a very small
number of doses for each agent may throw an error since it can make the spline difficult to identify.}

\item{E0}{An object to indicate the value(s) to use for the response at dose = 0 (i.e.
placebo) in the prediction. This can take a number of different formats depending
on how it will be used/calculated. The default is \code{0.2} since a default of \code{0} will typically lead
to non-sensical predictions unless an identify link function has been used for the MBNMA model in \code{object}.
\itemize{
\item \code{numeric()} A single numeric value representing the deterministic response at dose = 0,
given on the natural scale - so for binomial data, proportions should be given and
for Poisson data, a rate should be given.
\item \code{character()} A single string representing a stochastic distribution for the response
at dose = 0, given on the natural scale - so for binomial data, proportions should be given and
for Poisson data, a rate should be given. This is specified as a random number generator
(RNG) given as a string, and can take any RNG distribution for which a function exists
in R. For example: \code{"rnorm(n, 7, 0.5)"}.
\item \code{data.frame()} A data frame containing data in the long format (one row per study arm) to be meta-analysed
to estimate the dose = 0 (placebo) response. This could be a set of observational
studies that are specific to the population on which to make
predictions, or it can be a subset of the study arms within the MBNMA dataset
that investigate placebo. See \code{\link[=ref.synth]{ref.synth()}}
}}

\item{synth}{A character object that can take the value \code{"fixed"} or \code{"random"} to
specify the the type of pooling to use for synthesis of \code{E0} if a data frame
has been provided for it. Using \code{"random"} rather
than \code{"fixed"} for \code{synth} will result in wider 95\\\% CrI for predictions.}

\item{lim}{Specifies calculation of either 95\% credible intervals (\code{lim="cred"}) or 95\% prediction intervals (\code{lim="pred"}).}

\item{regress.vals}{A named numeric vector of effect modifier values at which results should
be predicted. Named elements must match variable names specified in \code{regress.vars} within
the MBNMA model.}

\item{...}{Arguments to be sent to \code{\link[R2jags:jags]{R2jags::jags()}} for synthesis of the network
reference treatment effect (using \code{\link[=ref.synth]{ref.synth()}})}
}
\value{
An S3 object of class \code{mbnma.predict} that contains the following
elements:
\itemize{
\item \code{predicts} A named list of
matrices. Each matrix contains the MCMC results of predicted responses at
follow-up times specified in \code{times} for each treatment specified in
\code{treats}
\item \code{likelihood} The likelihood used in the MBNMA model \code{object}
\item \code{link} The link function used in the MBNMA model \code{object}
\item \code{network} The dataset in \code{mbnma.network} format
\item \code{E0} A numeric vector of value(s) used for E0 in the prediction, on the
link scale.
}
}
\description{
Used to predict responses for different doses of agents or to predict
the results of a new study. This is calculated by combining
relative treatment effects with a given reference treatment response
(specific to the population of interest).
}
\details{
The range of doses on which to make predictions can be specified in one of two ways:
\enumerate{
\item Use \code{max.dose} and \code{n.doses} to specify the maximum dose for each agent and the
number of doses within that agent for which to predict responses. Doses will be chosen
that are equally spaced from zero to the maximum dose for each agent. This is useful
for generating plots of predicted responses (using \verb{[plot-mbnma.predict]}) as it will
lead to fitting a smooth dose-response curve (provided \code{n.doses} is sufficiently high).
\item Use \code{exact.doses} to specify the exact doses for which to predict responses for each
agent. This may be more useful when ranking different predicted responses using
\verb{[rank-mbnma.predict]}
}
}
\examples{
\donttest{
# Using the triptans data
network <- mbnma.network(triptans)

# Run an Emax dose-response MBNMA
emax <- mbnma.run(network, fun=demax(), method="random")


###########################
###### Specifying E0 ######
###########################

#### Predict responses using deterministic value for E0 ####
# Data is binomial so we specify E0 on the natural scale as a probability
pred <- predict(emax, E0 = 0.2)

# Specifying non-sensical values will return an error
#pred <- predict(emax, E0 = -10)
### ERROR ###

#### Predict responses using stochastic value for E0 ####
# Data is binomial so we might want to draw from a beta distribution
pred <- predict(emax, E0 = "rbeta(n, shape1=1, shape2=5)")

# Misspecifying the RNG string will return an error
#pred <- predict(emax, E0 = "rbeta(shape1=1, shape2=5)")
### ERROR ###


#### Predict responses using meta-analysis of dose = 0 studies ####

# E0 is assigned a data frame of studies to synthesis
# Can be taken from placebo arms in triptans dataset
ref.df <- network$data.ab[network$data.ab$agent==1,]

# Synthesis can be fixed/random effects
pred <- predict(emax, E0 = ref.df, synth="random")



######################################################################
#### Specifying which doses/agents for which to predict responses ####
######################################################################

# Change the number of predictions for each agent
pred <- predict(emax, E0 = 0.2, n.doses=20)
pred <- predict(emax, E0 = 0.2, n.doses=3)

# Specify several exact combinations of doses and agents to predict
pred <- predict(emax, E0 = 0.2,
            exact.doses=list("eletriptan"=c(0:5), "sumatriptan"=c(1,3,5)))
plot(pred) # Plot predictions

# Print and summarise `mbnma.predict` object
print(pred)
summary(pred)

# Plot `mbnma.predict` object
plot(pred)
}

}
