# Functions for predicting responses in MBNMAdose
# Author: Hugo Pedder
# Date created: 2019-04-25

## quiets concerns of R CMD check re: the .'s that appear in pipelines
if(getRversion() >= "2.15.1")  utils::globalVariables(c("."))

#' Predict responses for different doses of agents in a given population based on MBNMA
#' dose-response models
#'
#' Used to predict responses for different doses of agents or to predict
#' the results of a new study. This is calculated by combining
#' relative treatment effects with a given reference treatment response
#' (specific to the population of interest).
#'
#' @param object An S3 object of class `"mbnma"` generated by running
#'   a dose-response MBNMA model
#'
#' @param max.doses A list of numbers. Each named element in the list correponds to an
#'   agent (either named similarly to agent names given in the data, or named
#'   correspondingly to the codes for agents given in `mbnma`)
#'   and each number for that element corresponds to the maximum dose of the given agent, below which several
#'   predictions will be calculated at different doses (the number of these is determined by `n.doses`).
#'   Can only take positive values. If left as `NULL` (the default) results will be predicted based on the
#'   maximum dose of each agent given in the data.
#' @param n.doses A number indicating the number of doses at which to make predictions
#'   within each agent. The default is `15`.
#' @param exact.doses A list of numeric vectors. Each named element in the list correponds to an
#'   agent (either named similarly to agent names given in the data, or named
#'   correspondingly to the codes for agents given in `mbnma`) and each number within the vector
#'   for that element corresponds to a dose of the agent for which to predict responses.
#'   Doses can only take positive values.
#' @param E0 An object to indicate the value(s) to use for the response at dose = 0 (i.e.
#'   placebo) in the prediction. This can take a number of different formats depending
#'   on how it will be used/calculated. The default is `0` but this will typically lead
#'   to non-sensical predictions.
#'   * `numeric()` A single numeric value representing the deterministic response at dose = 0,
#'   given on the natural scale - so for binomial data, proportions should be given and
#'   for Poisson data, a rate should be given.
#'   * `character()` A single string respresenting a stochastic distribution for the response
#'   at dose = 0, given on the natural scale - so for binomial data, proportions should be given and
#'   for Poisson data, a rate should be given. This is specified as a random number generator
#'   (RNG) given as a string, and can take any RNG distribution for which a function exists
#'   in R. For example: `"rnorm(n, 7, 0.5)"`.
#'   * `data.frame()` A data frame containing data in the long format (one row per study arm) to be meta-analysed
#'   to estimate the dose = 0 (placebo) response. This could be a set of observational
#'   studies that are specific to the population on which to make
#'   predictions, or it can be a subset of the study arms within the MBNMA dataset
#'   that investigate placebo. See [ref.synth()]
#' @param synth A character object that can take the value `"fixed"` or `"random"` to
#'   specify the the type of pooling to use for synthesis of `E0` if a data frame
#'   has been provided for it. Using `"random"` rather
#'   than `"fixed"` for `synth` will result in wider 95\\% CrI for predictions.
#' @param ... Arguments to be sent to [R2jags::jags()] for synthesis of the network
#'   reference treatment effect (using [ref.synth()])
#'
#'
#' @return An S3 object of class `mbnma.predict` that contains the following
#'   elements:
#'
#' * `summary` A named list of data frames. Each data frame contains
#'   a summary of predicted responses at follow-up times specified in `times`
#'   for each treatment specified in `treats`
#'
#' * `pred.mat` A named list of
#'   matrices. Each matrix contains the MCMC results of predicted responses at
#'   follow-up times specified in `times` for each treatment specified in
#'   `treats`
#'
#' @details
#' The range of doses on which to make predictions can be specified in one of two ways:
#'
#' 1. Use `max.dose` and `n.doses` to specify the maximum dose for each agent and the
#' number of doses within that agent for which to predict responses. Doses will be chosen
#' that are equally spaced from zero to the maximum dose for each agent. This is useful
#' for generating plots of predicted responses (using `[plot-mbnma.predict]`) as it will
#' lead to fitting a smooth dose-response curve (provided `n.doses` is sufficiently high).
#'
#' 2. Use `exact.doses` to specify the exact doses for which to predict responses for each
#' agent. This may be more useful when ranking different predicted responses using
#' `[rank-mbnma.predict]`
#'
#' @examples
#' \donttest{
#' # Using the triptans data
#' network <- mbnma.network(HF2PPITT)
#'
#' # Run an Emax dose-response MBNMA
#' emax <- mbnma.emax(network, emax="rel", ed50="rel", method="random")
#'
#'
#' ###########################
#' ###### Specifying E0 ######
#' ###########################
#'
#' #### Predict responses using deterministic value for E0 ####
#' # Data is binomial so we specify E0 on the natural scale as a probability
#' pred <- predict(emax, E0 = 0.2)
#'
#' # Specifying non-sensical values will return an error
#' #pred <- predict(emax, E0 = -10)
#' ### ERROR ###
#'
#' #### Predict responses using stochastic value for E0 ####
#' # Data is binomial so we might want to draw from a beta distribution
#' pred <- predict(emax, E0 = "rbeta(n, shape1=1, shape2=5)")
#'
#' # Misspecifying the RNG string will return an error
#' #pred <- predict(emax, E0 = "rbeta(shape1=1, shape2=5)")
#' ### ERROR ###
#'
#'
#' #### Predict responses using meta-analysis of dose = 0 studies ####
#'
#' # E0 is assigned a data frame of studies to synthesis
#' # Can be taken from placebo arms in triptans dataset
#' ref.df <- network$data.ab[network$data.ab$agent==1,]
#'
#' # Synthesis can be fixed/random effects
#' pred <- predict(emax, E0 = ref.df, synth="random")
#'
#'
#'
#' ######################################################################
#' #### Specifying which doses/agents for which to predict responses ####
#' ######################################################################
#'
#' # Change the number of predictions for each agent
#' pred <- predict(emax, E0 = 0.2, n.doses=20)
#' pred <- predict(emax, E0 = 0.2, n.doses=3)
#'
#' # Change the range of predicted doses to be the same for all agents
#' # But only predict responses for a subset of agents
#' pred <- predict(emax, E0 = 0.2,
#'             max.doses=list("Placebo"=0, "eletriptan"=5, "sumatriptan"=5))
#' plot(pred) # Plot predictions
#'
#' # Specify several exact combinations of doses and agents to predict
#' pred <- predict(emax, E0 = 0.2,
#'             exact.doses=list("eletriptan"=c(0:5), "sumatriptan"=c(1,3,5)))
#' plot(pred) # Plot predictions
#'
#' # Print and summarise `mbnma.predict` object
#' print(pred)
#' summary(pred)
#'
#' # Plot `mbnma.predict` object
#' plot(pred)
#' }
#'
#' @export
predict.mbnma <- function(object, n.doses=15, max.doses=NULL, exact.doses=NULL,
                          E0=0, synth="fixed",
                          ...) {
  ######## CHECKS ########

  # Run checks
  argcheck <- checkmate::makeAssertCollection()
  checkmate::assertClass(object, "mbnma", add=argcheck)
  checkmate::assertList(max.doses, types="numeric", null.ok=TRUE, add=argcheck)
  checkmate::assertList(exact.doses, types="numeric", null.ok=TRUE, add=argcheck)
  checkmate::assertInt(n.doses, lower=1, add=argcheck)
  #checkmate::assertDataFrame(E0.estimate, null.ok=TRUE, add=argcheck)
  checkmate::assertChoice(synth, choices=c("random", "fixed"), add=argcheck)
  checkmate::reportAssertions(argcheck)

  agents <- object$model$data()$agent
  mbnma.agents <- object$network[["agents"]]

  # Checks for doses
  doses <- NULL
  if (!is.null(exact.doses) & !is.null(max.doses)) {
    warning("`exact.dose` and `max.doses` have both been specified in the arguments, yet only one of these can be used. Deferring to using argument given for `exact.doses`")
    max.doses <- NULL
  }
  if (!is.null(exact.doses)) {
    doses <- exact.doses
  } else if (!is.null(max.doses)) {
    doses <- max.doses

    for (i in seq_along(doses)) {
      doses[[i]] <- signif(seq(0,
                               max(doses[[i]]),
                               length.out=n.doses), 2)
    }

  }

  if (!is.null(doses)) {
    if (length(doses)>max(agents, na.rm=TRUE)) {
      stop("A greater number of agents have been supplied in either `max.doses` or `exact.doses` than are present in the model")
    }

    # If named elements of list are not numeric, check if they match agents in mbnma
    match.pass <- TRUE
    if (is.null(names(doses))) {
      if (length(doses)!=length(mbnma.agents)) {
        stop("If elements in `max.doses` or `exact.doses` are not named then there must be the same number of elements as there are agents in the model, so that they correspond to agent codes")
      }
      names(doses) <- mbnma.agents
      agent.num <- 1:length(mbnma.agents)
    } else if (!is.null(names(doses))) {
      if (any(is.na(suppressWarnings(as.numeric(names(doses)))))) {
        if (!all(names(doses) %in% mbnma.agents)) {
          match.pass <- FALSE
        }
        agent.num <- match(names(doses), mbnma.agents)
      } else {
        if (!all(names(doses) %in% c(1:max(agents, na.rm=TRUE)))) {
          match.pass <- FALSE
        }
        agent.num <- as.numeric(names(doses)) # Add an agent numerical identifier for included agents
      }
      if (match.pass==FALSE) {
        stop("Element names in `doses` must correspond either to agent names in data or agent codes in `object`")
      }
    }



    for (i in seq_along(doses)) {
      if (!all(doses[[i]]>=0)) {
        stop(paste0("Doses given in `doses` must be positive. Agent ", names(doses)[i], " contains negative values."))
      }
    }
  } else {
    # Automatically generate doses list for treatments included in data
    dose <- as.vector(object$model$data()$dose)
    agent <- as.vector(agents)

    df <- data.frame("agent"=agent, "dose"=dose)
    df <- unique(df[stats::complete.cases(df),])
    df <- dplyr::arrange(df, agent, dose)
    df$agent <- factor(df$agent, labels=mbnma.agents)

    doses <- list()
    for (i in seq_along(mbnma.agents)) {
      doses[[mbnma.agents[i]]] <- signif(seq(0,
                                             max(df$dose[df$agent==mbnma.agents[i]]),
                                             length.out=n.doses), 2)
    }
    agent.num <- c(1:length(mbnma.agents))
  }



  # Set model arguments
  if (length(object$model.arg$class.effect)>0) {
    stop("`predict() currently does not work with models that use class effects")
  }
  if (object$model.arg$fun[1] %in% c("nonparam.up", "nonparam.down")) {
    stop("`predict() does not work with non-parametric dose-response functions")
  }
  if (length(object$model.arg$fun)>1) {
    #stop("`predict() currently does not work with models that use multiple dose-response functions")

    funs <- c(NA, 1,1,2,3)
    names(funs) <- c("user", "linear", "exponential", "emax", "emax.hill")
    funs <- funs[names(funs) %in% object$model.arg$fun]

    # Want to find the location of the agents within the vector of agent names in network
    #funi <- which(names(doses) %in% object$network$agents)
    funi <- which(object$network$agents %in% names(doses))
    X <- sapply(object$model.arg$fun[funi], function(x) which(x==names(funs)))
  }

  link <- object$model.arg$link

  n <- object$BUGSoutput$n.keep * object$BUGSoutput$n.chains

  DR <- suppressMessages(
    write.dose.fun(fun=object$model.arg$fun, user.fun=object$model.arg$user.fun,
                   effect="abs"
                   )[[1]])
  DR <- gsub("(^.+<-)(.+)", "\\2", DR)

  betaparams <- get.model.vals(object)
  betas <- assignfuns(object$model.arg$fun, object$network$agents, object$model.arg$user.fun,
                      ifelse(is.null(object$model.arg$arg.fun), FALSE, TRUE))


  # Identify E0
  if (is.null(E0)) {
    stop("`E0` has not been given a value. Responses cannot be predicted without a value(s) for dose = 0 (placebo)")
  }
  if (!any(class(E0) %in% c("numeric", "character", "data.frame"))) {
    stop("`E0` can only be of type `numeric()`, `character()` or `data.frame()`")
  }

  if ((is.numeric(E0) | is.character(E0)) & length(E0)!=1) {
    stop("`E0` must take a single numeric (deterministic E0) or character (stochastic E0) value, or be provided with a data frame so that it may be estimated from the data")
  }
  if (is.character(E0)) {
    if (!grepl("^r[a-z]+\\(n,.+\\)", E0)) {
      stop("Distribution for `E0` does not match stochastic random number generator format.\nFormat must take any R random number generator function")
    }
    E0 <- eval(parse(text=E0))
  }

  if ((is.numeric(E0) | is.character(E0))) {
    E0 <- rescale.link(E0, direction="link", link=link)
  } else if (is.data.frame(E0)) {
    E0 <- ref.synth(data.ab=E0, mbnma=object, synth=synth, ...)

    if (!("sd.mu" %in% names(E0))) {
      E0 <- E0$m.mu
    } else {
      E0 <- stats::dnorm(E0$m.mu, E0$sd.mu)
    }
  }

  # if (!is.null(E0.data)) {
  #   if (length(E0.data)!=1) {
  #     stop("`E0.data` must take a single numeric (deterministic E0) or character (stochastic E0) value")
  #   }
  #   if (is.numeric(E0.data)) {
  #     E0 <- E0.data
  #   } else if (is.character(E0.data)) {
  #     if (!grepl("^r[a-z]+\\(n,.+\\)", E0.data)) {
  #       stop("Distribution for `E0.data` does not match stochastic random number generator format.\nFormat must take any R random number generator function")
  #     }
  #     E0 <- eval(parse(text=E0.data))
  #   }
  #
  #   # Convert to scale using link function
  #   E0 <- rescale.link(E0, direction="link", link=link)
  #
  # } else if (!is.null(E0.estimate)) {
  #   E0 <- ref.synth(data.ab=E0.estimate, mbnma=x, synth=synth)
  #
  #   if (!("sd.mu" %in% names(E0))) {
  #     E0 <- E0$m.mu
  #   } else {
  #     E0 <- dnorm(E0$m.mu, E0$sd.mu)
  #   }
  # }


  predict.result <- list()

  for (i in seq_along(doses)) {
    predict.result[[names(doses)[i]]] <- list()
    for (k in seq_along(doses[[i]])) {
      if (names(doses)[i] %in% c("1", "Placebo") | doses[[i]][k]==0) {
        # Ensures reference agent (placebo) takes E0
        # THIS NEEDS TO BE CHANGED IF INTERCEPT IS RELAXED! sHOULD ONLY BE USED IF INTERCEPT=FALSE
        pred <- E0

      } else {
        tempDR <- gsub("\\[agent\\[i,k\\]\\]", "", DR)
        tempDR <- gsub("\\[i,k\\]", "", tempDR)

        # For multiple DR functions
        tempDR <- gsub("X==", "X[i]==", tempDR)
        #tempDR <- gsub("s\\.beta\\.", "beta\\.", tempDR)

        # Need to enclose ifelse() matrices in list() to allow ifelse to return something in matrix form
        if (length(object$model.arg$fun)>1) {
          tempDR <- gsub("(ifelse)(.*?,)(.*?,)", "\\1\\2list(\\3!!!),", tempDR)
          tempDR <- gsub(",\\!\\!\\!", "", tempDR)
          tempDR <- gsub("(.+,)(.*?)$", "\\1list(\\2)", tempDR)
        }



        dose <- doses[[i]][k]
        for (param in seq_along(betaparams)) {
          #print(param)
          if (is.vector(betaparams[[param]]$result)) {
            assign(paste0("s.", names(betaparams)[param]),
                   betaparams[[param]]$result)
          } else if (is.matrix(betaparams[[param]]$result)) {

            # Look for correct column index for each beta param
            colnum <- which(grepl(paste0("\\[", agent.num[i], "\\]"),
              colnames(betaparams[[param]]$result)
            ))

            assign(paste0("s.", names(betaparams)[param]),
                   betaparams[[param]]$result[,colnum]
            )
            #print(betaparams[[param]]$result[,colnum])

            # assign(paste0("s.", names(betaparams)[param]),
            #        betaparams[[param]]$result[,
            #                                   grepl(
            #                                     paste0("\\[", agent.num[i], "\\]"),
            #                                     colnames(betaparams[[param]]$result)
            #                                   )
            #                                   ]
            # )
          }
        }

        chunk <- eval(parse(text=tempDR))
        if (is.list(chunk)) {
          chunk <- chunk[[1]]
        }
        pred <- E0 + chunk
        if (length(pred)<=1) {stop()}
      }

      # Convert to natural scale using link function
      pred <- rescale.link(pred, direction="natural", link=link)

      predict.result[[names(doses)[i]]][[as.character(doses[[i]][k])]] <-
        pred
    }
  }

  output <- list("predicts"=predict.result,
                 "likelihood"=object$model.arg$likelihood, "link"=object$model.arg$link,
                 "network"=object$network)

  class(output) <- "mbnma.predict"

  return(output)
}






#' Get MBNMA model values for dose-response parameters
#'
#' @inheritParams predict.mbnma
#' @noRd
get.model.vals <- function(mbnma) {

  betaparams <- list()
  for (i in 1:4) {
    beta <- paste0("beta.",i)
    if (!is.null(mbnma$model.arg[[beta]])) {
      temp <- list()
      temp$pool <- mbnma$model.arg[[beta]]

      if (is.null(mbnma$model.arg$arg.params)) {
        #temp$name <- betaparams[[beta]]
        temp$name <- i
      } else {
        temp$name <- mbnma$model.arg$arg.params$wrap.params[
          mbnma$model.arg$arg.params$run.params==beta
          ]
      }

      res.mat <- mbnma$BUGSoutput$sims.matrix
      if (temp$pool=="rel") {
        temp$result <- res.mat[,grepl(paste0("^d.", temp$name), colnames(res.mat))]
      } else if (temp$pool=="common") {
        temp$result <- res.mat[,grepl(paste0("^beta.", temp$name), colnames(res.mat))]
      } else if (temp$pool=="random") {
        temp$result <- stats::dnorm(res.mat[,grepl(paste0("^beta.", temp$name), colnames(res.mat))],
                             res.mat[,grepl(paste0("^sd.", temp$name), colnames(res.mat))]
        )
      } else if (is.numeric(temp$pool)) {
        temp$result <- rep(temp$pool, mbnma$BUGSoutput$n.sims)
      }

      betaparams[[beta]] <- temp
    }
  }

  return(betaparams)

}





#' Synthesise single arm dose = 0 / placebo studies to estimate E0
#'
#' Synthesises single arm studies to estimate E0. Used in predicting responses from a
#' dose-response MBNMA.
#'
#' @inheritParams predict.mbnma
#' @inheritParams R2jags::jags
#' @param data.ab A data frame of arm-level data in "long" format containing the
#'   columns:
#'   * `studyID` Study identifiers
#'   * `y` Numeric data indicating the aggregate response for a continuous outcome. Required for
#'   continuous data.
#'   * `se` Numeric data indicating the standard error for a given observation. Required for
#'   continuous data.
#'   * `r` Numeric data indicating the number of responders within a study arm. Required for
#'   binomial or poisson data.
#'   * `N` Numeric data indicating the total number of participants within a study arm. Required for
#'   binomial data
#'   * `E` Numeric data indicating the total exposure time for participants within a study arm. Required
#'   for poisson data.
#' @param mbnma An S3 object of class `"mbnma"` generated by running
#'   a dose-response MBNMA model
#'
#' @details `data.ab` can be a collection of studies that closely resemble the
#'   population of interest intended for the prediction, which could be
#'   different to those used to estimate the MBNMA model, and could include
#'   single arms of RCTs or observational studies. If other data is not
#'   available, the data used to estimate the MBNMA model can be used by
#'   selecting only the studies and arms that investigate dose = 0 (placebo).
#'
#'   Defaults for `n.iter`, `n.burnin`, `n.thin` and `n.chains` are those used to estimate
#'   `mbnma`.
#'
#' @return A list of named elements corresponding to E0 and the between-study standard deviation for
#'   E0 if `synth="random"`. Each element contains the full MCMC results from the synthesis.
#'
#' @examples
#' \donttest{
#' # Using the triptans data
#' network <- mbnma.network(HF2PPITT)
#'
#' # Run an Emax dose-response MBNMA
#' emax <- mbnma.emax(network, emax="rel", ed50="rel", method="random")
#'
#' # Data frame for synthesis can be taken from placebo arms
#' ref.df <- HF2PPITT[HF2PPITT$agent=="placebo",]
#'
#' # Meta-analyse placebo studies using fixed treatment effects
#' E0 <- ref.synth(ref.df, emax, synth="fixed")
#' names(E0)
#'
#' # Meta-analyse placebo studies using random treatment effects
#' E0 <- ref.synth(ref.df, emax, synth="random")
#' names(E0)
#' }
#'
#' @export
ref.synth <- function(data.ab, mbnma, synth="fixed",
                      n.iter=mbnma$BUGSoutput$n.iter,
                      n.burnin=mbnma$BUGSoutput$n.burnin,
                      n.thin=mbnma$BUGSoutput$n.thin,
                      n.chains=mbnma$BUGSoutput$n.chains,
                      ...) {

  # First need to validate data.frame to check dataset is in correct format...maybe another function for this
  # Change it to correct format if it is not already
  data.ab <- E0.validate(data.ab, likelihood=mbnma$model.arg$likelihood)[["data.ab"]]

  # Run checks
  argcheck <- checkmate::makeAssertCollection()

  checkmate::assertClass(mbnma, "mbnma", add=argcheck)
  checkmate::assertChoice(synth, choices=c("random", "fixed"), add=argcheck)
  checkmate::assertInt(n.iter, lower=1, add=argcheck)
  checkmate::assertInt(n.burnin, lower=1, add=argcheck)
  checkmate::assertInt(n.thin, lower=1, add=argcheck)
  checkmate::assertInt(n.chains, lower=1, add=argcheck)

  checkmate::reportAssertions(argcheck)

  # To get model for meta-analysis of placebo must create v similar model
  #to study model
  # Do all the mbnma.write bits but without the consistency bits

  jagsmodel <- write.E0.synth(synth=synth,
                              likelihood=mbnma$model.arg$likelihood,
                              link=mbnma$model.arg$link
  )

  parameters.to.save <- c("m.mu", "resdev", "totresdev")
  if (synth=="random") {
    parameters.to.save <- append(parameters.to.save, "sd.mu")
  }

  jags.result <- suppressWarnings(
    mbnma.jags(data.ab, model=jagsmodel,
               parameters.to.save=parameters.to.save,
               likelihood=mbnma$model.arg$likelihood,
               link=mbnma$model.arg$link,
               n.iter=n.iter, n.burnin=n.burnin,
               n.thin=n.thin, n.chains=n.chains,
               ...)[["jagsoutput"]]
  )

  result <- list("m.mu"=jags.result$BUGSoutput$sims.list[["m.mu"]])
  if (synth=="random") {
    result[["sd.mu"]] <- jags.result$BUGSoutput$sims.list[["sd.mu"]]
  }

  if (any(jags.result$BUGSoutput$summary[,
                                         colnames(jags.result$BUGSoutput$summary)=="Rhat"
                                         ]>1.02)) {
    warning("Rhat values for parameter(s) in reference treatment synthesis model are >1.02. Suggest running for more iterations.")
  }

  return(result)

}





#' Checks the validity of ref.estimate
#'
#' Ensures `E0` takes the correct form to allow for synthesis of network
#' reference treatment response
#'
#' @param likelihood A character object that can take any of `"normal"`, `"binomial"` or `"poisson"`
#'
#' @inheritParams ref.synth
#' @noRd
E0.validate <- function(data.ab, likelihood=NULL) {

  argcheck <- checkmate::makeAssertCollection()
  checkmate::assertDataFrame(data.ab, any.missing=FALSE, add=argcheck)
  checkmate::assertNames(names(data.ab), must.include = c("studyID"), add=argcheck)
  checkmate::reportAssertions(argcheck)

  # Check/assign link and likelihood
  likelink <- check.likelink(data.ab, likelihood=likelihood)
  likelihood <- likelink[["likelihood"]]


  # Sort data.ab
  data.ab <- dplyr::arrange(data.ab, studyID)

  data.ab <- data.ab %>%
    dplyr::group_by(studyID) %>%
    dplyr::mutate(narm=dplyr::n())

  if (any(data.ab$narm>1)) {
    stop("Studies in `data.ab` contain >1 arm. ref.synth() only pools single arms.")
  }

  print("Data frame must contain only data from reference treatment")

  #### Prepare data frame ####
  # Add arm index (=1 since only one arm in each study)
  data.ab[["arm"]] <- 1
  data.ab[["narm"]] <- 1

  # Ensuring studies are numbered sequentially
  if (!is.numeric(data.ab[["studyID"]])) {
    print("Studies being recoded to allow sequential numbering")
    data.ab <- transform(data.ab,studyID=as.numeric(factor(studyID, levels=as.character(unique(data.ab$studyID)))))
  } else if (all(abs(diff(data.ab[["studyID"]])) != TRUE)) {
    print("Studies being recoded to allow sequential numbering")
    data.ab <- transform(data.ab,studyID=as.numeric(factor(studyID, levels=as.character(unique(data.ab$studyID)))))
  }
  data.ab <- dplyr::arrange(data.ab, studyID)

  data.ab <- add_index(data.ab)

  return(data.ab)

}




#' Rescale data depending on the link function provided
#'
#' @inheritParams mbnma.run
#' @param x A numeric vector of data to be rescaled
#' @param direction Can take either `"link"` to convert data to a particular scale
#'   as defined by the `link` function, or `"natural"` to return it to the natural scale.
#'
#' @return A rescaled numeric vector
#'
#' @export
rescale.link <- function(x, direction="link", link="logit") {

  argcheck <- checkmate::makeAssertCollection()
  checkmate::assertNumeric(x, add=argcheck)
  checkmate::assertChoice(direction, choices=c("natural", "link"), null.ok = FALSE, add=argcheck)
  checkmate::assertChoice(link, choices=c("logit", "identity", "log", "probit", "cloglog"), null.ok = FALSE, add=argcheck)
  checkmate::reportAssertions(argcheck)

  if (direction=="link") {
    if (link=="logit") {
      x <- stats::qlogis(x)
    } else if (link=="log") {
      x <- log(x)
    } else if (link=="probit") {
      x <- stats::qnorm(x)
    } else if (link=="cloglog") {
      x <- log(-log(1-x))
    }

  } else if (direction=="natural") {
    if (link=="logit") {
      x <- exp(x) / (1+exp(x))
    } else if (link=="log") {
      x <- exp(x)
    } else if (link=="cloglog") {
      x <- 1-exp(-exp(x))
    } else if (link=="probit") {
      x <- stats::pnorm(x)
    }
  }
  return(x)
}
