if(getRversion() >= "3.5")
  utils::globalVariables(
    c(
      "Series",
      "xlab",
      "ylab",
      "geom_point",
      "geom_line",
      "unit",
      "element_text",
      "element_line",
      "element_rect",
      "scale_x_continuous",
      "scale_y_continuous"
    )
  )
####################################################################################################################
#' @import rootSolve graphics stats svDialogs ggplot2 svglite
#' @importFrom grDevices dev.off
#' @importFrom grDevices png
# options(warn = 1)
####################################################################################################################
# Merchuk's Equation to fit Binodal Experimental Data
#' @rdname AQSys
#' @name AQSys
#' @description .
#' @export
#' @seealso \itemize{
#' \item \code{\link{AQSys.default}}
#' \item \code{\link{AQSys.plot}}
#' \item \code{\link{AQSys.tielines}}
#' \item \code{\link{AQSys.crpt}}
#' \item \code{\link{AQSysOthmer}}
#' \item \code{\link{AQSysBancroft}}
#' }
####################################################################################################################
AQSys <- function(dataSET, ...)
  UseMethod("AQSys")
####################################################################################################################
#' @rdname AQSys
#' @title Merchuk's nonlinear Equation
#' @description Perform a nonlinear regression fit using several mathemmatical descriptors in order to determine the
#' equation's parameters.
#' @details The function returns functions parameters after fitting experimental data to the equations listed in AQSysList().
#' @param modelName - Character String specifying the nonlinear empirical equation to fit data.
#' The default method uses Merchuk's equation. Other mathematical descriptors can be listed using AQSysList().
#' @param dataSET - Binodal Experimental data that will be used in the nonlinear fit
#' @param ... Additional optional arguments. None are used at present.
#' @method AQSys default
#' @export
#' @return A list containing three data.frame variables with all data parsed from the worksheet and parameters calculated
#' through the available mathematical descriptions.
#' @examples
#' #Populating variable dataSET with binodal data
#' dataSET <- peg4kslt[,1:2]
#' #Fitting dataSET using Merchuk's function
#' AQSys(dataSET)
AQSys.default <- function(dataSET, modelName = "merchuk", ...) {
  # each switch option calls a correspondent equation to fit dataSET
  # equations are functions declared in AQSysFormulas.R
  if (modelName %in% names(AQSysList(TRUE))) {
    ans <- do.call(modelName, list(dataSET))
  } else{
    AQSys.err("0")
  }
  # return fitting parameters ans statistical data
  return(ans)
}
#
####################################################################################################################
# MERCHUK PLOT TEST FUNCTION
#' @rdname AQSys.plot
#' @title Dataset and Fitted Function plot
#' @description The function returns a plot after fitting a dataset to a given equation.
#' @details This version uses the plot function and return a regular bidimensional plot.
#' @method AQSys plot
#' @export AQSys.plot
#' @export
#' @param dataSET - Binodal Experimental data that will be used in the nonlinear fit. It might hold multiple systems stacked side-by-side. [type:data.frame]
#' @param xlbl Plot's Horizontal axis label.
#' @param ylbl Plot's Vertical axis label.
#' @param main Legacy from plot package. For more details, see \code{\link{plot.default}}
#' 
#' @param col Legacy from plot package. For more details, see \code{\link{plot.default}}
#' @param type Legacy from plot package. For more details, see \code{\link{plot.default}}
#' @param cex Legacy from plot package. For more details, see \code{\link{plot.default}}
#' @param cexlab Legacy from plot package. For more details, see \code{\link{plot.default}}
#' @param cexaxis Legacy from plot package. For more details, see \code{\link{plot.default}}
#' @param cexmain Legacy from plot package. For more details, see \code{\link{plot.default}}
#' @param cexsub Legacy from plot package. For more details, see \code{\link{plot.default}}
#' 
#' @param modelName - Character String specifying the nonlinear empirical equation to fit data. The default method uses
#' Merchuk's equation. Other possibilities can be seen in AQSysList().
#' @param NP Number of points used to build the fitted curve. Default is 100. [type:Integer]
#' @param xmax Maximum value for the Horizontal axis' value
#' @param ymax Maximum value for the Vertical axis' value
#' @param colDis Defines how the data is organized in the Worksheet. Use "xy" whether the first column corresponds to the lower phase fraction and "yx" whether the opposite.
#' @param save Save the generated plot in the disk using path and filename provided by the user. [type:Boulean]
#' @param HR Adjust Plot's text to be compatible with High Resolution size [type:Boulean]
#' @param filename Filename provided by the user to save a given plot. [type:String]
#' @param wdir The directory in which the plot file will be saved. [type:String]
#' @param silent save plot file without actually showing it to the user. [type:Boulean]
#' @param ... Additional optional arguments. None are used at present.
#' @return A plot containing the experimental data, the correspondent curve for the binodal in study and the curve's raw XY data.
#' @examples
#' #Populating variable dataSET with binodal data
#' dataSET <- peg4kslt[,1:2]
#' #Plot dataSET using Merchuk's function
#' #
#' AQSys.plot(dataSET)
#' #
AQSys.plot <-
  function  (dataSET,
             xlbl = "",
             ylbl = "",
             main = NULL,
             col = "blue",
             type = "p",
             cex = 1,
             cexlab = 1,
             cexaxis = 1,
             cexmain = 1,
             cexsub = 1,
             modelName = "merchuk",
             NP = 100,
             xmax = "",
             ymax = "",
             colDis = "xy",
             save = FALSE,
             HR = FALSE,
             filename = NULL,
             wdir = NULL,
             silent = FALSE,
             ...)
  {
    #
    plot_image = NULL
    # verify and adjust data.frame names
    if (xlbl == "") {
      xlbl <- names(dataSET)[1]
    }
    if (ylbl == "") {
      ylbl <- names(dataSET)[2]
    }
    # guarantee all lines are valid (non-na and numeric)
    dataSET <- toNumeric(dataSET, colDis)
    # Calculate aesthetically better xmax and ymax
    if ((xmax == "") | (xmax > 1) | (xmax < round(max(dataSET[1]) / 0.92, 1))) {
      xmax <- ceiling(round(max(dataSET[1]) / 0.92, 1)/5)*5
    }
    if ((ymax == "") | (ymax > 1) | (ymax < round(max(dataSET[2]) / 0.92, 1))) {
      ymax <- ceiling(round(max(dataSET[2]) / 0.92, 1)/5)*5
    }
    # If save=TRUE adjust variables and parameters to save plot
    wdir <- saveConfig(save, HR, filename, wdir, silent)
    # Select which model will be used to generate the plot. Function return list of plots and respective number of parameters
    models_npars <- AQSysList(TRUE)
    # calculate coefficients using the non-linear regression
    CoefSET <- summary(AQSys(dataSET, modelName))$coefficients[, 1]
    # choose model based in the user choice or standard value
    Fn <- ifelse(modelName %in% names(models_npars), AQSys.mathDesc(modelName), AQSys.err("0"))
    #plot phase diagram using experimental data and with previously calculated parameters
    plot_image <- ggplot(data = dataSET, aes_string(x = "XC", y = "YC")) + geom_point(shape = 8, size = 2) +
      theme_light() + xlab(paste(xlbl, "(%, m/m)")) + ylab(paste(ylbl, "(%, m/m)")) + theme(
        validate = FALSE,
        plot.margin = unit(c(1, 1, 1, 1), "cm"),
        text = element_text(size = 16),
        legend.position = "top",
        axis.title.y = element_text(vjust = 5),
        axis.title.x = element_text(vjust = -2),
        panel.grid.major = element_line(size = .70, colour = "black"),
        panel.grid.minor = element_line(size = .70),
        panel.border = element_rect(size = .5, colour = "white"),
        axis.text.x = element_text(size = 15),
        axis.text.y = element_text(size = 15),
        axis.line = element_line(colour = 'black', size = 1.25),
        legend.title = element_text(
          colour = "black",
          size = 12,
          face = "bold",
          angle = 0
        ),
        legend.text = element_text(
          colour = "black",
          size = 12,
          face = "plain"
        )
      ) +
      scale_y_continuous(
        expand = c(0, 0),
        limits = c(0.001, ymax),
        breaks = seq(0, ymax, by = 5),
        labels = seq(0, ymax, by = 5)
      ) +
      scale_x_continuous(
        expand = c(0, 0),
        limits = c(0, xmax),
        breaks = seq(0, ymax, by = xmax / 10),
        labels = seq(0, ymax, by = xmax / 10)
      )
    #
    # add curve generated using regression parameters
    rawdt <- data.frame(dataSET[1], Fn(CoefSET, dataSET[1]))
    names(rawdt) <- c("XC", "YC")
    plot_image <-
      plot_image + geom_line(
        data = rawdt,
        aes_string(x = "XC", y = "YC"),
        color = "red",
        linetype = 2,
        size = 1.1
      )
    #
    if (save == TRUE) {
      ggsave(
        filename = wdir,
        plot = plot_image,
        width = 21.14 / 2,
        height = 14.39 / 2
      )
    }
    # make available data from fitted curve to user. Function returns it silently
    # but user can get data using simple assign '<-'
    if (silent == FALSE) {
      print(plot_image)
      invisible(rawdt)
    } else {
      invisible(plot_image)
    }
  }
####################################################################################################################
