% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/get_data.R
\name{get_data_by_var}
\alias{get_data_by_var}
\title{Extract Variable Data from SL4 or HAR Objects}
\usage{
get_data_by_var(
  var_names = NULL,
  ...,
  experiment_names = NULL,
  subtotal_level = FALSE,
  rename_cols = NULL,
  merge_data = FALSE
)
}
\arguments{
\item{var_names}{Character vector. Variable names to extract. Use \code{"ALL"} or \code{NULL} to extract all available variables.}

\item{...}{One or more SL4 or HAR data objects loaded using \code{load_sl4x()} or \code{load_harx()}.}

\item{experiment_names}{Character vector. Names assigned to each dataset. If \code{NULL}, names are inferred.}

\item{subtotal_level}{Character or logical. Determines which decomposition levels to retain:
\itemize{
\item \code{"total"}: Keeps only \code{"TOTAL"} values.
\item \code{"decomposed"}: Keeps only decomposed values (excludes \code{"TOTAL"}).
\item \code{"all"}: Keeps all rows.
\item \code{TRUE}: Equivalent to \code{"all"}, retaining both \code{"TOTAL"} and decomposed values.
\item \code{FALSE}: Equivalent to \code{"total"}, keeping only \code{"TOTAL"} values.
}}

\item{rename_cols}{Named vector. Column name replacements (\code{c("old_name" = "new_name")}).}

\item{merge_data}{Logical. If \code{TRUE}, attempts to merge data across multiple experiments. Default is \code{FALSE}.}
}
\value{
A list of structured data:
\itemize{
\item If \code{merge_data = FALSE}, returns a named list where each element corresponds to an experiment.
\item If \code{merge_data = TRUE}, returns a named list of all merged data
}
}
\description{
Extracts structured data for one or more variables from SL4 or HAR objects,
transforming array-like data into a tidy format.
}
\details{
\itemize{
\item Retrieves specific variables, multiple variables, or all available variables from SL4 or HAR datasets.
\item Supports merging data from multiple experiments (\code{merge_data = TRUE}).
\item Allows renaming of column names (\code{rename_cols}).
\item Handles subtotal filtering (\code{subtotal_level}), controlling whether \code{"TOTAL"} or decomposed values are retained.
}
}
\examples{
# Import sample data:
sl4_data <- load_sl4x(system.file("extdata", "TAR10.sl4", package = "HARplus"))
sl4_data1 <- load_sl4x(system.file("extdata", "SUBT10.sl4", package = "HARplus"))

# Extract a single variable
data_qo <- get_data_by_var("qo", sl4_data)

# Extract multiple variables
data_multiple <- get_data_by_var(c("qo", "qgdp"), sl4_data)

# Extract all variables separately from multiple datasets
data_all <- get_data_by_var(NULL, sl4_data, sl4_data1, merge_data = FALSE)

# Merge variable data across multiple datasets
data_merged <- get_data_by_var(NULL, sl4_data, sl4_data1, merge_data = TRUE)

# Retain only "TOTAL" values, removing decomposed components (subtotal_level = "total" or FALSE)
data_total_only <- get_data_by_var("qo", sl4_data, subtotal_level = "total")
data_total_only_alt <- get_data_by_var("qo", sl4_data, subtotal_level = FALSE)

# Retain only decomposed components, removing "TOTAL" (subtotal_level = "decomposed")
data_decomposed_only <- get_data_by_var("qo", sl4_data, subtotal_level = "decomposed")

# Retain all value levels (subtotal_level = "all" or TRUE)
data_all_decomp <- get_data_by_var("qo", sl4_data, subtotal_level = "all")
data_all_decomp_alt <- get_data_by_var("qo", sl4_data, subtotal_level = TRUE)

# Rename specific columns
data_renamed <- get_data_by_var("qo", sl4_data, rename_cols = c(REG = "Region", COMM = "Commodity"))

# Merge data across multiple datasets with custom experiment names
data_merged_experiments <- get_data_by_var("qo", sl4_data, sl4_data1,
experiment_names = c("EXP1", "EXP2"),
merge_data = TRUE)

}
\seealso{
\code{\link{get_data_by_dims}}, , \code{\link{group_data_by_dims}}, \code{\link{load_sl4x}}, \code{\link{load_harx}}
}
\author{
Pattawee Puangchit
}
