#' Generate text using the Generative AI model based on a prompt
#'
#' This function establishes a connection to a Generative AI model by providing essential
#' parameters and generates a response based on the prompt.
#'
#' @param model.parameter A character vector containing the Generative AI service provider,
#' corresponding model, version, API key, and proxy status.
#' @param temperature A numeric value. A higher value results in more creative responses,
#' while a lower value produces more straightforward text.
#' @param prompt A character string representing the query for text generation.
#'
#' @return If successful, a character string generated by the Generative AI model based on
#' the provided query and parameters. If the API response indicates an error, the function
#' halts execution and provides an error message.
#'
#' @details Providing accurate and valid information for each parameter is crucial
#' to ensure successful text generation by the Generative AI model. If any of the
#' provided parameters is incorrect, the function will respond with an error message based
#' on the information received from the API. Use the function \code{available.models()} to
#' see all supported Generative AI models.
#'
#' @examples
#' \dontrun{
#'  # Get available models
#'  models = available.models()
#'
#'  # Connect to the model, replace API_KEY with your api key
#'  google.model = connect.genai("google",
#'                               models$google$model[1],
#'                               models$google$version[1],
#'                               "API_KEY",
#'                               FALSE)
#'
#'  # Connect to the model, replace API_KEY with your api key
#'  openai.model = connect.genai("openai",
#'                               models$openai$model[1],
#'                               models$openai$version[1],
#'                               "API_KEY",
#'                               FALSE)
#'
#'  # Generate text
#'  temperature = 0.9
#'  prompt = "Write a story about a magic backpack. 100 words is enough."
#'  google.text = text(google.model,
#'                     temperature,
#'                     prompt)
#'  cat(google.text)
#'
#'  openai.text = text(openai.model,
#'                     temperature,
#'                     prompt)
#'  cat(openai.text)
#' }
#'
#' @export
#'
#' @importFrom GenAI moderation.openai
text = function (model.parameter, temperature, prompt) {
  if (prompt == "" || is.na(prompt) || !inherits(prompt, "character")) {
    stop("Prompt is not in correct format.")
  }
  switch (model.parameter["provider"],
          google = {
            api.URL = ifelse(
              model.parameter["proxy"],
              paste0(
                "https://api.genai.gd.edu.kg/google/",
                model.parameter["version"],
                "/models/",
                model.parameter["model"],
                ":generateContent?key=",
                model.parameter["api"]
              ),
              paste0(
                "https://generativelanguage.googleapis.com/",
                model.parameter["version"],
                "/models/",
                model.parameter["model"],
                ":generateContent?key=",
                model.parameter["api"]
              )
            )
            requestBody = list(
              contents = list(parts = list(text = prompt)),
              generationConfig = list(temperature = temperature)
            )
            requestBodyJSON = jsonlite::toJSON(requestBody, auto_unbox = TRUE)
            response = httr::POST(
              url = api.URL,
              body = requestBodyJSON,
              httr::add_headers("Content-Type" = "application/json")
            )
            responseJSON = httr::content(response, "parsed")
            if (!is.null(responseJSON$error)) {
              stop(responseJSON$error$message)
            }
            if (!is.null(responseJSON$blockReason)) {
              stop("The prompt may contain harmful content.")
            }
            return (as.character(responseJSON$candidates[[1]]$content$parts[[1]]$text))
          },
          openai = {
            moderation.openai(model.parameter, prompt)
            api.URL = ifelse(
              model.parameter["proxy"],
              paste0(
                "https://api.genai.gd.edu.kg/openai/",
                model.parameter["version"],
                "/chat/completions"
              ),
              paste0(
                "https://api.openai.com/",
                model.parameter["version"],
                "/chat/completions"
              )
            )
            requestBody = list(
              model = model.parameter["model"],
              messages = list(
                list(role = "system",
                     content = "You are a helpful assistant."),
                list(role = "user",
                     content = prompt)
              ),
              temperature = temperature
            )
            requestBodyJSON = jsonlite::toJSON(requestBody, auto_unbox = TRUE)
            response = httr::POST(
              url = api.URL,
              body = requestBodyJSON,
              httr::add_headers(
                "Content-Type" = "application/json",
                "Authorization" = paste("Bearer", model.parameter["api"])
              )
            )
            responseJSON = httr::content(response, "parsed")
            if (!is.null(responseJSON$error)) {
              stop(responseJSON$error$message)
            }
            return (as.character(responseJSON$choices[[1]]$message$content))
          })
}
