% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/SpatialStatistics.R
\name{SpatialStatistics}
\alias{SpatialStatistics}
\title{Spatial statistics}
\usage{
SpatialStatistics(trees, areas = NULL, grid = NULL, variables = NULL,
  statFuns = NULL)
}
\arguments{
\item{trees}{\link[sp]{SpatialPointsDataFrame} or \link[sp]{SpatialPolygonsDataFrame}. The locations of a
set of trees, typically detected from a canopy height model using \code{\link{TreeTopFinder}}.
Tree attributes, such as height or crown size, should be stored within this object's \code{@data} slot.
Tree crowns delineated using \code{\link{SegmentCrowns}} can also be used.}

\item{areas}{\link[sp]{SpatialPolygonsDataFrame}. An optional set of polygons corresponding to areas of
interest. Tree counts and statistics will be returned for each area.}

\item{grid}{RasterLayer (see \link[raster]{raster}) or numeric. An alternative to the \code{areas} argument.
Using \code{grid} will compute tree counts and statistics within the cells of a spatial grid.
Grid size and placement can be defined by inputting a \link[raster]{raster} object.
A single numeric value can also be used, in which case the function will generate a grid with a
cell size equal to this value.}

\item{variables}{character. The names of tree attribute variables (stored in the \code{trees@data}
slot). In addition to tree counts, the function will compute statistics for each of these variables.
Only numeric variables are accepted.}

\item{statFuns}{list. A named list of custom functions that are used to compute tree attribute
statistics. If none are provided, default statistics are mean, median, standard deviation,
minimum and maximum. Note that each element of the list should have a name that describes the
statistics generated by the function. See below for details on defining custom functions.}
}
\value{
Tree count and, if any \code{variables} are supplied, tree attribute statistics. If no
\code{areas} or \code{grid} is supplied, the tree count and statistics are computed for the entire
\code{trees} dataset, and returned as a 'data.frame' object. If \code{areas} are defined, an
identical \link[sp]{SpatialPolygonsDataFrame} will be returned, with all computed statistics appended
to the object's \code{@data} slot. If a \code{grid} is defined, tree count will be returned as a RasterLayer,
with cell values equal to the number of trees in each cell. If a \code{grid} and \code{variables} are defined,
a RasterBrick (see \link[raster]{brick}) will be returned instead, with tree count and attribute statistics
stored as stacked layers.
}
\description{
Summarization tool for calculating tree counts and statistics within various spatial units.
}
\details{
Input trees can either be point locations (\link[sp]{SpatialPointsDataFrame})
or crown outlines (\link[sp]{SpatialPolygonsDataFrame}). If crown outlines (or other polygons)
are inputted, they will be partioned between spatial units according to their geographic centroids.

In addition to tree counts, statistics for the trees' attributes can also be
generated. These attributes should be defined within the \code{@data} slot of the \code{input}. Only
numeric variables are accepted.

By default, the statistics generated for each attribute will be its mean, median, standard deviation,
minimum and maximum. However, custom functions can also be used with the \code{statFuns} argument.
This should be a named list of functions, wherein each list element is given a name to represent the
statistic computed by the function.

For example: \code{list(qunt98 = function(x, ...) quantile(x, c(.98), na.rm = TRUE))}

Furthermore, custom functions should:
\itemize{
  \item Be able to accept numeric vectors.
  \item Be able to handle NA values.
  \item Have an ellipsis (three dots) in their arguments: \code{function(x, ...)}
  \item Return a single numeric value.
}
}
\examples{
# Load sample data
library(ForestTools)
library(sp)
data("kootenayTrees", "kootenayBlocks", "kootenayCrowns")

# Get total tree count
SpatialStatistics(kootenayTrees)

# Get total tree count, tree height and crown area statistics
SpatialStatistics(kootenayCrowns, variables = c("height", "crownArea"))

# Get tree count, height statistics for specific areas of interest
areaStats <- SpatialStatistics(kootenayTrees, areas = kootenayBlocks, variables = "height")

# Plot according to tree count
plot(areaStats, col = heat.colors(3)[order(areaStats$TreeCount)])

# Get tree count and height statistics for a 20 x 20 m spatial grid
gridStats <- SpatialStatistics(kootenayTrees, grid = 20, variables = "height")

# Plot gridded tree count and statistics
plot(gridStats$TreeCount)
plot(gridStats$heightMax)

}
\seealso{
\code{\link{TreeTopFinder}} \code{\link{SegmentCrowns}}
}

