% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/htsrec.R
\name{htsrec}
\alias{htsrec}
\title{Cross-sectional (contemporaneous) forecast reconciliation}
\usage{
htsrec(basef, comb, C, res, Ut, nb, mse = TRUE, corpcor = FALSE,
       type = "M", sol = "direct", keep = "list", nn = FALSE,
       nn_type = "osqp", settings = list(), bounds = NULL, W = NULL)
}
\arguments{
\item{basef}{(\mjseqn{h \times n}) matrix of base forecasts to be reconciled;
\mjseqn{h} is the forecast horizon and \mjseqn{n} is the total number of time series.}

\item{comb}{Type of the reconciliation. Except for Bottom-up, each
option corresponds to a specific (\mjseqn{n \times n}) covariance matrix:
\itemize{
    \item \bold{bu} (Bottom-up);
    \item \bold{ols} (Identity);
    \item \bold{struc} (Structural variances);
    \item \bold{wls} (Series variances) - uses res;
    \item \bold{shr} (Shrunk covariance matrix - MinT-shr) - uses res;
    \item \bold{sam} (Sample covariance matrix - MinT-sam) - uses res;
    \item \bold{w} use your personal matrix W in param \code{W}.
  }}

\item{C}{(\mjseqn{n_a \times n_b}) cross-sectional (contemporaneous) matrix
mapping the bottom level series into the higher level ones.}

\item{res}{(\mjseqn{N \times n}) in-sample residuals matrix needed when \code{comb =}
\code{\{"wls",} \code{"shr",} \code{"sam"\}}. The columns must be in
the same order as \code{basef}.}

\item{Ut}{Zero constraints cross-sectional (contemporaneous) kernel matrix
\mjseqn{(\mathbf{U}'\mathbf{y} = \mathbf{0})} spanning the null space valid
for the reconciled forecasts. It can be used instead of parameter
\code{C}, but \code{nb} (\mjseqn{n = n_a + n_b}) is needed if
\mjseqn{\mathbf{U}' \neq [\mathbf{I} \ -\mathbf{C}]}{}. If the hierarchy
admits a structural representation, \mjseqn{\mathbf{U}'} has dimension
(\mjseqn{n_a \times n}).}

\item{nb}{Number of bottom time series; if \code{C} is present, \code{nb}
and \code{Ut} are not used.}

\item{mse}{Logical value: \code{TRUE} (\emph{default}) calculates the
covariance matrix of the in-sample residuals (when necessary) according to
the original \pkg{hts} and \pkg{thief} formulation: no mean correction,
T as denominator.}

\item{corpcor}{Logical value: \code{TRUE} if \pkg{corpcor} (\enc{Schäfer}{Schafer} et
al., 2017) must be used to shrink the sample covariance matrix according to
\enc{Schäfer}{Schafer} and Strimmer (2005), otherwise the function uses the
same implementation as package \pkg{hts}.}

\item{type}{Approach used to compute the reconciled forecasts: \code{"M"} for
the projection approach with matrix M (\emph{default}), or \code{"S"} for the
structural approach with summing matrix S.}

\item{sol}{Solution technique for the reconciliation problem: either
\code{"direct"} (\emph{default}) for the closed-form matrix solution, or
\code{"osqp"} for the numerical solution (solving a linearly constrained
quadratic program using \code{\link[osqp]{solve_osqp}}).}

\item{keep}{Return a list object of the reconciled forecasts at all levels
(if keep = "list") or only the reconciled forecasts matrix (if keep = "recf").}

\item{nn}{Logical value: \code{TRUE} if non-negative reconciled forecasts
are wished.}

\item{nn_type}{"osqp" (default), "KAnn" (only \code{type == "M"}) or "sntz".}

\item{settings}{Settings for \pkg{osqp} (object \code{\link[osqp]{osqpSettings}}).
The default options are: \code{verbose = FALSE}, \code{eps_abs = 1e-5},
\code{eps_rel = 1e-5}, \code{polish_refine_iter = 100} and \code{polish = TRUE}.
For details, see the \href{https://osqp.org/}{\pkg{osqp} documentation}
(Stellato et al., 2019).}

\item{bounds}{(\mjseqn{n \times 2}) matrix containing the cross-sectional bounds:
the first column is the lower bound, and the second column is the upper bound.}

\item{W}{This option permits to directly enter the covariance matrix:
\enumerate{
  \item \code{W} must be a p.d. (\mjseqn{n \times n}) matrix or a list of \mjseqn{h} matrix (one for each forecast horizon);
  \item if \code{comb} is different from "\code{w}", \code{W} is not used.
}}
}
\value{
If the parameter \code{keep} is equal to \code{"recf"}, then the function
returns only the (\mjseqn{h \times n}) reconciled forecasts matrix, otherwise (\code{keep="all"})
it returns a list that mainly depends on what type of representation (\code{type})
and solution technique (\code{sol}) have been used:
\item{\code{recf}}{(\mjseqn{h \times n}) reconciled forecasts matrix, \mjseqn{\widetilde{\mathbf{Y}}}.}
\item{\code{W}}{Covariance matrix used for forecast reconciliation, \mjseqn{\mathbf{W}}.}
\item{\code{nn_check}}{Number of negative values (if zero, there are no values below zero).}
\item{\code{rec_check}}{Logical value: has the hierarchy been respected?}
\item{\code{varf} (\code{type="direct"})}{(\mjseqn{n \times 1}) reconciled forecasts variance vector for \mjseqn{h=1}, \mjseqn{\mbox{diag}(\mathbf{MW}}).}
\item{\code{M} (\code{type="direct"})}{Projection matrix, \mjseqn{\mathbf{M}} (projection approach).}
\item{\code{G} (\code{type="S"} and \code{type="direct"})}{Projection matrix, \mjseqn{\mathbf{G}} (structural approach, \mjseqn{\mathbf{M}=\mathbf{S}\mathbf{G}}).}
\item{\code{S} (\code{type="S"} and \code{type="direct"})}{Cross-sectional summing matrix, \mjseqn{\mathbf{S}}.}
\item{\code{info} (\code{type="osqp"})}{matrix with information in columns
for each forecast horizon \mjseqn{h} (rows): run time (\code{run_time}),
number of iteration (\code{iter}), norm of primal residual (\code{pri_res}),
status of osqp's solution (\code{status}) and polish's status
(\code{status_polish}). It will also be returned with \code{nn = TRUE} if
a solver (see \code{nn_type}) will be used.}

Only if \code{comb = "bu"}, the function returns \code{recf}, \code{S} and \code{M}.
}
\description{
Cross-sectional (contemporaneous) forecast reconciliation of a linearly constrained
(e.g., hierarchical/grouped) multiple time series.
The reconciled forecasts are calculated either through a
projection approach (Byron, 1978, see also van Erven and Cugliari, 2015, and
Wickramasuriya et al., 2019), or the equivalent structural approach by Hyndman
et al. (2011). Moreover, the classic bottom-up approach is available.
}
\details{
\loadmathjax
Let \mjseqn{\mathbf{y}} be a (\mjseqn{n \times 1}) vector of target point
forecasts which are wished to satisfy the system of linearly independent
constraints \mjsdeqn{\mathbf{U}'\mathbf{y} = \mathbf{0}_{(r \times 1)},}
where \mjseqn{\mathbf{U}'} is a (\mjseqn{r \times n}) matrix, with
rank\mjseqn{(\mathbf{U}') = r \leq n}, and \mjseqn{\mathbf{0}_{(r \times 1)}}
is a (\mjseqn{r \times 1}) null vector. Let \mjseqn{\widehat{\mathbf{y}}}
be a (\mjseqn{n \times 1}) vector of unbiased point forecasts, not
fulfilling the linear constraints (i.e., \mjseqn{\mathbf{U}'\widehat{\mathbf{y}}
\ne \mathbf{0}}).

We consider a regression-based reconciliation method assuming
that \mjseqn{\widehat{\mathbf{y}}} is related to \mjseqn{\mathbf{y}} by
\mjsdeqn{\widehat{\mathbf{y}} = \mathbf{y} + \mathbf{\varepsilon},}
where \mjseqn{\mathbf{\varepsilon}} is a (\mjseqn{n \times 1}) vector
of zero mean disturbances, with known p.d. covariance matrix
\mjseqn{\mathbf{W}}. The reconciled forecasts \mjseqn{\widetilde{\mathbf{y}}}
are found by minimizing the generalized least squares (GLS) objective
function \mjseqn{\left(\widehat{\mathbf{y}} - \mathbf{y}\right)'\mathbf{W}^{-1}
\left(\widehat{\mathbf{y}} - \mathbf{y}\right)} constrained by
\mjseqn{\mathbf{U}'\mathbf{y} = \mathbf{0}_{(r \times 1)}}:

\mjsdeqn{\widetilde{\mathbf{y}} = \mbox{argmin}_\mathbf{y} \left(\mathbf{y} -
\widehat{\mathbf{y}} \right)' \mathbf{W}^{-1} \left(\mathbf{y} -
\widehat{\mathbf{y}} \right), \quad \mbox{s.t. } \mathbf{U}'\mathbf{y} =
\mathbf{0}.}

The solution is given by
\mjsdeqn{\widetilde{\mathbf{y}}= \widehat{\mathbf{y}} - \mathbf{W}\mathbf{U}
\left(\mathbf{U}’\mathbf{WU}\right)^{-1}\mathbf{U}'\widehat{\mathbf{y}}=
\mathbf{M}\widehat{\mathbf{y}},}
where \mjseqn{\mathbf{M} = \mathbf{I}_n - \mathbf{W}\mathbf{U}\left(
\mathbf{U}’\mathbf{WU}\right)^{-1}\mathbf{U}’}
is a (\mjseqn{n \times n}) projection matrix. This solution is used by
\code{\link[FoReco]{htsrec}} when \code{type = "M"}.

Denoting with \mjseqn{\mathbf{d}_{\widehat{\mathbf{y}}} = \mathbf{0} -
\mathbf{U}'\widehat{\mathbf{y}}} the (\mjseqn{r \times 1}) vector containing
the \emph{coherency} errors of the base forecasts, we can re-state the solution as
\mjsdeqn{\widetilde{\mathbf{y}}= \widehat{\mathbf{y}} + \mathbf{WU} \left(
\mathbf{U}'\mathbf{WU}\right)^{-1}\mathbf{d}_{\widehat{y}},}
which makes it clear that the reconciliation formula simply adjusts the
vector \mjseqn{\widehat{\mathbf{y}}} with a linear
combination -- according to the smoothing matrix
\mjseqn{\mathbf{L} = \mathbf{WU} \left(\mathbf{U}’\mathbf{WU}\right)^{-1}} --
of the coherency errors of the base forecasts.

In addition, if the error term \mjseqn{\mathbf{\varepsilon}} is gaussian, the reconciliation
error \mjseqn{\widetilde{\varepsilon} = \widetilde{\mathbf{y}} - \mathbf{y}} is
a zero-mean gaussian vector with covariance matrix
\mjsdeqn{E\left( \widetilde{\mathbf{y}} - \mathbf{y}\right) \left(
\widetilde{\mathbf{y}} - \mathbf{y}\right)' = \mathbf{W} - \mathbf{WU}
\left(\mathbf{U}'\mathbf{WU}\right)^{-1}\mathbf{U}' = \mathbf{MW}.}

Hyndman et al. (2011, see also Wickramasuriya et al., 2019) propose an
equivalent, alternative formulation as for the reconciled estimates, obtained
by GLS estimation of the model
\mjsdeqn{\widehat{\mathbf{y}} = \mathbf{S}\mathbf{\beta} + \mathbf{\varepsilon},}
where \mjseqn{\mathbf{S}} is the \emph{structural summation matrix} describing
the aggregation relationships operating on \mjseqn{\mathbf{y}}, and
\mjseqn{\mathbf{\beta}} is a subset of \mjseqn{\mathbf{y}} containing the
target forecasts of the bottom level series, such that
\mjseqn{\mathbf{y} = \mathbf{S}\mathbf{\beta}}. Since the hypotheses on
\mjseqn{\mathbf{\varepsilon}} remain unchanged,
\mjsdeqn{\widetilde{\mathbf{\beta}} = \left(\mathbf{S}'\mathbf{W}^{-1}\mathbf{S}
\right)^{-1}\mathbf{S}'\mathbf{W}^{-1}\widehat{\mathbf{y}}}
is the best linear unbiased estimate of \mjseqn{\mathbf{\beta}}, and
the whole reconciled forecasts vector is given by
\mjsdeqn{\widetilde{\mathbf{y}} = \mathbf{S}\widetilde{\mathbf{\beta}} = \mathbf{SG}
\widehat{\mathbf{y}},}
where \mjseqn{\mathbf{G} = \left(\mathbf{S}'\mathbf{W}^{-1}
\mathbf{S}\right)^{-1}\mathbf{S}'\mathbf{W}^{-1}}, and \mjseqn{\mathbf{M}=\mathbf{SG}}.
This solution is used by \code{\link[FoReco]{htsrec}} when \code{type = "S"}.

\strong{Bounds on the reconciled forecasts}

The user may impose bounds on the reconciled forecasts.
The parameter \code{bounds} permits to consider lower (\mjseqn{\mathbf{a}}) and
upper (\mjseqn{\mathbf{b}}) bounds like \mjseqn{\mathbf{a} \leq
\widetilde{\mathbf{y}} \leq \mathbf{b}} such that:
\mjsdeqn{ \begin{array}{c}
a_1 \leq \widetilde{y}_1 \leq b_1 \cr
\dots \cr
a_n \leq \widetilde{y}_n \leq b_n \cr
\end{array} \Rightarrow
\mbox{bounds} = [\mathbf{a} \; \mathbf{b}] =
\left[\begin{array}{cc}
a_1 & b_1 \cr
\vdots & \vdots \cr
a_n & b_n \cr
\end{array}\right],}
where \mjseqn{a_i \in [- \infty, + \infty]} and \mjseqn{b_i \in [- \infty, + \infty]}.
If \mjseqn{y_i} is unbounded, the i-th row of \code{bounds} would be equal
to \code{c(-Inf, +Inf)}.
Notice that if the \code{bounds} parameter is used, \code{sol = "osqp"} must be used.
This is not true in the case of non-negativity constraints:
\itemize{
  \item \code{sol = "direct"}: first the base forecasts
  are reconciled without non-negativity constraints, then, if negative reconciled
  values are present, the \code{"osqp"} solver is used;
  \item \code{sol = "osqp"}: the base forecasts are
  reconciled using the \code{"osqp"} solver.
}

In this case it is not necessary to build a matrix containing
the bounds, and it is sufficient to set \code{nn = "TRUE"}.

Non-negative reconciled forecasts may be obtained by setting \code{nn_type} alternatively as:
\itemize{
  \item \code{nn_type = "KAnn"} (Kourentzes and Athanasopoulos, 2021)
  \item \code{nn_type = "sntz"} ("set-negative-to-zero")
  \item \code{nn_type = "osqp"} (Stellato et al., 2020)
}
}
\examples{
data(FoReco_data)
# monthly base forecasts
id <- which(simplify2array(strsplit(colnames(FoReco_data$base),split = "_"))[1, ] == "k1")
mbase <- t(FoReco_data$base[, id])
# monthly residuals
id <- which(simplify2array(strsplit(colnames(FoReco_data$res),split = "_"))[1, ] == "k1")
mres <- t(FoReco_data$res[, id])
obj <- htsrec(mbase, C = FoReco_data$C, comb = "shr", res = mres)

# FoReco is able to work also with covariance matrix that are not equal
# across all the forecast horizon. For example, we can consider the
# normalized squared differences (see Di Fonzo and Marini, 2011) where
# Wh = diag(|yh|):
Wh <- lapply(split(mbase, row(mbase)), function(x) diag(abs(x)))

# Now we can introduce the list of the covariance matrix in htsrec throught
# the parameter "W" and setting comb = "w".
objh <- htsrec(mbase, C = FoReco_data$C, W = Wh, comb = "w")

}
\references{
Byron, R.P. (1978), The estimation of large social accounts matrices,
\emph{Journal of the Royal Statistical Society A}, 141, 3, 359-367.

Di Fonzo, T., Girolimetto, D. (2020), Cross-Temporal Forecast Reconciliation:
Optimal Combination Method and Heuristic Alternatives, Department of Statistical
Sciences, University of Padua, \href{https://arxiv.org/abs/2006.08570}{arXiv:2006.08570}.

Di Fonzo, T., Marini, M. (2011), Simultaneous and two-step reconciliation of
systems of time series: methodological and practical issues,
\emph{Journal of the Royal Statistical Society. Series C (Applied Statistics)},
60, 2, 143-164

Hyndman, R.J., Ahmed, R.A., Athanasopoulos, G., Shang, H.L.(2011),
Optimal combination forecasts for hierarchical time series,
\emph{Computational Statistics & Data Analysis}, 55, 9, 2579-2589.

Kourentzes, N., Athanasopoulos, G. (2021),
Elucidate structure in intermittent demand series,
\emph{European Journal of Operational Research}, 288, 1, pp. 141–152.

\enc{Schäfer}{Schafer}, J.L., Opgen-Rhein, R., Zuber, V., Ahdesmaki, M.,
Duarte Silva, A.P., Strimmer, K. (2017), \emph{Package `corpcor'}, R
package version 1.6.9 (April 1, 2017), \href{https://CRAN.R-project.org/package=corpcor}{https://CRAN.R-project.org/package= corpcor}.

\enc{Schäfer}{Schafer}, J.L., Strimmer, K. (2005), A Shrinkage Approach to Large-Scale Covariance
Matrix Estimation and Implications for Functional Genomics, \emph{Statistical
Applications in Genetics and Molecular Biology}, 4, 1.

Stellato, B., Banjac, G., Goulart, P., Bemporad, A., Boyd, S. (2020). OSQP:
An Operator Splitting Solver for Quadratic Programs, \emph{Mathematical Programming Computation},
12, 4, 637-672.

Stellato, B., Banjac, G., Goulart, P., Boyd, S., Anderson, E. (2019), OSQP:
Quadratic Programming Solver using the `OSQP' Library, R package version 0.6.0.3
(October 10, 2019), \href{https://CRAN.R-project.org/package=osqp}{https://CRAN.R-project.org/package=osqp}.

van Erven, T., Cugliari, J. (2015), Game-theoretically Optimal Reconciliation
of Contemporaneous Hierarchical Time Series Forecasts, in Antoniadis, A.,
Poggi, J.M., Brossat, X. (eds.), \emph{Modeling and Stochastic Learning for
Forecasting in High Dimensions}, Berlin, Springer, 297-317.

Wickramasuriya, S.L., Athanasopoulos, G., Hyndman, R.J. (2019), Optimal forecast
reconciliation for hierarchical and grouped time series through trace minimization,
\emph{Journal of the American Statistical Association}, 114, 526, 804-819.
}
\seealso{
Other reconciliation procedures: 
\code{\link{cstrec}()},
\code{\link{ctbu}()},
\code{\link{iterec}()},
\code{\link{lccrec}()},
\code{\link{octrec}()},
\code{\link{tcsrec}()},
\code{\link{tdrec}()},
\code{\link{thfrec}()}
}
\concept{reconciliation procedures}
\keyword{bottom-up}
