% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/network-summary.R
\docType{methods}
\name{NetworkSummary}
\alias{NetworkSummary}
\alias{NetworkSummary,ContactTrace-method}
\alias{NetworkSummary,data.frame-method}
\title{\code{NetworkSummary}}
\usage{
NetworkSummary(x, ...)

\S4method{NetworkSummary}{ContactTrace}(x)

\S4method{NetworkSummary}{data.frame}(
  x,
  root,
  tEnd = NULL,
  days = NULL,
  inBegin = NULL,
  inEnd = NULL,
  outBegin = NULL,
  outEnd = NULL
)
}
\arguments{
\item{x}{a ContactTrace object or a \code{data.frame} with
movements of animals between holdings, see \code{\link{Trace}} for
details.}

\item{...}{Additional arguments to the method}

\item{root}{vector of roots to calculate network summary for.}

\item{tEnd}{the last date to include ingoing movements. Defaults
to \code{NULL}}

\item{days}{the number of previous days before tEnd to include
ingoing movements. Defaults to \code{NULL}}

\item{inBegin}{the first date to include ingoing
movements. Defaults to \code{NULL}}

\item{inEnd}{the last date to include ingoing movements. Defaults
to \code{NULL}}

\item{outBegin}{the first date to include outgoing
movements. Defaults to \code{NULL}}

\item{outEnd}{the last date to include outgoing movements. Defaults
to \code{NULL}}
}
\value{
A \code{data.frame} with the following columns:
\describe{
  \item{root}{
    The root of the contact tracing
  }

  \item{inBegin}{
    Equals inBegin in \code{\link{Trace}}
  }

  \item{inEnd}{
    Equals inEnd in \code{\link{Trace}}
  }

  \item{outBegin}{
    Equals outBegin in \code{\link{Trace}}
  }

  \item{outEnd}{
    Equals outEnd in \code{\link{Trace}}
  }

  \item{inDegree}{
    The \code{\link{InDegree}} of the contact tracing
  }

  \item{outDegree}{
    The \code{\link{OutDegree}} of the contact tracing
  }

  \item{ingoingContactChain}{
    The \code{\link{IngoingContactChain}} of the contact tracing
  }

  \item{outgoingContactChain}{
    The \code{\link{OutgoingContactChain}} of the contact tracing
  }
}
}
\description{
\code{NetworkSummary} gives a summary of the contact tracing including the
time-window, \code{\link{InDegree}}, \code{\link{OutDegree}},
\code{\link{IngoingContactChain}} and \code{\link{OutgoingContactChain}}.
}
\details{
The time period used for \code{NetworkSummary} can either be specified
using \code{tEnd} and \code{days} or \code{inBegin}, \code{inEnd},
\code{outBegin} and \code{outEnd}.

If using \code{tEnd} and \code{days}, the time period for ingoing
and outgoing contacts ends at \code{tEnd} and starts at
\code{days} prior to \code{tEnd}. The network summary will be
calculated for each combination of \code{root}, \code{tEnd} and
\code{days}.

An alternative way is to use \code{inBegin}, \code{inEnd},
\code{outBegin} and \code{outEnd}. The time period for ingoing
contacts starts at inBegin and ends at inEndDate.  For outgoing
contacts the time period starts at outBegin and ends at outEnd.
The vectors \code{root} \code{inBegin}, \code{inEnd},
\code{outBegin} and \code{outEnd} must have the same lengths and
the network summary will be calculated for each index of them.

The movements in \code{NetworkSummary} is a \code{data.frame}
with the following columns:
\describe{

  \item{source}{
    an integer or character identifier of the source holding.
  }

  \item{destination}{
    an integer or character identifier of the destination holding.
  }

  \item{t}{
    the Date of the transfer
  }

  \item{id}{
    an optional character vector with the identity of the animal.
  }

  \item{n}{
    an optional numeric vector with the number of animals moved.
  }

  \item{category}{
    an optional character or factor with category of the animal e.g. Cattle.
  }
}
}
\section{Methods}{

\describe{
  \item{\code{signature(x = "ContactTrace")}}{
    Get the network summary for the ingoing and outgoing
    \code{Contacts} of a ContactTrace object.
  }

  \item{\code{signature(x = "data.frame")}}{
    Get the network summary for a data.frame with movements,
    see details and examples.
  }
}
}

\examples{
\dontrun{

## Load data
data(transfers)

## Perform contact tracing using tEnd and days
contactTrace <- Trace(movements=transfers,
                      root=2645,
                      tEnd='2005-10-31',
                      days=91)

## Calculate network summary from a ContactTrace object
ns.1 <- NetworkSummary(contactTrace)

## Calculate network summary using tEnd and days
ns.2 <- NetworkSummary(transfers,
                       root=2645,
                       tEnd='2005-10-31',
                       days=91)

## Check that the result is identical
identical(ns.1, ns.2)

## Calculate network summary using inBegin, inEnd
## outBegin and outEnd
ns.3 <- NetworkSummary(transfers,
                       root=2645,
                       inBegin='2005-08-01',
                       inEnd='2005-10-31',
                       outBegin='2005-08-01',
                       outEnd='2005-10-31')

## Check that the result is identical
identical(ns.2, ns.3)

## When calculating the network summary for a data.frame of movements
## a data.frame for each combination of root, tEnd and days are returned.
root <- c(1,2,3)
tEnd <- c("2005-09-01", "2005-10-01")
days <- c(30, 45)

## The network summary are calculated at the following
## 12 combinations.
## root = 1, tEnd = "2005-09-01", days = 30
## root = 1, tEnd = "2005-09-01", days = 45
## root = 1, tEnd = "2005-10-01", days = 30
## root = 1, tEnd = "2005-10-01", days = 45
## root = 2, tEnd = "2005-09-01", days = 30
## root = 2, tEnd = "2005-09-01", days = 45
## root = 2, tEnd = "2005-10-01", days = 30
## root = 2, tEnd = "2005-10-01", days = 45
## root = 3, tEnd = "2005-09-01", days = 30
## root = 3, tEnd = "2005-09-01", days = 45
## root = 3, tEnd = "2005-10-01", days = 30
## root = 3, tEnd = "2005-10-01", days = 45
NetworkSummary(transfers, root, tEnd, days)

## Create a network summary for all included herds
## First extract all source and destination from the dataset
root <- sort(unique(c(transfers$source,
                      transfers$destination)))

## Perform contact tracing using tEnd and days
result.1 <- NetworkSummary(transfers,
                           root=root,
                           tEnd='2005-10-31',
                           days=90)

## Perform contact tracing using inBegin, inEnd, outBegin and outEnd.
result.2 <- NetworkSummary(transfers,
                           root=root,
                           inBegin=rep('2005-08-02', length(root)),
                           inEnd=rep('2005-10-31', length(root)),
                           outBegin=rep('2005-08-02', length(root)),
                           outEnd=rep('2005-10-31', length(root)))
}
}
\references{
\itemize{
  \item Dube, C., et al., A review of network analysis terminology
    and its application to foot-and-mouth disease modelling and policy
    development. Transbound Emerg Dis 56 (2009) 73-85, doi:
    10.1111/j.1865-1682.2008.01064.x

  \item Noremark, M., et al., Network analysis
    of cattle and pig movements in Sweden: Measures relevant for
    disease control and riskbased surveillance.  Preventive Veterinary
    Medicine 99 (2011) 78-90, doi: 10.1016/j.prevetmed.2010.12.009
}
}
\keyword{methods}
