% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/process_date.r
\name{as_date_ecotox}
\alias{as_date_ecotox}
\title{Values represented by ECOTOX \code{character} to dates}
\usage{
as_date_ecotox(x, dd = 1L, mm = 1L, nr = 1L, ..., warn = TRUE)
}
\arguments{
\item{x}{A vector of \code{character} strings. It expects fields as commonly returned
from the ECOTOX database.}

\item{dd}{Replacement values for unspecified days in a date. Defaults to \code{1L}. If
you want dates with unspecified days to result in \code{NA}, use \code{dd = -1L}.}

\item{mm}{Replacement values for unspecified months in a date. Defaults to \code{1L}. If
you want dates with unspecified months to result in \code{NA}, use \code{mm = -1L}.}

\item{nr}{Replacement values for generically unspecified values in a date.
Defaults to \code{1L}. If you want dates with unspecified values to result in \code{NA},
use \code{nr = -1L}.}

\item{...}{Passed to \code{as.Date()}.}

\item{warn}{If set to \code{FALSE} warnings while converting text to dates are suppressed.}
}
\value{
A vector of \code{Date} class objects with the same length as \code{x}.
}
\description{
\ifelse{html}{\href{https://lifecycle.r-lib.org/articles/stages.html#experimental}{\figure{lifecycle-experimental.svg}{options: alt='[Experimental]'}}}{\strong{[Experimental]}} Similar to \code{as.Date()}, but it also
performs some text sanitising before coercing text to dates.
}
\details{
The following steps are performed (in the order as listed)
to sanitise text before coercing it to numerics:
\itemize{
\item Trim whitespaces
\item Replace hyphens with forward slashes
\item Replace double forward slashes, forward slashes followed by a zero and spaces,
with a single forward slash
\item Replace \code{"mm"} or \code{"dd"} (case insensitive) with the value specified as argument.
Add a forward slash to it when missing.
\item Treat \code{"na"}, \code{"nr"}, \code{"xx"} and \code{"00"} (case insensitive) as unreported
values when followed by a forward slash. Replace it with the \code{nr} argument
\item Remove alphabetical characters when directly followed by a numerical character.
\item Replace literal month names with its numerical calendar value (1-12).
\item When the date consists of one value, assume it is a calender year and add
\code{dd} and \code{mm} as day and month value.
\item If a date consists of two numbers, assume it is month, followed by year. In that
case insert the \code{dd} value for the day.
}

It is your own responsibility to check if the sanitising steps are appropriate for
your analyses.
}
\examples{
## a vector of commonly used notations in the database to represent
## dates. Most frequent format is \%m/\%d/\%Y
char_date <- c("5-19-1987   ", "5/dd/2021", "3/19/yyyy", "1985", "mm/19/1999",
               "October 2004", "nr/nr/2015")

as_date_ecotox(char_date)

## Set unspecified days to 15:
as_date_ecotox(char_date, dd = 15L)

## Unspecified days should result in NA:
as_date_ecotox(char_date, dd = -1L)

## Set unspecified months to 6:
as_date_ecotox(char_date, mm = 6L)

## Set generically unspecified value to 6:
as_date_ecotox(char_date, nr = 6L)
}
\seealso{
Other ecotox-sanitisers: 
\code{\link{as_numeric_ecotox}()},
\code{\link{as_unit_ecotox}()},
\code{\link{mixed_to_single_unit}()},
\code{\link{process_ecotox_dates}()},
\code{\link{process_ecotox_numerics}()},
\code{\link{process_ecotox_units}()}
}
\author{
Pepijn de Vries
}
\concept{ecotox-sanitisers}
