% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/mutate_edge_attrs_ws.R
\name{mutate_edge_attrs_ws}
\alias{mutate_edge_attrs_ws}
\title{Mutate edge attribute values for a selection of edges}
\usage{
mutate_edge_attrs_ws(graph, ...)
}
\arguments{
\item{graph}{a graph object of class
\code{dgr_graph}.}

\item{...}{expressions used for the mutation
of edge attributes. LHS of each expression is
either an existing or new edge attribute name.
The RHS can consist of any valid R code that
uses edge attributes as variables. Expressions
are evaluated in the order provided, so, edge
attributes created or modified are ready to
use in subsequent expressions.}
}
\value{
a graph object of class
\code{dgr_graph}.
}
\description{
Within a graph's internal edge
data frame (edf), mutate edge attribute
values only for edges in a selection by
using one or more expressions.
}
\examples{
# Create a graph with 3 edges
# and then select edge `1`
graph <-
  create_graph() \%>\%
  add_path(n = 4) \%>\%
  set_edge_attrs(
    edge_attr = width,
    values = c(3.4, 2.3, 7.2)) \%>\%
  select_edges(edges = 1)

# Get the graph's internal edf
# to show which edge attributes
# are available
get_edge_df(graph)
#>   id from to  rel width
#> 1  1    1  2 <NA>   3.4
#> 2  2    2  3 <NA>   2.3
#> 3  3    3  4 <NA>   7.2

# Mutate the `width` edge
# attribute for the edges
# only in the active selection
# of edges (edge `1`); here,
# we divide each value in the
# selection by 2
graph <-
  graph \%>\%
  mutate_edge_attrs_ws(
    width = width / 2)

# Get the graph's internal
# edf to show that the edge
# attribute `width` had its
# values changed
get_edge_df(graph)
#>   id from to  rel width
#> 1  1    1  2 <NA>   1.7
#> 2  2    2  3 <NA>   2.3
#> 3  3    3  4 <NA>   7.2

# Create a new edge attribute,
# called `length`, that is the
# log of values in `width` plus
# 2 (and, also, round all values
# to 2 decimal places)
graph <-
  graph \%>\%
  clear_selection() \%>\%
  select_edges(edges = 2:3) \%>\%
  mutate_edge_attrs_ws(
    length = (log(width) + 2) \%>\%
               round(2))

# Get the graph's internal edf
# to show that the edge attribute
# values had been mutated only
# for edges `2` and `3` (since
# edge `1` is excluded, an NA
# value is applied)
get_edge_df(graph)
#>   id from to  rel width length
#> 1  1    1  2 <NA>   1.7     NA
#> 2  2    2  3 <NA>   2.3   2.83
#> 3  3    3  4 <NA>   7.2   3.97

# Create a new edge attribute
# called `area`, which is the
# product of the `width` and
# `length` attributes
graph <-
  graph \%>\%
  mutate_edge_attrs_ws(
    area = width * length)

# Get the graph's internal edf
# to show that the edge attribute
# values had been multiplied
# together (with new attr `area`)
# for nodes `2` and `3`
get_edge_df(graph)
#>   id from to  rel width length   area
#> 1  1    1  2 <NA>   1.7     NA     NA
#> 2  2    2  3 <NA>   2.3   2.83  6.509
#> 3  3    3  4 <NA>   7.2   3.97 28.584

# We can invert the selection
# and mutate edge `1` several
# times to get an `area` value
# for that edge
graph <-
  graph \%>\%
  invert_selection() \%>\%
  mutate_edge_attrs_ws(
    length = (log(width) + 5) \%>\%
               round(2),
    area = width * length)

# Get the graph's internal edf
# to show that the 2 mutations
# occurred for edge `1`, yielding
# non-NA values for its edge
# attributes without changing
# those of the other edges
get_edge_df(graph)
#>   id from to  rel width length   area
#> 1  1    1  2 <NA>   1.7   5.53  9.401
#> 2  2    2  3 <NA>   2.3   2.83  6.509
#> 3  3    3  4 <NA>   7.2   3.97 28.584
}
