
# User-facing helpers -----------------------------------------------------

#' Get the code from a design
#' @param design A design that has code as an attribute.
#' @export
get_design_code <- function(design) attr(design, "code")


#' Argument matching with defaults
#' 
#' This is a version of \code{\link{match.call}} which also includes default arguments.
#' 
#' @param definition a function, by default the function from which match.call is called. See details.
#' @param call an unevaluated call to the function specified by definition, as generated by call.
#' @param expand.dots ogical. Should arguments matching \code{...} in the call be included or left as a \code{...} argument?
#' @param envir an environment, from which the \code{...} in call are retrieved, if any.
#' 
#' @return An object of class call.
#' 
#' @author Neal Fultz
#' @references \url{http://stackoverflow.com/questions/14397364/match-call-with-default-arguments/}
#' @export
#' 
#' @examples 
#' 
#' foo <- function(x=NULL,y=NULL,z=4, dots=TRUE, ...) {
#'   match.call.defaults(expand.dots=dots)
#' }
#' 
#' foo(4,nugan='hand')
#' foo(dots=FALSE,who='ami')
#' 

match.call.defaults <- function(definition = sys.function(sys.parent()),
                                call = sys.call(sys.parent()),
                                expand.dots = TRUE,
                                envir = parent.frame(2L)) {
  call <- match.call(definition, call, expand.dots, envir)
  formals <- formals(definition)
  
  if(expand.dots && '...' %in% names(formals))
    formals[['...']] <- NULL
  
  for(i in setdiff(names(formals), names(call)))
    call[i] <- list( formals[[i]] )
  
  
  match.call(definition, call, TRUE, envir)
}

# Internal helpers for {{{ }}} approach -----------------------------------

# This is the core function for grabbing code when using the {{{ }}} approach:

construct_design_code <- function(designer, args, arguments_as_values = FALSE, exclude_args = NULL){
  # get the code for the design 
  txt <- as.character(getSrcref(designer))
  if(length(txt)==0){
    txt <- find_triple_bracket(designer)
  }else{
    open <- grep("[{]{3}", txt)
    close <- grep("[}]{3}", txt)
    
    if(length(open) != 1) stop("could not find opening tag in ", substitute(designer))
    if(length(close) != 1) stop("could not find opening tag in ", substitute(designer))
    txt <- txt[seq(open + 1, close - 1)]
  }
  
  indentation <- strsplit(txt[1], "")[[1]]
  indentation <- indentation[cumprod(indentation == " ") == 1]
  indentation <- paste0("^", paste(indentation, collapse=""))
  
  code <- sub(indentation, "", txt)
  
  # Get names of arguments   
  arg_names <- names(args[-1])
  
  # If true, arguments are parsed as values -- be careful with functions
  if(arguments_as_values){
    # Evaluate args in order provided in formals
    for(j in 1:length(arg_names)) eval(parse(text = paste(arg_names[j], " <- ", args[arg_names[j]])))  
    arg_vals <- sapply(arg_names, function(x) eval(parse(text = paste0("c(", paste(x, collapse = ","), ")"))))
    # convert args to text
    args_text <- paste(sapply(arg_names, function(x) paste(x, " <- ", arg_vals[x])))
  } else {
    # convert args to text
    args_text <- as.character(sapply(names(args[2:length(args)]), function(x) paste0(x, " <- ", deparse(args[[x]]))))
  }
  
  # optionally exclude arguments
  if(!is.null(exclude_args)) args_text <- args_text[!(arg_names %in% exclude_args)]
  
  # add arguments and code
  code <- c(args_text, "", code)
  
  code
}

# These functions find triple braces when there is no source code 
find_triple_bracket <- function(f){
  clean <- function(ch, n=length(ch)-1) ch[2:n]
  ret <- Filter(pred, body(f))
  if(length(ret) == 0) "" else clean(deparse(ret[[1]][[2]][[2]]))
}
pred <- function(expr, depth=3) {
  (depth == 0) || (
    length(expr) > 1 &&
      expr[[1]] == as.symbol('{') &&
      Recall(expr[[2]], depth - 1)
  )
}


# Internal functions for substitute approach ------------------------------

# This function gets names of arguments and excludes any fixed ones 
return_args <- function(args, fixes){
  # Get names of arguments   
  arg_names <- names(args[2:(length(args)-1)])
  
  # Exclude any fixed arguments
  if(!is.null(fixes)) arg_names <- arg_names[!(arg_names%in%names(fixes))]
  
  # Format
  sapply(arg_names, function(x) paste0(x, " <- ", deparse(args[[x]])))
}

# This function cleans code comments -- be careful with braces
clean_code <- function(code) {
  out <- strsplit(paste(code), '\n')
  out <- gsub("\\{|\\}", "", out)
  out <- gsub("#", " \n #", out)
  out
}



