% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/Onoue.Friberg.R
\name{Onoue.Friberg}
\alias{Onoue.Friberg}
\title{POMP PK/PD model for docetaxel, combining Onoue et al (2016) with Friberg et
al (2002)}
\usage{
Onoue.Friberg(N, cycle.length.days = 21, data = data.frame(time =
  c(seq(0, 1.95, 0.05), seq(2, cycle.length.days * 24, 1)), y = NA),
  delta.t = 0.1)
}
\arguments{
\item{N}{Size of simulated population.}

\item{cycle.length.days}{Duration (in days) of chemotherapy cycle to be
simulated.}

\item{data}{Passed through as the \code{data} argument of the \code{pomp}
constructor.}

\item{delta.t}{Time-step (in hours) of pomp's \code{euler} plug-in.}
}
\value{
No value is returned; rather, the function sets global variables
in package environment \code{DTAT::sim}.
}
\description{
This function produces a POMP model combining docetaxel pharmacokinetics
(PK) drawn from Table 2 of Onoue et al (2016) with myelosuppression dynamics
drawn from Friberg et al (2002). This model enables simulation of
neutrophil-guided dose titration of docetaxel, as done in Norris (2017).
}
\examples{
# Reproduce the sim$pkpd model and sim$pop population from reference #3:
library(pomp)
Onoue.Friberg(N=25)
sim$pop # NB: this differs from pop of original paper...

# Whereas the present version of Onoue.Friberg() draws simulated populations
# using pomp::rprior(), to reproduce the original F1000Research paper [3] we
# re-draw sim$pop as originally & prosaically done (see https://osf.io/vwnqz/):
set.seed(2016)
N <- 25
dtx.mm <- 0.808 # molar mass (mg/µM) of docetaxel
sim$pop$Circ0 <- rlnorm(N, meanlog=log(5050), sdlog=0.42) # units=cells/mm^3
sim$pop$MTT <- rlnorm(N, meanlog=log(89.3), sdlog=0.16)  # mean transit time
sim$pop$gamma <- rlnorm(N, meanlog=log(0.163), sdlog=0.039) # feedback factor
sim$pop$Emax <- rlnorm(N, meanlog=log(83.9), sdlog=0.33)
sim$pop$EC50 <- rlnorm(N, meanlog=log(7.17*dtx.mm), sdlog=0.50)
# PK params from 2-compartment docetaxel model of Onoue et al (2016)
sim$pop$CL <- rlnorm(N, meanlog=log(32.6), sdlog=0.295)
sim$pop$Q  <- rlnorm(N, meanlog=log(5.34), sdlog=0.551)
sim$pop$Vc <- rlnorm(N, meanlog=log(5.77), sdlog=0.1) # Onoue gives no CV\% for V1
sim$pop$Vp <- rlnorm(N, meanlog=log(11.0), sdlog=0.598) # Called 'V2' in Onoue
sim$pop$kTR=4/sim$pop$MTT

# Now we run the sim$pkpd model, separately for each of N simultated individuals:
allout <- data.frame() # accumulator for N individual ODE solutions
for (id in 1:sim$N) {
  out <- trajectory(sim$pkpd,
                    params=c(sim$pop[sim$pop$id==id, -which(names(sim$pop) \%in\% c('id','MTT'))]
                             , sigma=0.05, dose=100, duration=1),
                    format="data.frame")
  # drop 'traj' and shift 'time' to first column
  out <- out[,c('time',setdiff(colnames(out),c('time','traj')))]
  out$id <- paste("id",id,sep="")
  allout <- rbind(allout, out)
}

library(Hmisc)
allout <- upData(allout
                 , id = ordered(id, levels=paste("id",1:sim$N,sep=""))
                 , units=c(Prol="cells/mm^3", Tx.1="cells/mm^3",
                           Tx.2="cells/mm^3", Tx.3="cells/mm^3",
                           Circ="cells/mm^3", CircMin="cells/mm^3",
                           tNadir="hours", Cc="ng/mL", Cp="ng/mL",
                           time="hours"), print=FALSE)

library(tidyr)
cout <- gather(allout, key="Series", value="Concentration"
, Cc, Cp
, factor_key = TRUE)

label(cout$Concentration) <- "Drug Concentration"

# Figure 1 from reference [3]:
library(RColorBrewer)
xYplot(Concentration ~ time | id, group=Series
       , data=cout, subset=time<6
       , layout=c(5,NA)
       , type='l', as.table=TRUE
       , label.curves=FALSE
       , par.settings=list(superpose.line=list(lwd=2,col=brewer.pal(4,"PRGn")[c(1,4)]))
       , scales=list(y=list(log=TRUE, lim=c(10^-3,10^1)))
       , auto.key=list(points=FALSE, lines=TRUE, columns=2))

mout <- gather(allout, key="Series", value="ANC"
, Prol, Tx.1, Tx.2, Tx.3, Circ
, factor_key = TRUE)

mout <- upData(mout
               , time = time/24
               , units = c(time="days")
               , print = FALSE)

# Figure 3 from citation [3]:
xYplot(ANC ~ time | id, group=Series
       , data=mout
       , layout=c(5,5)
       , type='l', as.table=TRUE
       , label.curves=FALSE
       , par.settings=list(superpose.line=list(lwd=2,col=brewer.pal(11,"RdYlBu")[c(1,3,4,8,10)]))
       , scales=list(y=list(log=TRUE, lim=c(100,15000), at=c(200, 500, 1000, 2000, 5000, 10000)))
       , auto.key=list(points=FALSE, lines=TRUE, columns=5))

}
\references{
\enumerate{
\item Onoue H, Yano I, Tanaka A, Itohara K, Hanai A, Ishiguro H, et al.
Significant effect of age on docetaxel pharmacokinetics in Japanese
female breast cancer patients by using the population modeling approach.
\emph{Eur J Clin Pharmacol}. 2016 Jun;72(6):703-10.
doi:10.1007/s00228-016-2031-3.
\item Friberg LE, Henningsson A, Maas H, Nguyen L, Karlsson MO. Model of
chemotherapy-induced myelosuppression with parameter consistency across
drugs. \emph{J Clin Oncol}. 2002 Dec 15;20(24):4713-21.
doi:10.1200/JCO.2002.02.140.
\item Norris DC. Dose Titration Algorithm Tuning (DTAT) should supersede
\sQuote{the} Maximum Tolerated Dose (MTD) in oncology dose-finding trials.
\emph{F1000Research}. 2017;6:112. doi:10.12688/f1000research.10624.3.
\url{https://f1000research.com/articles/6-112/v3}
}
}
\seealso{
\code{\link{pomp}}, \code{\link{sim}}
}
\author{
David C. Norris
}
