#' Selecting optimal number of mixture components based on various criteria
#'
#' \code{hyperparam.J} evaluates criterion for each \code{icp.torus} objects, and select
#'   the optimal number of mixture components based on the evaluated criterion.
#'
#' @param data n x d matrix of toroidal data on \eqn{[0, 2\pi)^d}
#'   or \eqn{[-\pi, \pi)^d}
#' @param icp.torus.objects a list whose elements are icp.torus objects, generated by
#'   \code{icp.torus.score}.
#' @param option a string one of "risk", "AIC", or "BIC", which determines the criterion
#'   for the model selection. "risk" is based on the negative log-likelihood, "AIC" for the
#'   Akaike Information Criterion, and "BIC" for the Bayesian Information Criterion.
#' @return returns a list object which contains a \code{data.frame} for
#'   the evaluated criterion corresponding to each number of components, the optimal
#'   number of components, and the corresponding \code{icp.torus} object.
#' @export
#' @seealso \code{\link[ClusTorus]{icp.torus.score}}, \code{\link[ClusTorus]{hyperparam.torus}},
#'   \code{\link[ClusTorus]{hyperparam.alpha}}
#' @references Akaike (1974), "A new look at the statistical model identification",
#'   Schwarz, Gideon E. (1978), "Estimating the dimension of a model"
#' @examples
#' \donttest{
#' data <- toydata2[, 1:2]
#' n <- nrow(data)
#' split.id <- rep(2, n)
#' split.id[sample(n, floor(n/2))] <- 1
#' Jvec <- 3:35
#' icp.torus.objects <- list()
#' for (j in Jvec){
#'   icp.torus.objects[[j]] <- icp.torus.score(data, split.id = split.id, method = "kmeans",
#'                                             kmeansfitmethod = "ge", init = "h",
#'                                             param = list(J = j), verbose = TRUE)
#' }
#' hyperparam.J(data, icp.torus.objects, option = "risk")
#' }

hyperparam.J <- function(data, icp.torus.objects, option = c("risk", "AIC", "BIC")){

  if (is.null(data)) {stop("data must be input.")}
  if (is.null(icp.torus.objects)) {stop("icp.torus.objects must be input.")}

  if (!is.matrix(data)) {data <- as.matrix(data)}
  data <- on.torus(data)

  option <- match.arg(option)

  Jvec <- c()
  for (j in 1:length(icp.torus.objects)){
    if (!is.null(icp.torus.objects[[j]])){
      Jvec <- c(Jvec, j)
    }
  }
  split.id <- icp.torus.objects[[Jvec[1]]]$split.id
  d <- ncol(data)
  n2 <- icp.torus.objects[[Jvec[1]]]$n2

  # option default is risk.
  if (is.null(option)) { option <- "risk" }

  if(!is.null(icp.torus.objects[[Jvec[1]]]$mixture)) {
    method <- "mixture"
    mixturefitmethod <- icp.torus.objects[[Jvec[1]]]$mixture$fittingmethod
  } else if(!is.null(icp.torus.objects[[Jvec[1]]]$kmeans)) {
    method <- "kmeans"
    kmeansfitmethod <- icp.torus.objects[[Jvec[1]]]$kmeans$fittingmethod
  } else {stop("method kde is not supported.")}

  output <- list()
  IC <- data.frame()
  # 1. kmeans -----------------------------------------------------
  if (method == "kmeans"){

    # preparing the number of model parameters
    if (kmeansfitmethod == "homogeneous-circular"){
      k <- d + 1
    } else if (kmeansfitmethod == "heterogeneous-circular"){
      k <- d + 2
    } else {k <- (d + 1)*(d + 2)/2}
    for (j in Jvec){

      # counting the singular matrices
      nsingular <- 0
      if (is.null(icp.torus.objects[[j]])) {next}
      # approximated 2 * log-likelihood for normal approximation
      sum.conformity.scores <- sum(icp.torus.objects[[j]]$kmeans$score_sphere)

      # evaluating 2 * log-likelihood for AIC/BIC
      if (option != "risk") {
        X1 <- data[split.id == 1, ]
        n2 <- nrow(X1)
        ehatj <- ehat.eval(X1, icp.torus.objects[[j]]$kmeans$spherefit)
        sum.conformity.scores <- sum(do.call(pmax, as.data.frame(ehatj)))

        for (jj in 1:j){
          if(icp.torus.objects[[j]]$kmeans$spherefit$Sigmainv[[jj]][1, 1] == 1e+6){
            nsingular <- nsingular + 1
          }
        }
      }

      # penalty for risk/AIC/BIC
      penalty <- ifelse(option == "AIC", 2,
                        ifelse(option == "BIC", log(n2), 0))

      # evaluate risk/AIC/BIC
      # nsingular term corrects the conformity score for the singular matrices
      criterion <- - sum.conformity.scores + 2 * n2 * d * log(2 * pi) + k * j * penalty +
        ifelse(option != "risk", nsingular * (log(1e+6^(d - 2))), 0)
      IC <- rbind(IC, data.frame(J = j, criterion = criterion))
    }

    IC.index <- which.min(IC$criterion)
    Jhat <- IC[IC.index, 1]
  }
  # 2. mixture ----------------------------------------------------
  else if (method == "mixture"){

    # preparing the number of model parameters
    if (mixturefitmethod == "circular"){
      k <- d + 2
    } else if (mixturefitmethod == "axis-aligned"){
      k <- 2 * d + 1
    } else if (mixturefitmethod == "general"){
      k <- (d + 1)*(d + 2)/2
    } else stop("Bayesian is not implemented yet.")

    for (j in Jvec){
      if (is.null(icp.torus.objects[[j]])) {next}
      # likelihood for mixture model
      sum.conformity.scores <- sum(log(icp.torus.objects[[j]]$mixture$score))

      # evaluating log-likelihood for AIC/BIC
      if (option != "risk") {
        X1 <- data[split.id == 1, ]
        n2 <- nrow(X1)
        phat <- BAMBI::dvmsinmix(X1, kappa1 = icp.torus.objects[[j]]$mixture$fit$parammat[2, ],
                                 kappa2 = icp.torus.objects[[j]]$mixture$fit$parammat[3, ],
                                 kappa3 = icp.torus.objects[[j]]$mixture$fit$parammat[4, ],
                                 mu1 = icp.torus.objects[[j]]$mixture$fit$parammat[5, ],
                                 mu2 = icp.torus.objects[[j]]$mixture$fit$parammat[6, ],
                                 pmix = icp.torus.objects[[j]]$mixture$fit$parammat[1, ], log = TRUE)
        sum.conformity.scores <- sum(phat)
      }

      # penalty for risk/AIC/BIC
      penalty <- ifelse(option == "AIC", 2,
                        ifelse(option == "BIC", log(n2), 0))

      # evaluate risk/AIC/BIC
      criterion <- - 2 * sum.conformity.scores + k * j * penalty
      IC <- rbind(IC, data.frame(J = j, criterion = criterion))
    }

    IC.index <- which.min(IC$criterion)
    Jhat <- IC[IC.index, 1]
  }

  output$IC.results <- IC
  output$Jhat <- Jhat
  output$icp.torus <- icp.torus.objects[[Jhat]]

  return(output)
}
