% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/CST_Analogs.R
\name{Analogs}
\alias{Analogs}
\title{Analogs based on large scale fields.}
\usage{
Analogs(
  expL,
  obsL,
  time_obsL,
  expVar = NULL,
  obsVar = NULL,
  criteria = "Large_dist",
  lonVar = NULL,
  latVar = NULL,
  region = NULL,
  nAnalogs = NULL,
  return_list = FALSE
)
}
\arguments{
\item{expL}{an array of N named dimensions containing the experimental field
on the large scale for which the analog is aimed. This field is used to in 
all the criterias. If parameter 'expVar' is not provided, the function will
return the expL analog. The element 'data' in the 's2dv_cube' object must 
have, at least, latitudinal and longitudinal dimensions. The object is expect
to be already subset for the desired large scale region.}

\item{obsL}{an array of N named dimensions containing the observational field 
on the large scale. The element 'data' in the 's2dv_cube' object must have 
the same latitudinal and longitudinal dimensions as parameter 'expL' and a
temporal dimension with the maximum number of available observations.}

\item{time_obsL}{a character string indicating the date of the observations 
in the format "dd/mm/yyyy"}

\item{expVar}{an array of N named dimensions containing the experimental 
field on the local scale, usually a different variable to the parameter 
'expL'. If it is not NULL (by default, NULL), the returned field by this 
function will be the analog of parameter 'expVar'.}

\item{obsVar}{an array of N named dimensions containing the field of the 
same variable as the passed in parameter 'expVar' for the same region.}

\item{criteria}{a character string indicating the criteria to be used for the
selection of analogs:
\itemize{
\item{Large_dist} minimum Euclidean distance in the large scale pattern; 
\item{Local_dist} minimum Euclidean distance in the large scale pattern 
and minimum Euclidean distance in the local scale pattern; and
\item{Local_cor} minimum Euclidean distance in the large scale pattern, 
minimum Euclidean distance in the local scale pattern and highest correlation
in the local variable to downscale.}}

\item{lonVar}{a vector containing the longitude of parameter 'expVar'.}

\item{latVar}{a vector containing the latitude of parameter 'expVar'.}

\item{region}{a vector of length four indicating the minimum longitude, 
the maximum longitude, the minimum latitude and the maximum latitude.}

\item{nAnalogs}{number of Analogs to be selected to apply the criterias 
'Local_dist' or 'Local_cor'. This is not the necessary the number of analogs 
that the user can get, but the number of events with minimum distance in 
which perform the search of the best Analog. The default value for the 
'Large_dist' criteria is 1, for 'Local_dist' and 'Local_cor'criterias must
be selected by the user otherwise the default value will be set as 10.}

\item{return_list}{TRUE to get a list with the best analogs. FALSE
to get a single analog, the best analog. For Downscaling return_list must be
FALSE.}
}
\value{
AnalogsFields, dowscaled values of the best analogs for the criteria 
selected.

AnalogsInfo, a dataframe with information about the number of the 
best analogs, the corresponding value of the metric used in the selected 
criteria (distance values for Large_dist and Local_dist,correlation values 
for Local_cor), date of the analog). The analogs are listed in decreasing 
order, the first one is the best analog (i.e if the selected criteria 
is Local_cor the best analog will be the one with highest correlation, while
for Large_dist criteria the best analog will be the day with minimum 
Euclidean distance)
}
\description{
This function perform a downscaling using Analogs. To compute 
the analogs, the function search for days with similar large scale conditions
to downscaled fields in the local scale. The large scale and the local scale 
regions are defined by the user. The large scale is usually given by 
atmospheric circulation as sea level pressure or geopotential height (Yiou 
et al, 2013) but the function gives the possibility to use another field. The
local scale will be usually given by precipitation or temperature fields, but
might be another variable. 
The analogs function will find the best analogs based in three criterias:
(1) Minimum Euclidean distance in the large scale pattern (i.e. SLP)
(2) Minimum Euclidean distance in the large scale pattern (i.e. SLP) 
and minimum Euclidean distance in the local scale pattern (i.e. SLP). 
(3) Minimum Euclidean distance in the large scale pattern (i.e. SLP), minimum 
distance in the local scale pattern (i.e. SLP) and highest correlation in the
local variable to downscale (i.e Precipitation).
The search of analogs must be done in the longest dataset posible. This is 
important since it is necessary to have a good representation of the 
possible states of the field in the past, and therefore, to get better 
analogs. Once the search of the analogs is complete, and in order to used the 
three criterias the user can select a number of analogs , using parameter 
'nAnalogs' to restrict 
the selection of the best analogs in a short number of posibilities, the best
ones. This function has not constrains of specific regions, variables to 
downscale, or data to be used (seasonal forecast data, climate projections 
data, reanalyses data). The regrid into a finner scale is done interpolating 
with CST_Load. Then, this interpolation is corrected selecting the analogs in
the large and local scale in based of the observations. The function is an 
adapted version of the method of Yiou et al 2013.
}
\examples{
require(zeallot)

# Example 1:Downscaling using criteria 'Large_dist' and a single variable:
# The best analog is found using a single variable (i.e. Sea level pressure, 
# SLP). The downscaling will be done in the same variable used to study the 
# minimal distance (i.e. SLP). obsVar and expVar NULLS or equal to obsL and 
# expL respectively region, lonVar and latVar not necessary here. 
# return_list=FALSE 

expSLP <- rnorm(1:20)
dim(expSLP) <- c(lat = 4, lon = 5)
obsSLP <- c(rnorm(1:180),expSLP*1.2)
dim(obsSLP) <- c(lat = 4, lon = 5, time = 10)
time_obsSLP <- paste(rep("01", 10), rep("01", 10), 1994 : 2003, sep = "-")
downscale_field <- Analogs(expL=expSLP, obsL=obsSLP, time_obsL=time_obsSLP)
str(downscale_field)

# Example 2: Downscaling using criteria 'Large_dist' and 2 variables:
# The best analog is found using 2 variables (i.e. Sea Level Pressure, SLP 
# and precipitation, pr): one variable (i.e. sea level pressure, expL) to 
# find the best analog day (defined in criteria 'Large_dist' as the day, in 
# obsL, with the minimum Euclidean distance to the day of interest in expL) 
# The second variable will be the variable to donwscale (i.e. precipitation, 
# obsVar). Parameter obsVar must be different to obsL.The downscaled 
# precipitation will be the precipitation that belongs to the best analog day
# in SLP. Region not needed here since will be the same for both variables.

expSLP <- rnorm(1:20)
dim(expSLP) <- c(lat = 4, lon = 5)
obsSLP <- c(rnorm(1:180),expSLP*2)
dim(obsSLP) <- c(lat = 4, lon = 5, time = 10)
time_obsSLP <- paste(rep("01", 10), rep("01", 10), 1994 : 2003, sep = "-")
obs.pr <- c(rnorm(1:200)*0.001)
dim(obs.pr)=dim(obsSLP)
downscale_field <- Analogs(expL=expSLP, obsL=obsSLP, 
                          obsVar=obs.pr,
                          time_obsL=time_obsSLP)
str(downscale_field)

# Example 3:List of best Analogs using criteria 'Large_dist' and a single 
# variable: same as Example 1 but getting a list of best analogs (return_list
# =TRUE) with the corresponding downscaled values, instead of only 1 single 
# donwscaled value instead of 1 single downscaled value. Imposing nAnalogs 
# (number of analogs to do the search of the best Analogs). obsVar and expVar
# NULL or equal to obsL and expL,respectively.

expSLP <- rnorm(1:20)
dim(expSLP) <- c(lat = 4, lon = 5)
obsSLP <- c(rnorm(1:1980),expSLP*1.5)
dim(obsSLP) <- c(lat = 4, lon = 5, time = 100)
time_obsSLP <- paste(rep("01", 100), rep("01", 100), 1920 : 2019, sep = "-")
downscale_field<- Analogs(expL=expSLP, obsL=obsSLP, time_obsSLP,
                         nAnalogs=5,return_list = TRUE)
str(downscale_field)

# Example 4:List of best Analogs using criteria 'Large_dist' and 2 variables:
# same as example 2 but getting a list of best analogs (return_list =TRUE) 
# with the values instead of only a single downscaled value. Imposing 
# nAnalogs (number of analogs to do the search of the best Analogs). obsVar 
# and expVar must be different to obsL and expL.

expSLP <- rnorm(1:20)
dim(expSLP) <- c(lat = 4, lon = 5)
obsSLP <- c(rnorm(1:180),expSLP*2)
dim(obsSLP) <- c(lat = 4, lon = 5, time = 10)
time_obsSLP <- paste(rep("01", 10), rep("01", 10), 1994 : 2003, sep = "-")
obs.pr <- c(rnorm(1:200)*0.001)
dim(obs.pr)=dim(obsSLP)
downscale_field <- Analogs(expL=expSLP, obsL=obsSLP, 
                          obsVar=obs.pr,
                          time_obsL=time_obsSLP,nAnalogs=5,
                          return_list = TRUE)
str(downscale_field)

# Example 5: Downscaling using criteria 'Local_dist' and 2 variables:
# The best analog is found using 2 variables (i.e. Sea Level Pressure, 
# SLP and precipitation, pr). Parameter obsVar must be different to obsL.The 
# downscaled precipitation will be the precipitation that belongs to the best 
# analog day in SLP. lonVar, latVar and Region must be given here to select 
# the local scale

expSLP <- rnorm(1:20)
dim(expSLP) <- c(lat = 4, lon = 5)
obsSLP <- c(rnorm(1:180),expSLP*2)
dim(obsSLP) <- c(lat = 4, lon = 5, time = 10)
time_obsSLP <- paste(rep("01", 10), rep("01", 10), 1994 : 2003, sep = "-")
obs.pr <- c(rnorm(1:200)*0.001)
dim(obs.pr)=dim(obsSLP)
# analogs of local scale using criteria 2
lonmin=-1
lonmax=2
latmin=30
latmax=33
region=c(lonmin,lonmax,latmin,latmax)
Local_scale <- Analogs(expL=expSLP,
                      obsL=obsSLP, time_obsL=time_obsSLP,
                      obsVar=obs.pr,
                      criteria="Local_dist",lonVar=seq(-1,5,1.5),
                      latVar=seq(30,35,1.5),region=region, 
                      nAnalogs = 10, return_list = FALSE)
str(Local_scale)

# Example 6: list of best analogs using criteria 'Local_dist' and 2 
# variables:
# The best analog is found using 2 variables. Parameter obsVar must be 
# different to obsL in this case.The downscaled precipitation will be the 
# precipitation that belongs to the best analog day in SLP. lonVar, latVar 
# and Region needed. return_list=TRUE

expSLP <- rnorm(1:20)
dim(expSLP) <- c(lat = 4, lon = 5)
obsSLP <- c(rnorm(1:180),expSLP*2)
dim(obsSLP) <- c(lat = 4, lon = 5, time = 10)
time_obsSLP <- paste(rep("01", 10), rep("01", 10), 1994 : 2003, sep = "-")
obs.pr <- c(rnorm(1:200)*0.001)
dim(obs.pr)=dim(obsSLP)
# analogs of local scale using criteria 2
lonmin=-1
lonmax=2
latmin=30
latmax=33
region=c(lonmin,lonmax,latmin,latmax)
Local_scale <- Analogs(expL=expSLP,
                      obsL=obsSLP, time_obsL=time_obsSLP,
                      obsVar=obs.pr,
                      criteria="Local_dist",lonVar=seq(-1,5,1.5),
                      latVar=seq(30,35,1.5),region=region, 
                      nAnalogs = 5, return_list = TRUE)
str(Local_scale)

# Example 7: Downscaling using Local_dist criteria
# without parameters obsVar and expVar. return list =FALSE 

expSLP <- rnorm(1:20)
dim(expSLP) <- c(lat = 4, lon = 5)
obsSLP <- c(rnorm(1:180),expSLP*2)
dim(obsSLP) <- c(lat = 4, lon = 5, time = 10)
time_obsSLP <- paste(rep("01", 10), rep("01", 10), 1994 : 2003, sep = "-")
# analogs of local scale using criteria 2
lonmin=-1
lonmax=2
latmin=30
latmax=33
region=c(lonmin,lonmax,latmin,latmax)
Local_scale <- Analogs(expL=expSLP,
                      obsL=obsSLP, time_obsL=time_obsSLP,
                      criteria="Local_dist",lonVar=seq(-1,5,1.5),
                      latVar=seq(30,35,1.5),region=region, 
                      nAnalogs = 10, return_list = TRUE)
str(Local_scale)

# Example 8: Downscaling using criteria 'Local_cor' and 2 variables:
# The best analog is found using 2 variables. Parameter obsVar and expVar 
# are necessary and must be different to obsL and expL, respectively.
# The downscaled precipitation will be the precipitation that belongs to
# the best analog day in SLP large and local scales, and to the day with 
# the higher correlation in precipitation. return_list=FALSE. two options 
# for nAnalogs

expSLP <- rnorm(1:20)
dim(expSLP) <- c(lat = 4, lon = 5)
obsSLP <- c(rnorm(1:180),expSLP*2)
dim(obsSLP) <- c(lat = 4, lon = 5, time = 10)
time_obsSLP <- paste(rep("01", 10), rep("01", 10), 1994 : 2003, sep = "-")
exp.pr <- c(rnorm(1:20)*0.001)
dim(exp.pr)=dim(expSLP)
obs.pr <- c(rnorm(1:200)*0.001)
dim(obs.pr)=dim(obsSLP)
# analogs of local scale using criteria 2
lonmin=-1
lonmax=2
latmin=30
latmax=33
region=c(lonmin,lonmax,latmin,latmax)
Local_scalecor <- Analogs(expL=expSLP,
                         obsL=obsSLP, time_obsL=time_obsSLP,
                         obsVar=obs.pr,expVar=exp.pr,
                         criteria="Local_cor",lonVar=seq(-1,5,1.5),
                         latVar=seq(30,35,1.5),nAnalogs=8,region=region, 
                         return_list = FALSE)
Local_scalecor$AnalogsInfo
Local_scalecor$DatesAnalogs
# same but without imposing nAnalogs, so nAnalogs will be set by default as 10 
Local_scalecor <- Analogs(expL=expSLP,
                         obsL=obsSLP, time_obsL=time_obsSLP,
                         obsVar=obs.pr,expVar=exp.pr,
                         criteria="Local_cor",lonVar=seq(-1,5,1.5),
                         latVar=seq(30,35,1.5),region=region, 
                         return_list = FALSE)
Local_scalecor$AnalogsInfo
Local_scalecor$DatesAnalogs

# Example 9: List of best analogs in the three criterias Large_dist, 
# Local_dist, and Local_cor return list TRUE same variable

expSLP <- rnorm(1:20)
dim(expSLP) <- c(lat = 4, lon = 5)
obsSLP <- c(rnorm(1:180),expSLP*2)
dim(obsSLP) <- c(lat = 4, lon = 5, time = 10)
time_obsSLP <- paste(rep("01", 10), rep("01", 10), 1994 : 2003, sep = "-")
# analogs of large scale using criteria 1
Large_scale <- Analogs(expL=expSLP,
                      obsL=obsSLP, time_obsL=time_obsSLP,
                      criteria="Large_dist", 
                      nAnalogs = 7, return_list = TRUE)
str(Large_scale)
Large_scale$AnalogsInfo
# analogs of local scale using criteria 2
lonmin=-1
lonmax=2
latmin=30
latmax=33
region=c(lonmin,lonmax,latmin,latmax)
Local_scale <- Analogs(expL=expSLP,
                      obsL=obsSLP, time_obsL=time_obsSLP,
                      criteria="Local_dist",lonVar=seq(-1,5,1.5),
                      latVar=seq(30,35,1.5),nAnalogs=7,region=region, 
                      return_list = TRUE)
str(Local_scale)
Local_scale$AnalogsInfo
# analogs of local scale using criteria 3
Local_scalecor <- Analogs(expL=expSLP,
                         obsL=obsSLP, time_obsL=time_obsSLP,
                         obsVar=obsSLP,expVar=expSLP,
                         criteria="Local_cor",lonVar=seq(-1,5,1.5),
                         latVar=seq(30,35,1.5),nAnalogs=7,region=region, 
                         return_list = TRUE)
str(Local_scalecor)
Local_scalecor$AnalogsInfo

# Example 10: Downscaling in the three criteria Large_dist, Local_dist, and 
# Local_cor return list FALSE, different variable

expSLP <- rnorm(1:20)
dim(expSLP) <- c(lat = 4, lon = 5)
obsSLP <- c(rnorm(1:180),expSLP*2)
dim(obsSLP) <- c(lat = 4, lon = 5, time = 10)
time_obsSLP <- paste(rep("01", 10), rep("01", 10), 1994 : 2003, sep = "-")
exp.pr <- c(rnorm(1:20)*0.001)
dim(exp.pr)=dim(expSLP)
obs.pr <- c(rnorm(1:200)*0.001)
dim(obs.pr)=dim(obsSLP)
# analogs of large scale using criteria 1
Large_scale <- Analogs(expL=expSLP,
                      obsL=obsSLP, time_obsL=time_obsSLP,
                      criteria="Large_dist", 
                      nAnalogs = 7, return_list = FALSE)
str(Large_scale)
Large_scale$AnalogsInfo
# analogs of local scale using criteria 2
lonmin=-1
lonmax=2
latmin=30
latmax=33
region=c(lonmin,lonmax,latmin,latmax)
Local_scale <- Analogs(expL=expSLP,
                      obsL=obsSLP, time_obsL=time_obsSLP,
                      obsVar=obs.pr,
                      criteria="Local_dist",lonVar=seq(-1,5,1.5),
                      latVar=seq(30,35,1.5),nAnalogs=7,region=region, 
                      return_list = FALSE)
str(Local_scale)
Local_scale$AnalogsInfo
# analogs of local scale using criteria 3
Local_scalecor <- Analogs(expL=expSLP,
                         obsL=obsSLP, time_obsL=time_obsSLP,
                         obsVar=obs.pr,expVar=exp.pr,
                         criteria="Local_cor",lonVar=seq(-1,5,1.5),
                         latVar=seq(30,35,1.5),nAnalogs=7,region=region, 
                         return_list = FALSE)
str(Local_scalecor)
Local_scalecor$AnalogsInfo

}
\references{
Yiou, P., T. Salameh, P. Drobinski, L. Menut, R. Vautard,
and M. Vrac, 2013 : Ensemble reconstruction of the atmospheric column 
from surface pressure using analogues.  Clim. Dyn., 41, 1419-1437. 
\email{pascal.yiou@lsce.ipsl.fr}
}
\author{
M. Carmen Alvarez-Castro, \email{carmen.alvarez-castro@cmcc.it}

Nuria Perez-Zanon \email{nuria.perez@bsc.es}
}
