% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/modelKriging.R
\name{modelKriging}
\alias{modelKriging}
\title{Kriging Model}
\usage{
modelKriging(x, y, distanceFunction, control = list())
}
\arguments{
\item{x}{list of samples in input space}

\item{y}{column vector of observations for each sample}

\item{distanceFunction}{a suitable distance function of type f(x1,x2), returning a scalar distance value, preferably between 0 and 1.
  Maximum distances larger 1 are no problem, but may yield scaling bias when different measures are compared.
     Should be non-negative and symmetric.  It can also be a list of several distance functions. In this case, Maximum Likelihood Estimation (MLE) is used 
 to determine the most suited distance measure.
 The distance function may have additional parameters. For that case, see distanceParametersLower/Upper in the controls.
If distanceFunction is missing, it can also be provided in the control list.}

\item{control}{(list), with the options for the model building procedure:
\describe{
\item{\code{lower}}{ lower boundary for theta, default is \code{1e-6}}
\item{\code{upper}}{ upper boundary for theta, default is \code{100}}
\item{\code{corr}}{ function to be used for correlation modelling, default is \code{fcorrGauss}}
\item{\code{algTheta}}{ algorithm used to find theta (as well as p and lambda), default is \code{\link{optimInterface}}.}
\item{\code{algThetaControl}}{ list of controls passed to \code{algTheta}.}
\item{\code{useLambda}}{ whether or not to use the regularization constant lambda (nugget effect). Default is \code{FALSE}.}
\item{\code{lambdaLower}}{ lower boundary for lambda (log scale), default is \code{-6}}
\item{\code{lambdaUpper}}{ upper boundary for lambda (log scale), default is \code{0}}
\item{\code{distanceParametersLower}}{ lower boundary for parameters of the distance function, default is \code{NA} which means there are no distance function parameters. If several distance functions are supplied, this should be a list of lower boundary vectors for each function.}
\item{\code{distanceParametersUpper}}{ upper boundary for parameters of the distance function, default is \code{NA} which means there are no distance function parameters. If several distance functions are supplied, this should be a list of upper boundary vectors for each function.}
\item{\code{distances}}{ a distance matrix. If available, this matrix is used for model building, instead of calculating the distance matrix using the parameters \code{distanceFunction}. Default is \code{NULL}.}
\item{\code{scaling}}{ If TRUE: Distances values are divided by maximum distance to avoid scale bias.}
\item{\code{reinterpolate}}{ If TRUE: reinterpolation is used to generate better uncertainty estimates in the presence of noise. }
\item{\code{combineDistances}}{ By default, several distance functions or matrices are subject to a likelihood based decision, choosing one. If this parameter is TRUE, they are instead combined by determining a weighted sum. The weighting parameters are determined by MLE.}
\item{\code{userParameters}}{ By default: (\code{NULL}). Else, this vector is used instead of MLE to specify the model parameters, in the following order: kernel parameters, distance weights, lambda, distance parameters.}
\item{\code{indefiniteMethod}}{ The specific method used for correction: spectrum \code{"clip"}, spectrum \code{"flip"}, spectrum \code{"square"}, spectrum \code{"diffusion"}, feature embedding "feature", nearest definite matrix "near". Default is no correction: \code{"none"}. See Zaefferer and Bartz-Beielstein (2016).}
\item{\code{indefiniteType}}{  The general type of correction for indefiniteness: \code{"NSD"},\code{"CNSD"} or the default \code{"PSD"}. See Zaefferer and Bartz-Beielstein (2016). Note, that feature embedding may not work in case of multiple distance functions.}
\item{\code{indefiniteRepair}}{ boolean, whether conditions of the distance matrix (in case of \code{"NSD"},\code{"CNSD"} correction type) or correlation matrix (in case of \code{"PSD"} correction type) are repaired.}
}}
}
\value{
an object of class \code{modelKriging} containing the options (see control parameter) and determined parameters for the model:
\describe{
\item{\code{theta}}{ parameters of the kernel / correlation function determined with MLE.}
\item{\code{lambda}}{ regularization constant (nugget) lambda}
\item{\code{yMu}}{ vector of observations y, minus MLE of mu}
\item{\code{SSQ}}{ Maximum Likelihood Estimate (MLE) of model parameter sigma^2}
\item{\code{mu}}{ MLE of model parameter mu}
\item{\code{Psi}}{ correlation matrix Psi}
\item{\code{Psinv}}{ inverse of Psi}
\item{\code{nevals}}{ number of Likelihood evaluations during MLE of theta/lambda/p}
\item{\code{distanceFunctionIndexMLE}}{ If a list of several distance measures (\code{distanceFunction}) was given, this parameter contains the index value of the measure chosen with MLE.}
}
}
\description{
Implementation of a distance-based Kriging model, e.g., for mixed or combinatorial input spaces.
It is based on employing suitable distance measures for the samples in input space.
}
\details{
The basic Kriging implementation is based on the work of Forrester et al. (2008). 
For adaptation of Kriging to mixed or combinatorial spaces, as well as
choosing distance measures with Maximum Likelihood Estimation, see the other two references (Zaefferer et al., 2014).
}
\examples{
# Set random number generator seed
set.seed(1)
# Simple test landscape
fn <- landscapeGeneratorUNI(1:5,distancePermutationHamming)
# Generate data for training and test
x <- unique(replicate(40,sample(5),FALSE))
xtest <- x[-(1:15)]
x <- x[1:15]
# Determin true objective function values
y <- fn(x)
ytest <- fn(xtest)
# Build model
fit <- modelKriging(x,y,distancePermutationHamming,
    control=list(algThetaControl=list(method="L-BFGS-B"),useLambda=FALSE))
# Predicted obj. function values
ypred <- predict(fit,xtest)$y
# Uncertainty estimate
fit$predAll <- TRUE
spred <- predict(fit,xtest)$s
# Plot
plot(ytest,ypred,xlab="true value",ylab="predicted value",
    pch=20,xlim=c(0.3,1),ylim=c(min(ypred)-0.1,max(ypred)+0.1))
segments(ytest, ypred-spred,ytest, ypred+spred)
epsilon = 0.02
segments(ytest-epsilon,ypred-spred,ytest+epsilon,ypred-spred)
segments(ytest-epsilon,ypred+spred,ytest+epsilon,ypred+spred)
abline(0,1,lty=2)
# Use a different/custom optimizer (here: SANN) for maximum likelihood estimation: 
# (Note: Bound constraints are recommended, to avoid Inf values.
# This is really just a demonstration. SANN does not respect bound constraints.)
optimizer1 <- function(x,fun,lower=NULL,upper=NULL,control=NULL,...){
  res <- optim(x,fun,method="SANN",control=list(maxit=100),...)
  list(xbest=res$par,ybest=res$value,count=res$counts)
}
fit <- modelKriging(x,y,distancePermutationHamming,
                   control=list(algTheta=optimizer1,useLambda=FALSE))
#One-dimensional optimizer (Brent). Note, that Brent will not work when 
#several parameters have to be set, e.g., when using nugget effect (lambda).
#However, Brent may be quite efficient otherwise.
optimizer2 <- function(x,fun,lower,upper,control=NULL,...){
 res <- optim(x,fun,method="Brent",lower=lower,upper=upper,...)
 list(xbest=res$par,ybest=res$value,count=res$counts)
}
fit <- modelKriging(x,y,distancePermutationHamming,
                    control=list(algTheta=optimizer2,useLambda=FALSE))
}
\references{
Forrester, Alexander I.J.; Sobester, Andras; Keane, Andy J. (2008). Engineering Design via Surrogate Modelling - A Practical Guide. John Wiley & Sons.

Zaefferer, Martin; Stork, Joerg; Friese, Martina; Fischbach, Andreas; Naujoks, Boris; Bartz-Beielstein, Thomas. (2014). Efficient global optimization for combinatorial problems. In Proceedings of the 2014 conference on Genetic and evolutionary computation (GECCO '14). ACM, New York, NY, USA, 871-878. DOI=10.1145/2576768.2598282 http://doi.acm.org/10.1145/2576768.2598282

Zaefferer, Martin; Stork, Joerg; Bartz-Beielstein, Thomas. (2014). Distance Measures for Permutations in Combinatorial Efficient Global Optimization. In Parallel Problem Solving from Nature - PPSN XIII (p. 373-383). Springer International Publishing.

Zaefferer, Martin and Bartz-Beielstein, Thomas (2016). Efficient Global Optimization with Indefinite Kernels. Parallel Problem Solving from Nature-PPSN XIV. Accepted, in press. Springer.
}
\seealso{
\code{\link{predict.modelKriging}}
}
