% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/Similarities.R
\name{treesimilarities}
\alias{treesimilarities}
\title{Calculating similarities between classification trees}
\usage{
treesimilarities(forest, m, tol = NULL, weight = NULL)
}
\arguments{
\item{forest}{Object of forest class, containing the trees between which similarities should be calculated.}

\item{m}{Similarity measure that should be used to calculate similarities.
Here, m=1 is based on counting equal predictors or predictor-split point combinations (Equation 3 or 6  in the Supplmentary Materials of Sies & Van Mechelen (2020));
m=2 is the measure of Shannon & Banks (1999), based on counting the number of equal paths from rootnode to leafs (See also Sies & Van Mechelen (2020), Equation 2);
m=3 is based on the agreement in predicted classification labels (Chipman, 1998, see also Sies & Van Mechelen (2020), Equation 7);
m=4 is based on the agreement of partitions (Chipman, 1998; see also Sies & Van Mechelen (2020), Equation 5);
m=5 is based on counting equal elementary conjunctions of trees transformed to disjunctive normal form ((limited to binary predictors), see Sies & Van Mechelen (2020), Supplementary materials Equation 15);
m=6 is based on comparing sets of predictors or predictor split point combinations (taking into account directions of the splits) associated with a leaf, while also taking into account the classification label of that leaf
(see Sies & Van Mechelen (2020), Supplementary Materials Equation 13 and 14).}

\item{tol}{A vector with for each predictor a number that defines the tolerance zone within which two split points of the predictor in question are assumed equal. For example, if the tolerance for predictor X
is 1, then a split on that predictor in tree A will be assumed equal to a split in tree B as long as the splitpoint in tree B is within the splitpoint in tree A + or - 1. Only applicable for m=1 and m=6. Default=NULL}

\item{weight}{If 1, the number of dissimilar paths in the Shannon and Banks measure (m=2), should be weighted by 1/their length (Otherwise they are weighted equally). Only applicable for m=2. Default=NULL}
}
\value{
The function returns an object of class treesimilarities. It includes two attributes:
\item{forest}{The forest for which similarities were calculated.}\item{sim}{Similarity matrix with similarities between all trees in the forest based on the chosen similarity measure}
}
\description{
A function to calculate similarities between classification trees in a forest, based on 6 different  possible similarity measures (described in Sies & Van Mechelen, 2020).
Returns an object of class treesimilarities.
}
\details{
This function calculates similarities between trees in a forest object (use the forest function to coerce a list of trees, a list of dataframes on which the trees were estimated,
and the full data set underlying the forest, to a forest object).
The six similarity measures that are implemented, are described in Sies & Van Mechelen (2020). Each measure takes into account different aspects of similarity between trees.

The function returns a treesimilarities object, which consists of two attributes: The forest object containing the trees between which the similarities were calculated, and the similarity matrix.
There are a couple of methods that can be used on treesimilarity objects, such as print.treesimilarities(), plot.treesimilarities(), summary.treesimilarities().
With the coercion method as.treesimilarites(), one can coerce a forest object and a similarity matrix to a treesimilarities object.
}
\examples{
require(rpart)
require(MASS)
#Function to draw a bootstrap sample from a dataset
DrawBoots <- function(dataset, i){
set.seed(2394 + i)
Boot <- dataset[sample(1:nrow(dataset), size = nrow(dataset), replace = TRUE),]
return(Boot)
}

#Function to grow a tree using rpart on a dataset
GrowTree <- function(x,y,BootsSample, minsplit = 40, minbucket = 20, maxdepth =3){
 controlrpart <- rpart.control(minsplit = minsplit, minbucket = minbucket,
 maxdepth = maxdepth, maxsurrogate = 0, maxcompete = 0)
 tree <- rpart(as.formula(paste(noquote(paste(y, "~")), noquote(paste(x, collapse="+")))),
 data = BootsSample, control = controlrpart)
 return(tree)
}

#Use functions to draw 20 boostrapsamples and grow a tree on each sample
Boots<- lapply(1:20, function(k) DrawBoots(Pima.tr ,k))
Trees <- lapply(1:20, function (i) GrowTree(x=c("npreg", "glu",  "bp",  "skin",
 "bmi", "ped", "age"), y="type", Boots[[i]] ))

#Turn the lists of dataframes and rpart trees to a forest object
myforest<- forest(Pima.tr,Boots,Trees)

#Calculate similarities between trees in forest, based on similarity measure m=1.
Simmatrix1 <- treesimilarities(forest=myforest,  m=1)

#Calculate similarities between trees in forest, based on similarity measure m=6,
#taking splitpoints into account.
Simmatrix6<- treesimilarities(forest=myforest,  m=6, tol=rep(1,7))
}
\references{
\cite{Shannon, W. D., & Banks, D. (1999). Combining classification trees using MLE. Statistics in medicine, 18(6), 727-740.}

\cite{Chipman, H. A., George, E. I., & McCulloh, R. E. (1998). Making sense of a forest of trees. Computing Science and Statistics, 84-92.}

\cite{Sies, A. & Van Mechelen I. (2020). C443: An R-package to see a forest for the trees. Journal of Classification.}
}
