# This Function is designed to produce a summary table of data sources.
# It was written by James B Dorey from the 20th of January 2023.

#' Build a table of data providers for bee occurrence records
#' 
#' This function will attempt to find and build a table of data providers that have contributed
#' to the input data, especially using the 'institutionCode' column. It will also look for a 
#' variety of other columns to find data providers using an internally set sequence of if-else
#' statements. Hence, this function is quite specific for bee data, but should work for other 
#' taxa in similar institutions.
#'
#'
#' @param data A data frame or tibble. Occurrence records as input.
#' @param runBeeDataChecks Logical. If TRUE, will search in other columns for specific clues to
#' determine the institution.
#' @param outPath A character path. The path to the directory in which the figure will be saved.
#' Default = OutPath_Report.
#' @param fileName Character. The name of the file to be saved, ending in ".csv".
#'
#'
#' @return Returns a table with the data providers, an specimen count, and a species count.
#' @export
#'
#' @importFrom dplyr %>%
#' 
#' @examples
#' 
# import data
#' data(beesFlagged)
#' 
#' testOut <- dataProvTables(
#' data = beesFlagged,
#' runBeeDataChecks = TRUE,
#' outPath = tempdir(),
#' fileName = "testFile.csv")
#' 
dataProvTables <- function(
    data = NULL,
    runBeeDataChecks = FALSE,
    outPath = OutPath_Report,
    fileName = NULL
){
  requireNamespace("dplyr")
  
  # locally bind variables to the function
  OutPath_Report <- occurrenceCount <- NULL
  
  #### 0.0 Warnings ####
  if(is.null(data)){
    stop("You must provide an input dataset.")
  }
  if(is.null(data)){
    stop("You must provide a fileName.")
  }
  # locally bind variables to the function
  dataSource<-institutionCode<-recordNumber<-institutionCodeNew<-recordedBy<-occurrenceID<-
    catalogNumber<-bibliographicCitation<-datasetName<-otherCatalogNumbers<-rightsHolder<-
    references<-eventID<-datasetID<-database_id<-id<-scientificName<-DataPath <- NULL
  
  #### 1.0 Data prep ####
  if(runBeeDataChecks == TRUE){
      ##### 1.1 Find codes ####
    # Find institutionCodes using other information
    temp <- data %>%
        # Select ALA as data source and only those missing institutionCode
      dplyr::filter(stringr::str_detect(dataSource, "ALA|GBIF|iDigBio") & 
                      is.na(institutionCode)) %>%
        # Find clues to institutionCode and add it
      dplyr::mutate(
        # WAM
        institutionCodeNew = ifelse(
          stringr::str_detect(recordNumber, "^WAM "), "WAM", NA),
        # HYM
        institutionCodeNew = dplyr::if_else(stringr::str_detect(recordNumber, "^HYM ") & 
                                              is.na(institutionCodeNew), "NMV", institutionCodeNew),
        institutionCodeNew = dplyr::if_else(stringr::str_detect(recordedBy, "Assorted Museum of Victoria") & 
                                              is.na(institutionCodeNew), "NMV", institutionCodeNew),
          # bowerbird
        institutionCodeNew = dplyr::if_else(stringr::str_detect(recordNumber, "bowerbird") & 
                                              is.na(institutionCodeNew), "bowerbird", institutionCodeNew),
        # PaDIL
        institutionCodeNew = dplyr::if_else(stringr::str_detect(tolower(recordNumber), "^bee|^rlj ") & 
                                              is.na(institutionCodeNew), "PaDIL", institutionCodeNew),
        # Flickr
        institutionCodeNew = dplyr::if_else( stringr::str_detect(tolower(occurrenceID), "flickr") & 
                                              is.na(institutionCodeNew), "Flickr", institutionCodeNew),
        # questagame
        institutionCodeNew = dplyr::if_else( stringr::str_detect(tolower(recordedBy), "questagame") & 
                                               is.na(institutionCodeNew), "QuestaGame", institutionCodeNew),
        # WINC
        institutionCodeNew = dplyr::if_else( stringr::str_detect(catalogNumber, "WINC") & 
                                               is.na(institutionCodeNew), "WINC", institutionCodeNew),
        # biocollectALA
        institutionCodeNew = dplyr::if_else( stringr::str_detect(recordNumber, "biocollect") & 
                                               is.na(institutionCodeNew), "Wildlife Watch NSC", institutionCodeNew),
        # Aus Museum
        institutionCodeNew = dplyr::if_else( stringr::str_detect(recordNumber, "^K   |AM  ") & 
                                               is.na(institutionCodeNew), "AM", institutionCodeNew),
        # WildNet
        institutionCodeNew = dplyr::if_else( stringr::str_detect(occurrenceID, "WildNet") & 
                                               is.na(institutionCodeNew), "WildNet", institutionCodeNew),
        ## OTHER
        institutionCodeNew = dplyr::if_else( stringr::str_detect(catalogNumber, "^RSKM_ENT") & 
                                               is.na(institutionCodeNew), "RSKM", institutionCodeNew),
        institutionCodeNew = dplyr::if_else( stringr::str_detect(occurrenceID, "^UVM") & 
                                               is.na(institutionCodeNew), "UVM", institutionCodeNew),
        institutionCodeNew = dplyr::if_else( stringr::str_detect(catalogNumber, "^BIOUG") & 
                                               is.na(institutionCodeNew), "BIOUG", institutionCodeNew),
        institutionCodeNew = dplyr::if_else( stringr::str_detect(catalogNumber, "^ZMUO") & 
                                               is.na(institutionCodeNew), "ZMUO", institutionCodeNew),
        institutionCodeNew = dplyr::if_else( stringr::str_detect(occurrenceID, "data\\.biodiversitydata\\.nl") & 
                                               is.na(institutionCodeNew), "Naturalis Biodiversity Center", institutionCodeNew),
        institutionCodeNew = dplyr::if_else( stringr::str_detect(bibliographicCitation, "FinBIF") & 
                                               is.na(institutionCodeNew), "FinBIF", institutionCodeNew),
        institutionCodeNew = dplyr::if_else( stringr::str_detect(datasetName, "Swiss National Apoidea Databank") & 
                                               is.na(institutionCodeNew), "CSCF", institutionCodeNew),
        institutionCodeNew = dplyr::if_else( stringr::str_detect(otherCatalogNumbers, "VTST_ENT") & 
                                               is.na(institutionCodeNew), "VTST", institutionCodeNew),
        institutionCodeNew = dplyr::if_else( stringr::str_detect(otherCatalogNumbers, "ECNRUFC") & 
                                               is.na(institutionCodeNew), "ECNRUFC", institutionCodeNew),
        institutionCodeNew = dplyr::if_else( stringr::str_detect(recordedBy, "SPIPOLL") & 
                                               is.na(institutionCodeNew), "SPIPOLL", institutionCodeNew),
        institutionCodeNew = dplyr::if_else( stringr::str_detect(datasetName, "Riiklik keskkonnaseire programm") & 
                                               is.na(institutionCodeNew), "Riiklik keskkonnaseire programm", institutionCodeNew),
        institutionCodeNew = dplyr::if_else( stringr::str_detect(rightsHolder, "Suffolk Biodiversity Information Service") & 
                                               is.na(institutionCodeNew), "Suffolk Biodiversity Information Service", institutionCodeNew),
        institutionCodeNew = dplyr::if_else( stringr::str_detect(rightsHolder, "Staffordshire Ecological Record") & 
                                               is.na(institutionCodeNew), "Staffordshire Ecological Record", institutionCodeNew),
        institutionCodeNew = dplyr::if_else( stringr::str_detect(rightsHolder, "NatureSpot") & 
                                               is.na(institutionCodeNew), "NatureSpot", institutionCodeNew),
        institutionCodeNew = dplyr::if_else( stringr::str_detect(rightsHolder, "Biological Records Centre") & 
                                               is.na(institutionCodeNew), "Biological Records Centre", institutionCodeNew),
        institutionCodeNew = dplyr::if_else( stringr::str_detect(references, "www\\.ebi\\.ac\\.uk/ena") & 
                                               is.na(institutionCodeNew), "European Nucleotide Archive", institutionCodeNew),
        institutionCodeNew = dplyr::if_else( stringr::str_detect(occurrenceID, "observation\\.org") & 
                                               is.na(institutionCodeNew), "Observation.org", institutionCodeNew),
        institutionCodeNew = dplyr::if_else( stringr::str_detect(eventID, "plutof.ut.ee") & 
                                               is.na(institutionCodeNew), "Plotuf", institutionCodeNew),
        institutionCodeNew = dplyr::if_else( stringr::str_detect(catalogNumber, "USNM ENT") & 
                                               is.na(institutionCodeNew), "USNM", institutionCodeNew),
        institutionCodeNew = dplyr::if_else( stringr::str_detect(occurrenceID, "boldsystems\\.org") & 
                                               is.na(institutionCodeNew), "BOLD", institutionCodeNew),
        institutionCodeNew = dplyr::if_else( stringr::str_detect(datasetName, "Chronicle of Nature") & 
                                               is.na(institutionCodeNew), "Chronicle of Nature", institutionCodeNew),
        institutionCodeNew = dplyr::if_else( stringr::str_detect(rightsHolder, "Isle of Wight Local Records Centre") & 
                                               is.na(institutionCodeNew), "Isle of Wight Local Records Centre", institutionCodeNew),
        institutionCodeNew = dplyr::if_else( stringr::str_detect(rightsHolder, "Gloucestershire Centre for Environmental Records") & 
                                               is.na(institutionCodeNew), "Gloucestershire Centre for Environmental Records", institutionCodeNew),
        institutionCodeNew = dplyr::if_else( stringr::str_detect(datasetName, "Bumble bees collected in a large-scale field experiment in power line clearings, southeast Norway") & 
                                               is.na(institutionCodeNew), "Norwegian University of Life Sciences (NMBU)", institutionCodeNew),
        institutionCodeNew = dplyr::if_else( stringr::str_detect(eventID, "WestAfricaBees") & 
                                               is.na(institutionCodeNew), "Station d'Ecologie de Lamto", institutionCodeNew),
        institutionCodeNew = dplyr::if_else( stringr::str_detect(eventID, "TaborW") & 
                                               is.na(institutionCodeNew), "Vermont Center for Ecostudies", institutionCodeNew),
        institutionCodeNew = dplyr::if_else( stringr::str_detect(datasetName, "Abeilles fichier Nico Schneider") & 
                                               is.na(institutionCodeNew), "National Museum of Natural History, Luxembourg", institutionCodeNew),
        institutionCodeNew = dplyr::if_else( stringr::str_detect(occurrenceID, "BSRU") & 
                                               is.na(institutionCodeNew), "Chulalongkorn University Natural History Museum (CUNHM)", institutionCodeNew),
        
institutionCodeNew = dplyr::if_else( stringr::str_detect(occurrenceID, "MFV:VT|USFWS-RCN|VCE:|USfWS-RCN|MNWR:bombus|^Frey21-") & 
                                               is.na(institutionCodeNew), "Vermont Center for Ecostudies", institutionCodeNew),
        institutionCodeNew = dplyr::if_else( stringr::str_detect(catalogNumber, "21BAM") & 
                                               is.na(institutionCodeNew), "Vermont Center for Ecostudies", institutionCodeNew),

        institutionCodeNew = dplyr::if_else( stringr::str_detect(occurrenceID, "Par\\.V\\.|Viter\\.|Vit\\.S|^UNCG|Rufford\\.UA|Parkh\\.faun|Medobory\\.data|Ins\\.Khar") & 
                                               is.na(institutionCodeNew), "Ukrainian Nature Conservation Group (UNCG)", institutionCodeNew),

        institutionCodeNew = dplyr::if_else( stringr::str_detect(occurrenceID, "Calabuig:Ringsted") & 
                                               is.na(institutionCodeNew), "Natural History Museum of Denmark", institutionCodeNew),
        institutionCodeNew = dplyr::if_else( stringr::str_detect(rightsHolder, "Cambridgeshire & Peterborough Environmental Records Centre") & 
                                               is.na(institutionCodeNew), "Cambridgeshire & Peterborough Environmental Records Centre", institutionCodeNew),
        institutionCodeNew = dplyr::if_else( stringr::str_detect(references, "natureshare\\.org\\.au") & 
                                               is.na(institutionCodeNew), "NatureShare", institutionCodeNew),
        institutionCodeNew = dplyr::if_else( stringr::str_detect(rightsHolder, "Manx Wildlife Trust") & 
                                               is.na(institutionCodeNew), "Manx Wildlife Trust", institutionCodeNew),
        institutionCodeNew = dplyr::if_else( stringr::str_detect(rightsHolder, "Ministry of Justice") & 
                                               is.na(institutionCodeNew), "Ministry of Justice, UK", institutionCodeNew),
        institutionCodeNew = dplyr::if_else( stringr::str_detect(rightsHolder, "Natural England") & 
                                               is.na(institutionCodeNew), "Natural England", institutionCodeNew),
        institutionCodeNew = dplyr::if_else( stringr::str_detect(rightsHolder, "Queensland Government") & 
                                               is.na(institutionCodeNew), "Queensland Government", institutionCodeNew),
        institutionCodeNew = dplyr::if_else( stringr::str_detect(rightsHolder, "Royal Horticultural Society") & 
                                               is.na(institutionCodeNew), "Royal Horticultural Society", institutionCodeNew),
        institutionCodeNew = dplyr::if_else( stringr::str_detect(rightsHolder, "Sheffield and Rotherham Wildlife Trust") & 
                                               is.na(institutionCodeNew), "Sheffield and Rotherham Wildlife Trust", institutionCodeNew),
        institutionCodeNew = dplyr::if_else( stringr::str_detect(rightsHolder, "Prioksko-Terrasnyi Biosphere Reserve") & 
                                               is.na(institutionCodeNew), "Prioksko-Terrasnyi Biosphere Reserve", institutionCodeNew),
        institutionCodeNew = dplyr::if_else( stringr::str_detect(occurrenceID, "mohonk:") & 
                                               is.na(institutionCodeNew), "Mohonk Preserve", institutionCodeNew),
        institutionCodeNew = dplyr::if_else( stringr::str_detect(occurrenceID, "^LUXNATFUND|^DSS00") & 
                                               is.na(institutionCodeNew), "National Museum of Natural History, Luxembourg", institutionCodeNew),
        institutionCodeNew = dplyr::if_else( stringr::str_detect(datasetName, "Collection hymenopteres MNHNL") & 
                                               is.na(institutionCodeNew), "National Museum of Natural History, Luxembourg", institutionCodeNew),
        
        institutionCodeNew = dplyr::if_else( stringr::str_detect(occurrenceID, "IIA-ENT") & 
                                               is.na(institutionCodeNew), "INSTITUTO DE INVESTIGACaO AGRONOMICA - IIA", institutionCodeNew),
        institutionCodeNew = dplyr::if_else( stringr::str_detect(occurrenceID, "indiabiodiversity\\.org") & 
                                               is.na(institutionCodeNew), "IndiaBiodiversity.org", institutionCodeNew),
        institutionCodeNew = dplyr::if_else( stringr::str_detect(catalogNumber, "^OPI") & 
                                               is.na(institutionCodeNew), "South Australia, Department for Environment and Water", institutionCodeNew),
        institutionCodeNew = dplyr::if_else( stringr::str_detect(catalogNumber, "^WFM[0-9]+") & 
                                               is.na(institutionCodeNew), "Western Australia, Department of Biodiversity, Conservation and Attractions", institutionCodeNew),
        institutionCodeNew = dplyr::if_else( stringr::str_detect(recordedBy, "LIMOUSIN,BUREAU D'ETUDE DGE|PNR PL|Agniele Touret-Alby, Quentin Rome|AGASSE-YVER Florence, MARLE Micka|AUBERT Matthieu|BESSIERE A\\., BLEOMELEN Alwin \\(PNM\\)|Bottinelli Julien|PERCHE NATURE|BRUGEROLLES Yvan|CENSE Thierry,CENSE Colette|CESARI Lily \\(Naturoptere\\)|CHOREIN Adrien \\(CEN Centre\\)|Cocquempot C\\.|DAMOISEAU Sebastien \\(CERCOPE\\)|GENOUD David|Gosselin M\\.|Grumdi|Jean-Pierre Viallard|Jean-Pierre Carreras|Jean-Loup d'Hondt|Jean-Loup d'Hondt|Jean-Louis PRATZ|Jean-Jacques Laffitte|Jean-Laurent HENTZ|Jean-Francois Campion|Jean-Sebastien Carteron|LE DIVELEC Romain|MAILLIER Sebastien|MARTHA Benoit|PLATEAUX Luc|PRATZ Jean-Louis|Sebastien LAGUET|Sebbbounet|Thierry ROBERT|\\(ECOSPHERE\\)|\\(CEN AQUITAINE\\)") & 
                                               is.na(institutionCodeNew), "OFB-CNRS-MNHN", institutionCodeNew),
        institutionCodeNew = dplyr::if_else( stringr::str_detect(datasetID, "E053-2614A8C02B7C|B2C9849D2ACF|2614A8C021C1|2614A8C0FB88|2614A8C0E6FC|2614A8C0CF63|2614A8C05B99|2614A8C00BBE|2614A8C0C021|2614A8C0FC45|2614A8C067D6|2614A8C0753D|2614A8C057EA|5014A8C02001|5014A8C04A0C|2614A8C00722|-E053-|-e053-|F0DFF9845389|8EA38E099656|26033513335E|C9074EB78761|9AC2D5DAB4DF|DF2C4D61871E|E06C2F2DB641|A83B5C8B393F|3D6B50E67F30|6DB84CF2329A|EEE933FAF77E|9776F5D05D87|A174FB52126E|26E5D9FC07CC|1CF99798EDF1|4E1ECB87BC1F|B6A9BC006CB2|6943DF45F77E|E17388AB56E2|B65CCE479F2E|C02F4B1A3FE8") & 
                                               is.na(institutionCodeNew), "OFB-CNRS-MNHN", institutionCodeNew),
        institutionCodeNew = dplyr::if_else( stringr::str_detect(bibliographicCitation, "Wild bees of Belgium") & 
                                               is.na(institutionCodeNew), "Belgian Biodiversity Platform", institutionCodeNew),
        # COMBINE with    institutionCode                                         
        institutionCode = dplyr::if_else(is.na(institutionCode),
                                         institutionCodeNew,
                                         institutionCode) 
        ) %>%
      dplyr::select(!institutionCodeNew)
    
    # Combine
    data <- data %>%
      # Remove the occs from temp and then add them in again
      dplyr::filter(!database_id %in% temp$database_id) %>%
      dplyr::bind_rows(temp) 
    
      #### 1.2 Make edits ####
    data <- data %>%
        # Make sure USGS is included
      dplyr::mutate(institutionCode = dplyr::if_else(
        is.na(institutionCode) & stringr::str_detect(id, "USGS_DRO"),
        "USGS", institutionCode)) 
    
    data <- data %>%
        # Make some corrections for consistency
    dplyr::mutate(institutionCode = dplyr::if_else(
      stringr::str_detect(institutionCode, "C\\.A\\. Triplehorn Insect Collection, Ohio State University"),
      "C.A. Triplehorn Insect Collection, Ohio State University, Columbus, OH (OSUC)", institutionCode)) %>%
    dplyr::mutate(institutionCode = dplyr::if_else(
      stringr::str_detect(institutionCode, "Caledonian Conservation"),
      "Caledonian Conservation", institutionCode)) %>%
      dplyr::mutate(institutionCode = dplyr::if_else(
        stringr::str_detect(institutionCode, "^CardObs$"),
        "CardObs", institutionCode)) %>%
      dplyr::mutate(institutionCode = dplyr::if_else(
        stringr::str_detect(institutionCode, "Instituto Nacional de Pesquisas da Amaz"),
        "Instituto Nacional de Pesquisas da Amazonia (INPA)", institutionCode)) %>%
      dplyr::mutate(institutionCode = dplyr::if_else(
        stringr::str_detect(institutionCode, "\\(JBB\\)"),
        "Jardin Botanico de Bogota Jose Celestino Mutis (JBB)", institutionCode)) %>%
      dplyr::mutate(institutionCode = dplyr::if_else(
        stringr::str_detect(institutionCode, "^SDA$"),
        "SDA - Secretaria Distrital de Ambiente", institutionCode)) %>%
      dplyr::mutate(institutionCode = dplyr::if_else(
        stringr::str_detect(institutionCode, "Universidad del Magdalena"),
        "Universidad del Magdalena (UniMagdalena)", institutionCode)) %>%
      dplyr::mutate(institutionCode = dplyr::if_else(
        stringr::str_detect(institutionCode, "Universidad Nacional de Colombia"),
        "Universidad Nacional de Colombia (UNAL)", institutionCode)) %>%
      dplyr::mutate(institutionCode = dplyr::if_else(
        stringr::str_detect(institutionCode, "University of Guelph"),
        "University of Guelph", institutionCode)) %>%
      dplyr::mutate(institutionCode = dplyr::if_else(
        stringr::str_detect(institutionCode, "USDA[ -]ARS"),
        "USDA-ARS", institutionCode))
  }
  
  
  #### 2.0 Transformations table ####
    # Read the transformations table that has been manually constructed to link the input 
  # institutionCodes with those that hopefully match the intended institution code and name
  instTable <- dplyr::tribble(
                                                                                                                            ~institutionCode_in,                                           ~Code_out,                                                                                                                                                                                            ~Name_out,
                                                                                                                                         "SMNH",                                              "SMNH",                                                         "Saitama Museum of Natural History or Schmidt Museum of Natural History, Emporia State University or Saskatchewan Museum of Natural History",
                                                                                                                                          "ABS",                                               "ABS",                                                                                                                                                                             "Aberystwyth University",
                                                                                                                                          "MNA",                                               "MNA",                                                                                                                                                                     "Museo Nazionale dell'Antartide",
                                                                                                                                         "FMNH",                                              "FMNH",                                                                                                                                                                  "Finnish Museum of Natural History",
                                                                                                                                        "BISON",                                             "BISON",                                                                                                                                                        "Biodiversity Information Serving Our Nation",
                                                                                                                                        "AUMNH",                                             "AUMNH",                                                                                                                                                        "Auburn University Museum of Natural History",
                                                                "C.A. Triplehorn Insect Collection, Ohio State University, Columbus, OH (OSUC)",                                              "OSUC",                                                                                                                                                                              "Ohio State University",
                                                                                                                                        "NTSRV",                                             "NTSRV",                                                                                                                                                                                "NatureServe Network",
                                                                                                                                           "UI",                                                "UI",                                                                                                                                         "Bureau of Land Management, U.S. Department of the Interior",
                                                                                                                                          "MSM",                                               "MSM",                                                                                                                                     "Arizona Museum of Natural History or University of Puerto Rico",
                                                                                                                                           "AM",                                                "AM",                                                                                                                                                                                  "Australian Museum",
                                                                                                                                         "AMNH",                                              "AMNH",                                                                                                                                                                 "American Museum of Natural History",
                                                                                                                                       "AMNHTC",                                              "AMNH",                                                                                                                                                                 "American Museum of Natural History",
                                                                                                                                         "ANSP",                                              "ANSP",                                                                                                                                                                        "Academy of Natural Sciences",
                                                                                                                                          "ASU",                                               "ASU",                                                                                                                                               "Arizona State University Hasbrouck Insect Collection",
                                                                                                                                          "AUM",                                               "AUM",                                                                                                                                                                                  "Auburn University",
                                                                                                                                         "BBSL",                                              "BBSL",                                                                                                                                                       "Bee Biology and Systematics Laboratory, Utah",
                                                                                                                                        "BIOUG",                                             "BIOUG",                                                                                                                                                                   "Centre for Biodiversity Genomics",
                                                                                                                                          "BLM",                                               "BLM",                                                                                                                                                                          "Bureau of Land Management",
                                                                                                                                         "BRFC",                                              "BRFC",                                                                                                                                                                       "Black Rock Forest Collection",
                                                                                                                                          "BYU",                                               "BYU",                                                                                                                                                                           "Brigham Young University",
                                                                                                                                         "CAES",                                              "CAES",                                                                                                                                                        "Connecticut Agricultural Experiment Station",
                                                                                                                                         "CASC",                                               "CAS",                                                                                                                                                                     "California Academy of Sciences",
                                                                                                                                          "CAS",                                               "CAS",                                                                                                                                                                     "California Academy of Sciences",
                                                                                                                                         "CHAS",                                              "CHAS",                                                                                                                                                                        "Chicago Academy of Sciences",
                                                                                               "Cleveland Museum of Natural History, OH (CLEV)",                                              "CLEV",                                                                                                                                                            "Cleveland Museum of Natural History, OH",
                                                                                                                                          "CNC",                                               "CNC",                                                                                                                                  "Canadian National Collection of Insects, Arachnids, and Nematodes",
                                                                                                                                         "CSCA",                                              "CSCA",                                                                                                                                                          "California State Collection of Arthropods",
                                                                                                                                        "CSIRO",                                             "CSIRO",                                                                                                                                                       "Australian National Insect Collection, CSIRO",
                                                                                                                                          "CSU",                                               "CSU",                                                                                                                                        "Colorado State University or University of Central Oklahoma",
                                                                                                                                         "CUIC",                                              "CUIC",                                                                                                                                                               "Cornell University Insect Collection",
                                                                                                                                          "DBG",                                               "DBG",                                                                                                                                                                             "Denver Botanic Gardens",
                                                                                                                                          "DCH",                                               "DCH",                                                                                                                                                                                   "Davidson College",
                                                                                                                                       "DUGWAY",                                            "DUGWAY",                                                                                                                                                              "Dugway Proving Ground Natural History",
                                                                                                                                       "ECOSUR",                                            "ECOSUR",                                                                                                                                                                      "El Colegio de la Frontera Sur",
                                                                                                                                         "EMEC",                                              "EMEC",                                                                                                                                                   "Essig Museum, University of California, Berkeley",
                                                                                   "Glacier National Park Collections, West Glacier, MT (GLNP)",                                              "GLNP",                                                                                                                                         "Glacier National Park Collections, West Glacier, MT (GLNP)",
                                                                                                                                       "IBUNAM",                                            "IBUNAM",                                                                                                                  "Instituto de BiIolog\\u00eda, Universidad Nacional Aut\\u00f3noma de M\\u00e9xico",
                                                                                                                                  "iNaturalist",                                       "iNaturalist",                                                                                                                                                                                        "iNaturalist",
                                                                                                                                        "INBio",                                             "INBio",                                                                                                                                                                 "Instituo Nacional de Biodiversidad",
                                                                                                                                         "INHS",                                              "INHS",                                                                                                                                                                    "Illinois Natural History Survey",
                                                                                     "Instituto Nacional de Pesquisas da Amaz\\u00f4nia (INPA)",                                              "INPA",                                                                                                                                                  "Instituto Nacional de Pesquisas da Amaz\\u00f4nia",
                                                                                                                                         "KNWR",                                              "KNWR",                                                                                                                                                            "DOI/FWS, Kenai National Wildlife Refuge",
                                                                                                                                           "KU",                                                "KU",                                                                                                                                                                           "The University of Kansas",
                                                                                                                                           "KY",                                                "KY",                                                                                                                                                                             "University of Kentucky",
                                                                                                                                         "LACM",                                              "LACM",                                                                                                                                                       "Los Angeles County Museum of Natural History",
                                                                                                                                         "LCDI",                                              "LCDI",                                                                                                                                                           "Luther Entomological Research Collection",
                                                                                                                                          "MCZ",                                               "MCZ",                                                                                                                                                  "Harvard University, Museum of Comparative Zoology",
                                                                                                                                          "MEL",                                               "MEL",                                                                                                                                                                         "Museo Entomologico de Leon",
                                                                                                                                         "MHNG",                                              "MHNG",                                                                                                                                      "Mus\\u00e9um d'histoire naturelle de la Ville de Gen\\u00e8ve",
                                                                                                                                         "MHNN",                                              "MHNN",                                                                                                                                               "Mus\\u00e9um d'histoire naturelle de Neuch\\u00e2tel",
                                                                                                                                        "MISSA",                                             "MISSA",                                                                                                                                                                       "Mississippi State University",
                                                                                                                                         "MIZA",                                              "MIZA",                                                                                                                                 "Museo del Instituto de Zoologia Agricola Francisco Fernandez Yepez",
                                                                                                                                         "MNHN",                                              "MNHN",                                                                                                                                                        "MNHN - Museum national d'Histoire naturelle",
                                                                                                                                          "MSU",                                               "MSU",                                                                                                                                                                   "Michigan State University Museum",
                                                                                                 "Montana State University, Bozeman, MT (MTEC)",                                              "MTEC",                                                                                                                                                              "Montana State University, Bozeman, MT",
                                                                                                                                          "MZH",                                               "MZH",                                                                                                                                                                  "Finnish Museum of Natural History",
                                                                                                                                          "NAU",                                               "NAU",                                                                                                                                                                        "Northern Arizona University",
                                                                                                                                         "NAUF",                                              "NAUF",                                                                                                                                                             "Northern Arizona University, Flagstaff",
                                                                                                                                         "NCSU",                                              "NCSU",                                                                                                                                                  "North Carolina State University Insect Collection",
                                                                                                                                         "NHMO",                                              "NHMO",                                                                                                                                                              "University of Oslo, Zoological Museum",
                                                                                                                                        "NHMUK",                                             "NHMUK",                                                                                                                                                                     "Natural History Museum, London",
                                                                                                                                         "NMSU",                                              "NMSU",                                                                                                                                                                        "New Mexico State University",
                                                                                                                                         "NYBG",                                              "NYBG",                                                                                                                                                                          "New York Botanical Garden",
                                                                                                                                         "NYSM",                                              "NYSM",                                                                                                                                                                          "The New York State Museum",
                                                                                                             "New Zealand Arthropod Collection",                                              "NZAC",                                                                                                                                                                   "New Zealand Arthropod Collection",
                                                                                                                                         "NZCS",                                              "NZCS",                                                                                                                                             "University, National Zoological Collection of Suriname",
                                                                                                                                         "OMNH",                                              "OMNH",                                                                                                                                                                 "Oklahoma Museum of Natural History",
                                                                                                                                         "OSAC",                                              "OSAC",                                                                                                                                                                  "Oregon State Arthropod Collection",
                                                                                                                                         "PPRI",                                              "PPRI",                                                                                                          "ARC-Plant Protection Research Institute, National Collection of Fungi: Culture Collection",
                                                                                                                                         "PSUC",                                              "PSUC",                                                                                                                                                  "Frost Entomological Museum, Penn State University",
                                                                                                                                           "PU",                                                "PU",                                                                                                                                                       "The Purdue Entomological Research Collection",
                                                                                                                                         "PUCV",                                              "PUCV",                                                                                                                                                 "Pontifical Catholic University of Valpara\\u00edso",
                                                                                                                                         "QCAZ",                                              "QCAZ",                                                                                                                                     "Museo de Zoologia, Pontificia Universidad Catolica del Ecuador",
                                                                                                                                           "QM",                                                "QM",                                                                                                                                                                                  "Queensland Museum",
                                                                                                                                        "QVMAG",                                             "QVMAG",                                                                                                                                                              "Queen Victoria Museum and Art Gallery",
                                                                                                                                         "RBCM",                                              "RBCM",                                                                                                                                                                      "Royal British Columbia Museum",
                                                                                                                                         "RLMC",                                              "RLMC",                                                                                                                                                            "RL Minckley Insect and Plant Collection",
                                                                                                                                         "RMBL",                                              "RMBL",                                                                                                                                                               "Rocky Mountain Biological Laboratory",
                                                                                                                                         "RSKM",                                              "RSKM",                                                                                                                                                                          "Royal Saskatchewan Museum",
                                                                                                                    "Royal Saskatchewan Museum",                                              "RSKM",                                                                                                                                                                          "Royal Saskatchewan Museum",
                                                                                                                                         "RUAC",                                              "RUAC",                                                                                                                                                            "Rutgers University Entomological Museum",
                                                                                                                                          "SAM",                                               "SAM",                                                                                                                                                                            "South Australian Museum",
                                                                                                                                         "SBMN",                                             "SBMNH",                                                                                                                                                            "Santa Barbara Museum of Natural History",
                                                                                                                                        "SBMNH",                                             "SBMNH",                                                                                                                                      "Santa Barbara Museum of Natural History Entomology Collection",
                                                                                                                                         "SDMC",                                              "SDMC",                                                                                                                                                                   "San Diego Natural History Museum",
                                                                                                                                        "SDNHM",                                             "SDNHM",                                                                                                                                                           "San Diego Natural History Museum, theNAT",
                                                                                                                                         "SDSU",                                              "SDSU",                                                                                                                                                  "San Diego State University Museum of Biodiversity",
                                                                                                                                          "SFU",                                               "SFU",                                                                                                                                                                            "Simon Fraser University",
                                                                                                                                         "SWRS",                                              "SWRS",                                                                                                                                                                      "Southwestern Research Station",
                                                                                                                                         "TAMU",                                            "TAMUIC",                                                                                                                                                             "Texas A&M University Insect Collection",
                                                                                                                                          "TTU",                                               "TTU",                                                                                                                                                                              "Texas Tech University",
                                                                                                                                           "UA",                                                "UA",                                                                                                                                                            "University of Arizona Insect Collection",
                                                                                                          "University of Alberta Museums (UAM)",                                               "UAM",                                                                                                                                                                      "University of Alberta Museums",
                                                                                                                                          "UAM",                                               "UAM",                                                                                               "University of Alberta Museums or University of Arkansas at Monticello or University of Alaska Museum",
                                                                                                               "University of British Columbia",                                               "UBC",                                                                                                                                                                     "University of British Columbia",
                                                                                                                                          "UCD",                                               "UCD",                                                                                                                                                                    "University of California, Davis",
                                                                                 "University of Central Florida Collection of Arthopods (UCFC)",                                              "UCFC",                                                                                                                                              "University of Central Florida Collection of Arthopods",
                                                                                                                                          "UCM",                                               "UCM",                                                                                                                                                                      "University of Colorado Museum",
                                                                                                                                         "UCSB",                                              "UCSB",                                                                                                                                                             "University of California Santa Barbara",
                                                                                                                                         "UCSC",                                              "UCSC",                                                                                                                                                                "University of California Santa Cruz",
                                                                                                                                           "UD",                                                "UD",                                                                                                                                                  "University of Delaware Insect Research Collection",
                                                                                                                                         "UHIM",                                              "UHIM",                                                                                                                                                             "The University of Hawaii Insect Museum",
                                                                                                                                         "UMCE",                                              "UMCE",                                                                                                                                              "Universidad Metropolitana de Ciencias de la Educacion",
                                                                                                                                      "UMN EXT",                                           "UMN EXT",                                                                                                                                                                                "Minnesota Bee Atlas",
                                                                                                                                         "UMNH",                                              "UMNH",                                                                                                                                                                     "Natural History Museum of Utah",
                                                                                                      "Universidad Nacional de Colombia (UNAL)",                                              "UNAL",                                                                                                                                                                   "Universidad Nacional de Colombia",
                                                                                                                                         "UNHP",                                              "UNHP",                                                                                                                                                                        "University of New Hampshire",
                                                                                                                                          "UNM",                                               "UNM",                                                                                                                    "University of New Mexico, Division of Arthropods Museum of Southwestern Biology",
                                                                                                                                     "USDA-ARS",                                          "USDA-ARS",                                                                                                                                     "U.S. National Insects Collection Agricultural Research Service",
                                                                                                                                 "NPA-ARS-USDA",                                          "USDA-ARS",                                                                                                                                     "U.S. National Insects Collection Agricultural Research Service",
                                                                                                                                         "USGS",                                              "USGS",                                                                                                                                                                    "United States Geological Survey",
                                                                                                                                         "USNM",                                              "USNM",                                                                                                                                        "Smithsonian Institution, National Museum of Natural History",
                                                                                                                                         "UTEP",                                              "UTEP",                                                                                                                                                                       "University of Texas, El Paso",
                                                                                                                                          "UVM",                                               "UVM",                                                                                                                                  "Zadock Thompson Natural History Collection, University of Vermont",
                                                                                                   "UVM Zadock Thompson Zoological Collections",                                               "UVM",                                                                                                                                  "Zadock Thompson Natural History Collection, University of Vermont",
                                                                                                                                       "UWYMED",                                            "UWYMED",                                                                                                                                                 "University of Wyoming Dillon Lab Insect Collection",
                                                                                                                "Vermont Center for Ecostudies",                                               "VCE",                                                                                                                                                                      "Vermont Center for Ecostudies",
                                                                                                                                          "VCE",                                               "VCE",                                                                                                                                                                      "Vermont Center for Ecostudies",
                                                                                                                                          "VCU",                                               "VCU",                                                                                                                                                                   "Virginia Commonwealth University",
                                                                                                                                          "VPI",                                               "VPI",                                                                                                                                                "Virginia Polytechnic Institute and State University",
                                                                   "University of Idaho, W.F. Barr Entomological Collection, Moscow, ID (WFBM)",                                              "WFBM",                                                                                                                                                                 "W.F. Barr Entomological Collection",
                                                                            "M.T. James Museum, Washington State University, Pullman, WA (WSU)",                                               "WSU",                                                                                                                                                                        "Washington State University",
                                                                                                                                          "WVW",                                               "WVW",                                                                                                                                                                     "West Virginia Wesleyan College",
                                                                                                                                          "YPM",                                               "YPM",                                                                                                                                                                                "Yale Peabody Museum",
                                                                                                                                         "ZFMK",                                              "ZFMK",                                                                                                                                                     "Zoologisches Forschungsmuseum Alexander Koenig",
                                                                                                                                         "ZMUC",                                              "ZMUC",                                                                                                                                                        "University of Copenhagen, Zoological Museum",
                                                                                                                                          "ZSM",                                               "ZSM",                                                                                                                                                                  "Zoologische Museum Staatssammlung",
                                                                                                                                         "GMNH",                                              "GMNH",                                                                                                                                                                  "Georgia Museum of Natural History",
                                                                                                                                        "ABReC",                                             "ABReC",                                                                                                                                                                   "Argyll Biological Records Centre",
                                                                                                                                          "ACA",                                               "ACA",                                                                                                                                                                   "Alberta Conservation Association",
                                                                                                                                          "ADT",                                               "ADT",                                                                                                                                                                                 "Antarctic Division",
                                                                                                                  "AK Entomologie NABU Sachsen",                                    "AK Entomologie",                                                                                                                                                           "Arbeitskreis Entomologie im NABU Sachsen",
                                                                                                                                          "AMU",                                               "AMU",                                                                                                                                                         "Adam Mickiewicz University in Pozna\\u0144",
                                                                                                                                          "ARC",                                               "ARC",                                                                                                                        "Agricultural Research Council, National Collection of Insects, South Africa",
                                                                                                                  "Artenfinder Rheinland-Pfalz",                                              "ARPf",                                                                                                                                                                        "Artenfinder Rheinland-Pfalz",
                                                                                                                "Alberta Environment and Parks",                                              "BDUC",                                                                                                                                                                      "Alberta Environment and Parks",
                                                                                                                             "BioDiversity4All",                                  "BioDiversity4All",                                                                                                                                                                                   "BioDiversity4All",
                                                                                                                                     "BioFokus",                                          "BioFokus",                                                                                                                                                                                           "BioFokus",
                                                                 "Borror Laboratory of Bioacoustics, Ohio State University, Columbus, OH (BLB)",                                               "BLB",                                                                                                                             "Borror Laboratory of Bioacoustics, Ohio State University, Columbus, OH",
                                                                                                                                          "BNM",                                               "BNM",                                                                                                                                                                              "Belau National Museum",
                                                                                                                                           "LI",                                                "LI",                                                                                                                                                  "Biology Centre of the Upper Austrian State Museum",
                                                                                                            "Biodiversity Institute of Ontario",                                               "OAC",                                                                                                                                                                  "Biodiversity Institute of Ontario",
                                                                                                                                         "UFPB",                                              "UFPB",                                                                                                                                                             "Departamento de Sistematica e Ecologia",
                                                                                                                          "Manx Wildlife Trust",                                               "MWT",                                                                                                                                                                                "Manx Wildlife Trust",
                                                                                                                                         "AMDC",                                              "AMDC",                                                                                                                                                                                               "AMDC",
                                                                                                          "Universidad de los Andes (UniAndes)",                                             "ANDES",                                                                                                                                                                           "Universidad de los Andes",
                                                                                                                                           "AU",                                                "AU",                                                                                                                                                                                                 "AU",
                                                                                                                 "Bumblebee Conservation Trust",                                               "BCT",                                                                                                                                                                       "Bumblebee Conservation Trust",
                                                                                                                                          "BIS",                                               "BIS",                                                                                                                                                                                                "BIS",
                                                                                                                                          "BJS",                                               "BJS",                                                                                                                                                                                                "BJS",
                                                                                                                                        "BRERC",                                             "BRERC",                                                                                                                                                      "Bristol Regional Environmental Records Centre",
                                                                                                                                         "BROW",                                              "BROW",                                                                                                                                                              "The Broward College Insect Collection",
                                                                                                                                      "Buglife",                                           "Buglife",                                                                                                                                                                "The Invertebrate Conservation Trust",
                                                                                                                                       "BURKLE",                                            "BURKLE",                                                                                                                                                                                             "BURKLE",
                                                                                                                      "Caledonian Conservation",                           "Caledonian Conservation",                                                                                                                                                                            "Caledonian Conservation",
                                                                                                                                   "Calluna AB",                                        "Calluna AB",                                                                                                                                                                                         "Calluna AB",
                                                                                                                                         "CAWM",                                              "CAWM",                                                                                                                                                         "The College of African Wildlife Management",
                                                                                                             "Cumbria Biodiversity Data Centre",                                              "CBDC",                                                                                                                                                                   "Cumbria Biodiversity Data Centre",
                                                                                                                                          "CBM",                                               "CBM",                                                                                                          "Natural History Museum and Institute, Chiba or Working Collection M. Baehr, M\\u00fcnchen",
                                                                                                                                 "Pierre Fabre",                                              "CBPF",                                                                                                                                                               "Conservatoire Botanique Pierre Fabre",
                                                        "Corporaci\\u00f3n Aut\\u00f3noma Regional para la Defensa de la Meseta de Bucaramanga",                                              "CDMB",                                                                                                       "CDMB - Corporaci\\u00f3n Aut\\u00f3noma Regional Para la Defensa de la Meseta de Bucaramanga",
                                                                                                                                        "CEDaR",                                             "CEDaR",                                                                                                                                                                                              "CEDaR",
                                      "Federaci\\u00f3n Nacional de Cafeteros - Centro Nacional de Investigaciones de Caf\\u00e9 (\\ CENICAFf)",                                       "\\ CENICAFf",                                                                                                                                                   "Centro Nacional de Investigaciones de Caf\\u00e9",
                                                                                                                           "CENID-COMEF-INIFAP",                                "CENID-COMEF-INIFAP", "Centro Nacional en Investigaci\\u00f3n Disciplinaria en Conservaci\\u00f3n y Mejoramiento de Ecosistemas Forestales, Instituto Nacional de Investigaciones Forestales, Agr\\u00edcolas y Pecuarias",
                                                                                                                                   "COMFENALCO",                                              "CEPB",                                                                                                                                    "Colecci\\u00f3n entomol\\u00f3gica Piedras Blancas (Comfenalco)",
                                                                                                                                         "CfGA",                                              "CfGA",                                                                                                                                                                              "Caring For God's Acre",
                                                                                                                                       "CIBNOR",                                            "CIBNOR",                                                                                                                                               "Centro de Investigaciones Biologicas del Noroeste SC",
                                                                                                                          "Tullie House Museum",                                               "CLE",                                                                                                                                                                                "Tullie House Museum",
                                                                                                                                          "CLO",                                               "CLO",                                                                                                                                                                                                "CLO",
                                                                                                                                       "Cofnod",                                            "Cofnod",                                                                                                                                             "The Local Environmental Records Centre for North Wales",
                                                                                                                   "Conservation International",                        "Conservation International",                                                                                                                                                                         "Conservation International",
                                                                                                                                         "CORN",                                              "CORN",                                                                                                                                                                                               "CORN",
                                                                                                                                       "IMEDEA",                                       "CSIC-IMEDEA",                                                                                                                                          "Instituto Mediterr\\u00e1neo de Estudios Avanzados (CSIC)",
                                                                                                                                         "DART",                                              "DART",                                                                                                                                                                                  "Dartmouth College",
                                                                                                         "Derbyshire Biological Records Centre",                                              "DBRC",                                                                                                                                                               "Derbyshire Biological Records Centre",
                                                                                                                                         "DEBU",                                              "DEBU",                                                                                                                                                    "Ontario Insect Collection, University of Guelph",
                                                                                                                         "University of Guelph",                                              "DEBU",                                                                                                                                                             "University of Guelph Insect Collection",
                                                                                                          "Dorset Environmental Records Centre",                                              "DERC",                                                                                                                                                                "Dorset Environmental Records Centre",
                                                                                                                                           "DF",                                                "DF",                                                                                                                                                                                       "Douglas Frew",
                                                                                                                           "Middlebury College",                                              "DMIC",                                                                                                                                         "The Duncan MacDonald Insect Collection, Middlebury College",
                                                                                                                                          "DMP",                                               "DMP",                                                                                                                                                                                                "DMP",
                                                                                                                                         "DMSA",                                              "DMSA",                                                                                                                                                                                      "Durban Museum",
                                                                                                                    "Derbyshire Wildlife Trust",                                           "DWTrust",                                                                                                                                                                          "Derbyshire Wildlife Trust",
                                                                                                                                         "EBDA",                                               "BAH",                                                                                                                                                   "Empresa Baiana de Desenvolvimento Agr\\u00edcola",
                                                                                                                                       "Ecocom",                                            "EcoCom",                                                                                                                                                                              "Ecological Complexity",
                                                                                            "European Distributed Institute of Taxonomy (EDIT)",                                              "EDIT",                                                                                                                                                         "European Distributed Institute of Taxonomy",
                                                                                                                                     "EMEC-UTB",                                          "EMEC-UTB",                                                                                                          "Pollinator interaction flexibility across scales affects patch colonization and occupancy",
                                                                                                                           "Espace pour la vie",                                              "EPLV",                                                                                                                                                                                 "Espace pour la vie",
                                                                                                                                         "ETHZ",                                              "ETHZ",                                                                                                                                                 "Erdgen\\u00e4ssische Technische Hochschule-Zentrum",
                                                                                                                                     "FCB-UANL",                                          "FCB-UANL",                                                                                                                                                   "Facultad de Ciencias Biol\\u00f3gicas de la UANL",
                                                                                                                                      "FC-UNAM",                                           "FC-UNAM",                                                                                                                          "Facultad De Ciencias, Universidad Nacional Aut\\u00f3noma de M\\u00e9xico",
                                                                                                                                         "FiBL",                                              "FiBL",                                                                                          "Forschungsinstitut f\\u00fcr biologischen Landbau Frick | Research Institute of Organic Agriculture Frick",
                                                                                                                   "Fife Nature Records Centre",                                              "Fife",                                                                                                                                                                         "Fife Nature Records Centre",
                                                                                                                                         "Fife",                                              "Fife",                                                                                                                                                                         "Fife Nature Records Centre",
                                                  "Instituto de Investigaci\\u00f3n de Recursos Biol\\u00f3gicos Alexander von Humboldt (IAvH)",                                               "FMB",                                                                                                               "Instituto de Investigaci\\u00f3n de Recursos Biol\\u00f3gicos Alexander von Humboldt",
                                                                                                                                         "FMIC",                                              "FMIC",                                                                                                                                                                                               "FMIC",
                                                                                                                                    "FMVZ-UADY",                                         "FMVZ-UADY",                                                       "Veterinary Parasitology Laboratory at the Campus of Ciencias Biol\\u00f3gicas y Agropecuarias of Universidad Aut\\u00f3noma of Yucat\\u00e1n",
                                                                                   "Friends of the Earth (England, Wales and Northern Ireland)",                                              "FotE",                                                                                                                                         "Friends of the Earth (England, Wales and Northern Ireland)",
                                                                                                                                          "GEO",                                               "GEO",                                                                                                                                                                                   "Emory University",
                                                                                                                                        "GMNHJ",                                             "GMNHJ",                                                                                                                                                                    "Gunma Museum of Natural History",
                                                                                                                                 "Greensway AB",                                      "Greensway AB",                                                                                                                                                                                       "Greensway AB",
                                                                                                                                          "GSC",                                               "GSC",                                                                                                                                             "Geological Survey of Canada or Glenville State College",
                                                                                                                         "G\\u00e4teborgs Stad",                              "G\\u00e4teborgs Stad",                                                                                                                                                                               "G\\u00e4teborgs Stad",
                                                                                                                                         "HBRG",                                              "HBRG",                                                                                                                                                                "Highland Biological Recording Group",
                                                                                                                                          "HNS",                                               "HNS",                                                                                                                                                         "Mus\\u00e9um national d'Histoire naturelle",
                                                                                                                                          "HOC",                                               "HOC",                                                                                                                                                                                                "HOC",
                                                                                                                         "H\\u00e4rryda kommun",                              "H\\u00e4rryda kommun",                                                                                                                                                                               "H\\u00e4rryda kommun",
                                                                                                                                          "HYO",                                               "HYO",                                                                                                                                                       "Museum of Nature and Human Activities, Hyogo",
                                                                                                                                         "HZIC",                                              "HZIC",                                                                                                                                                         "Herbert Zettel Collection, Vienna, Austria",
                                                                                                                                      "IFIT-CP",                                           "IFIT-CP",                                                                                                                                                                                            "IFIT-CP",
                                                                                                                                     "IGL-UNAM",                                          "IGL-UNAM",                                                                                                                   "Instituto de Geolog\\u00eda, Universidad Nacional Aut\\u00f3noma de M\\u00e9xico",
                                                                                                                                   "IIZD-UASLP",                                        "IIZD-UASLP",                                                                                    "Instituto de Investigaci\\u00f3n de Zonas Des\\u00e9rticas, Universidad Aut\\u00f3noma de San Luis Potos\\u00ed",
                 "Instituto para la Investigaci\\u00f3n y la Preservaci\\u00f3n del Patrimonio Cultural y Natural del Valle del Cauca - INCIVA",                                            "INCIVA",                                                                       "Instituto para la Investigaci\\u00f3n y la Preservaci\\u00f3n del Patrimonio Cultural y Natural del Valle del Cauca - INCIVA",
                                                                                                                                          "INM",                                               "INM",                                                                                                                                                                              "Ibaraki Nature Museum",
                                                                                                                                         "INTA",                                              "INTA",                                                                                                                          "Instituto National Tecnolog\\u00eda Agropecuaria, Buenos Aires, Argentina",
                                                                                                                                         "IPMM",                                              "IPMM",                                                                                                                                                                           "Iwate Prefectural Museum",
                                                                                                                                        "IRNAD",                                             "IRNAD",                                                                                                          "INSTITUTO DE INVESTIGACIONES EN RECURSOS NATURALES, AGROECOLOGIA Y DESARROLLO RURAL IRNAD",
                                                                                                                           "Isagen S.A. E.S.P.",                                "Isagen S.A. E.S.P.",                                                                                                                                                                                 "Isagen S.A. E.S.P.",
                                                                 "Jard\\u00edn Bot\\u00e1nico de Bogot\\u00e1 Jos\\u00e9 Celestino Mutis (JBB)",                                               "JBB",                                                                                                                                             "Jard\\u00edn Bot\\u00e1nico Jos\\u00e9 Celestino Mutis",
                                                                                                                                          "JML",                                               "JML",                                                                                                                                                                                                "JML",
                                                                                                                              "John Muir Trust",                                           "JMTrust",                                                                                                                                                                                    "John Muir Trust",
                                                                                                                                          "JSA",                                               "JSA",                                                                                                                                                                                      "John S Ascher",
                                                                                                                                       "JSANUS",                                            "JSANUS",                                                                                                                                                                                             "JSANUS",
                                                                                                                                       "JSAOBS",                                            "JSAOBS",                                                                                                                                                                                      "John S Ascher",
                                                                                                                  "J\\u00e4rf\\u00e4lla Kommun",                       "J\\u00e4rf\\u00e4lla Kommun",                                                                                                                                                                        "J\\u00e4rf\\u00e4lla Kommun",
                                                                                                                                          "KAP",                                               "KAP",                                                                                                                                                                       "Kagoshima Prefectural Museum",
                                                                                                                                          "KMM",                                               "KMM",                                                                                                                                                                "Collection of Marine Microorganisms",
                                                                                                   "Kevin M. O'Neill Private Collection (KMOC)",                                              "KMOC",                                                                                                                                                         "Kevin M. O'Neill Private Collection (KMOC)",
                                                                                                                                         "KNPS",                                              "KNPS",                                                                                                                                                                      "Kootenay Native Plant Society",
                                                                                                                     "Kristianstads Vattenrike",                          "Kristianstads Vattenrike",                                                                                                                                                                           "Kristianstads Vattenrike",
                                                                                                                                         "ELKU",                                              "KYUM",                                                                                                                                                                           "Kyushu University Museum",
                                                                                                        "Lancashire Environment Record Network",                                              "LERN",                                                                                                                                                              "Lancashire Environment Record Network",
                                                                                                                                        "LRERC",                                             "LRERC",                                                                                                                                                               "Rutland Environmental Records Centre",
                                                                                                      "Laboratory of forest Sciences (LSF/UAC)",                                               "LSF",                                                                                                                                                                      "Laboratory of forest Sciences",
                                                                                                 "L\\u00e4nsstyrelsen \\u00d6sterg\\u00e4tland",      "L\\u00e4nsstyrelsen \\u00d6sterg\\u00e4tland",                                                                                                                                                       "L\\u00e4nsstyrelsen \\u00d6sterg\\u00e4tland",
                                                                                                             "L\\u00e4nsstyrelsen \\u00d6rebro",                  "L\\u00e4nsstyrelsen \\u00d6rebro",                                                                                                                                                                   "L\\u00e4nsstyrelsen \\u00d6rebro",
                                                                                                                 "L\\u00e4nsstyrelsen Blekinge",                      "L\\u00e4nsstyrelsen Blekinge",                                                                                                                                                                       "L\\u00e4nsstyrelsen Blekinge",
                                                                                                                  "L\\u00e4nsstyrelsen Dalarna",                       "L\\u00e4nsstyrelsen Dalarna",                                                                                                                                                                        "L\\u00e4nsstyrelsen Dalarna",
                                                                                                                  "L\\u00e4nsstyrelsen Gotland",                       "L\\u00e4nsstyrelsen Gotland",                                                                                                                                                                        "L\\u00e4nsstyrelsen Gotland",
                                                                                                                  "L\\u00e4nsstyrelsen Halland",                       "L\\u00e4nsstyrelsen Halland",                                                                                                                                                                        "L\\u00e4nsstyrelsen Halland",
                                                                                                                   "L\\u00e4nsstyrelsen Kalmar",                        "L\\u00e4nsstyrelsen Kalmar",                                                                                                                                                                         "L\\u00e4nsstyrelsen Kalmar",
                                                                                                                "L\\u00e4nsstyrelsen Kronoberg",                     "L\\u00e4nsstyrelsen Kronoberg",                                                                                                                                                                      "L\\u00e4nsstyrelsen Kronoberg",
                                                                                                               "L\\u00e4nsstyrelsen Norrbotten",                    "L\\u00e4nsstyrelsen Norrbotten",                                                                                                                                                                     "L\\u00e4nsstyrelsen Norrbotten",
                                                                                                       "L\\u00e4nsstyrelsen S\\u00e4dermanland",            "L\\u00e4nsstyrelsen S\\u00e4dermanland",                                                                                                                                                             "L\\u00e4nsstyrelsen S\\u00e4dermanland",
                                                                                                                  "L\\u00e4nsstyrelsen Uppsala",                       "L\\u00e4nsstyrelsen Uppsala",                                                                                                                                                                        "L\\u00e4nsstyrelsen Uppsala",
                                                                                                           "L\\u00e4nsstyrelsen V\\u00e4rmland",                "L\\u00e4nsstyrelsen V\\u00e4rmland",                                                                                                                                                                 "L\\u00e4nsstyrelsen V\\u00e4rmland",
                                                                                                       "L\\u00e4nsstyrelsen V\\u00e4sterbotten",            "L\\u00e4nsstyrelsen V\\u00e4sterbotten",                                                                                                                                                             "L\\u00e4nsstyrelsen V\\u00e4sterbotten",
                                                                                                     "L\\u00e4nsstyrelsen V\\u00e4sternorrland",          "L\\u00e4nsstyrelsen V\\u00e4sternorrland",                                                                                                                                                           "L\\u00e4nsstyrelsen V\\u00e4sternorrland",
                                                                                                        "L\\u00e4nsstyrelsen V\\u00e4stmanland",             "L\\u00e4nsstyrelsen V\\u00e4stmanland",                                                                                                                                                              "L\\u00e4nsstyrelsen V\\u00e4stmanland",
                                                                                              "L\\u00e4nsstyrelsen V\\u00e4stra G\\u00e4taland",   "L\\u00e4nsstyrelsen V\\u00e4stra G\\u00e4taland",                                                                                                                                                    "L\\u00e4nsstyrelsen V\\u00e4stra G\\u00e4taland",
                                                                                                                                          "LUL",                                               "LUL",                                                                                                                                                                                                "LUL",
                                                                                                                                           "MA",                                                "MA",                                                                                                                                                                                                 "MA",
                                                                                                                                        "MAGNT",                                             "MAGNT",                                                                                                                                                   "Museum and Art Gallery of the Northern Territory",
                                                                                                                                          "MBB",                                               "MBB",                                                                                                                                                                                                "MBB",
                                                                                              "Missoula Butterfly House and Insectarium (MBHI)",                                              "MBHI",                                                                                                                                                    "Missoula Butterfly House and Insectarium (MBHI)",
                                                                                                                                    "MBM-UACAM",                                         "MBM-UACAM",                                                                                                                                "Museo de Biodiversidad Maya, Universidad Aut\\u00f3noma de Campeche",
                                                                                                                                         "MBRP",                                              "MBRP",                                                                                                                                                                                               "MBRP",
                                                                                                                                         "MCSN",                                              "MCSN",                                                                                                                                                                    "Museo Civico di Storia Naturale",
                                                                                                                                        "MDEIE",                                             "MDEIE",                                                                                                                                                          "Darder Natural History Museum of Banyoles",
                                                                                                                                          "MEM",                                               "MEM",                                                                                                                                                                              "University of Memphis",
                                                                                                                                          "MfN",                                               "MfN",                                                                                                                                                                        "Museum f\\u00fcr Naturkunde",
                                                                                                                                        "MHNCM",                                             "MHNCM",                                                                                                                                                       "Natural History Museum of the City of Mexico",
                                                                                                                                         "MHNF",                                              "MHNF",                                                                                                                                                          "Mus\\u00e9e d'histoire naturelle Fribourg",
                                                                                                                                         "MJIM",                                              "MJIM",                                                                                                                                                                                               "MJIM",
                                                                                                                                         "MNHM",                                              "MNHM",                                                                           "Naturhistorisches Museum Mainz/Landessammlung f\\u00fcr Naturkunde Rheinland-Pfalz or John May Museum of Natural History",
                                                                                                                                        "MNHNC",                                             "MNHNC",                                                                                                                                                           "Museo Nacional de Historia Natural, Cuba",
                                                                                                                                        "MNHNL",                                             "MNHNL",                                                                                                                                                     "National Museum of Natural History, Luxembourg",
                                                                                                                                         "MNHW",                                              "MNHW",                                                                                                                                                                                               "MNHW",
                                                                                                                                         "MNVS",                                              "MNVS",                                                                                                                                                                 "Mus\\u00e9e de la nature du Valais",
                                                                                                                                         "MPEG",                                              "MPEG",                                                                                                                                                                       "Museu Paraense Emilio Goeldi",
                                                                                                                                 "MST-and-NHMD",                                      "MST-and-NHMD",                                                                                                                                                                  "Natural History Museum of Denmark",
                                                                                                                                          "MVY",                                               "MVY",                                                                                                                                                                                                "MVY",
                                                                                                                                          "MZL",                                               "MZL",                                                                                                                                                         "Mus\\u00e9e cantonal de zoologie, Lausanne",
                                                                                                    "North Ayrshire Countryside Ranger Service",                                             "NACRS",                                                                                                                                                          "North Ayrshire Countryside Ranger Service",
                                                                                                                              "Natagriwal asbl",                                   "Natagriwal asbl",                                                                                                                                                                                    "Natagriwal asbl",
                                                                                                                              "Natural England",                                   "Natural England",                                                                                                                                                                                    "Natural England",
                                                                                                                                  "naturgucker",                                       "naturgucker",                                                                                                                                                                                        "naturgucker",
                                                                                                                                   "Natuurpunt",                                        "Natuurpunt",                                                                                                                                                                                         "Natuurpunt",
                                                                                                                                         "NBDC",                                              "NBDC",                                                                                                                                                     "The National Biodiversity Data Centre, Ireland",
                                                                                                                                         "NBIS",                                              "NBIS",                                                                                                                                                           "Norfolk Biodiversity Information Service",
                                                                                                                                         "NDBC",                                              "NDBC",                                                                                                                                                                  "National Biodiversity Data Centre",
                                                                                                                                           "NE",                                                "NE",                                                                                                                                                                          "University of New England",
                                                                                                                                         "NEMU",                                              "NEMU",                                                                                                                                                                                      "Newark Museum",
                                                                                                                                         "NHMD",                                              "NHMD",                                                                                                                                                                  "Natural History Museum of Denmark",
                                                                                                                                         "NIEK",                                              "NIEK",                                                                                                                                                                                               "NIEK",
                                                                                                                                         "NINA",                                              "NINA",                                                                                                                                                            "Norwegian Institute for Nature Research",
                                                                                                                                          "NMB",                                               "NMB",                                                                                                                                                                     "Naturhistorisches Museum Basel",
                                                                                                                                         "NMBE",                                              "NMBE",                                                                                                                                                   "Naturhistorisches Museum der Burgergemeinde Bern",
                                                                                                                                    "NMBU:MINA",                                         "NMBU:MINA",                                                                                                         "Norwegian University of Life Sciences and Museo Naturalistico Francesco Min\\u00e0 Palumbo",
                                                                                                                                         "NMDG",                                              "NMDG",                                                                                                                                                                "Nathaniel Green Research Collection",
                                                                                                                                         "NMLU",                                              "NMLU",                                                                                                                                                                                "Natur-Museum Luzern",
                                                                                                                                       "NMMNHS",                                            "NMMNHS",                                                                                                                                                   "New Mexico Museum of Natural History and Science",
                                                                                                                                         "NMOK",                                              "NMOK",                                                                                                                                                             "Naturkundemuseum im Ottoneum of Kassel",
                                                                                                                                         "NMOL",                                              "NMOL",                                                                                                                                                                                  "Naturmuseum Olten",
                                                                                                                                          "NMR",                                               "NMR",                                                                                                                                                                     "Herbarium - Semyung University",
                                                                                                                                         "NMSA",                                              "NMSA",                                                                                                                                                                               "KwaZulu-Natal Museum",
                                                                                                                                         "NMSG",                                              "NMSG",                                                                                                                                                                             "Naturmuseum St. Gallen",
                                                                                                                                         "NMSH",                                              "NMSH",                                                                                                                                                               "Museum zu Allerheiligen Schaffhausen",
                                                                                                                                         "NMSO",                                              "NMSO",                                                                                                                                                                              "Naturmuseum Solothurn",
                                                                                                                                         "NMTG",                                              "NMTG",                                                                                                                                                                                "Naturmuseum Thurgau",
                                                                                                                                          "NMV",                                               "NMV",                                                                                                                                                                                 "Museum of Victoria",
                                                                                                                                        "NMWIN",                                             "NMWIN",                                                                                                                                                                             "Naturmuseum Winterthur",
                                                                                                                                          "NRM",                                               "NRM",                                                                                                                                         "Swedish Museum of Natural History - Zoological Collections",
                                                                                                                                          "NRW",                                               "NRW",                                                                                                                                                                     "NRW Regional Data: North Wales",
                                                                                                                      "Natural Resources Wales",                                           "NRWales",                                                                                                                                                                            "Natural Resources Wales",
                                                                                                                                         "NSBC",                                              "NSBC",                                                                                                                                                                                "North Sea Bird Club",
                                                                                                                                         "NSMK",                                              "NSMK",                                                                                                                                                                            "National Science Museum",
                                                                                                                                         "NSMT",                                              "NSMT",                                                                                                                                                              "National Museum of Nature and Science",
                                                                                                                                         "NSUR",                                              "NSUR",                                                                                                                                                                        "Naturkundliche Sammlung Uri",
                                                                                               "NSW Dept of Planning, Industry and Environment",                                          "NSW DPIE",                                                                                                                                                     "NSW Dept of Planning, Industry and Environment",
                                                                                                                                           "NT",                                                "NT",                                                                                  "Northern Territory Department of Land Resources Management or Department of Environment, Parks and Water Security",
                                                                                                                                      "NTNU-VM",                                           "NTNU-VM",                                                                                                                                                     "NTNU Museum of Natural History and Archaeology",
                                                                                                                                          "NTS",                                               "NTS",                                                                                                                                                "Nevada Operations Office, U.S. Department of Energy",
                                                                                                                                 "Nybro kommun",                                      "Nybro kommun",                                                                                                                                                                                       "Nybro kommun",
                                                                                                                                         "OHBR",                                              "OHBR",                                                                                                                                                                                      "Ontario Hydro",
                                                                                                                                         "PCYU",                                              "PCYU",                                                                                                                                                           "The Packer Collection at York University",
                                                                                                                                         "PMNH",                                              "PMNH",                                                                                                                                                                 "Pakistan Museum of Natural History",
                                                                                                                                   "Pro Natura",                                        "Pro Natura",                                                                                                                                                                                         "Pro Natura",
                                                                                                                                         "PRUN",                                              "PRUN",                                                                                                                                                                                               "PRUN",
                                                                                                                                        "PUCRS",                                             "PUCRS",                                                                                                                                  "Pontif\\u00edcia Universidade Cat\\u00f3lica do Rio Grande do Sul",
                                                                                                                                         "RBRC",                                              "RBRC",                                                                                                                                                             "USDA Regional Biomass Research Centers",
                                                  "Record, the Biodiversity Information System for Cheshire, Halton, Warrington and the Wirral",                                            "RECORD",                                                                                                                "The Biodiversity Information System for Cheshire, Halton, Warrington and the Wirral",
                                                                                                                                     "RESEARCH",                                          "RESEARCH",                                                                                                                                                                                           "RESEARCH",
                                                                                                                                          "RHS",                                               "RHS",                                                                                                                                                   "Plant Pathology, The Royal Horticultural Society",
                                                                                                                                         "RSPB",                                              "RSPB",                                                                                                                                                      "The Royal Society for the Protection of Birds",
                                                                                                                                         "SAMA",                                              "SAMA",                                                                                                                                                                            "South Australian Museum",
                                                                                                 "SDA - Secretar\\u00eda Distrital de Ambiente",                                               "SDA",                                                                                                                                                             "Secretar\\u00eda Distrital de Ambiente",
                                                                                                                                         "SEDN",                                              "SEDN",                                                                                                                                                                 "Shropshire Ecological Data Network",
                                                                                                                                         "SEHU",                                              "SEHU",                                                                                           "Laboratory of Systematic Entomology, The Hokkaido University Museum, Hokkaido University, Sapporo, Japan",
                                                                                                                                      "SEWBReC",                                           "SEWBReC",                                                                                                                                                       "South East Wales Biodiversity Records Centre",
                                                                                                      "Shire Group of Internal Drainage Boards",                                             "SGIDB",                                                                                                                                                            "Shire Group of Internal Drainage Boards",
                                                                                                                                         "SLIC",                                              "SLIC",                                                                                                                                                                                               "SLIC",
                                                                                                                            "SLU Artdatabanken",                                 "SLU Artdatabanken",                                                                                                                                                                                  "SLU Artdatabanken",
                                                                                                                    "Scottish Natural Heritage",                                               "SNH",                                                                                                                                                                          "Scottish Natural Heritage",
                                                                                                              "Staffordshire Ecological Record",                                           "StaffER",                                                                                                                                                                    "Staffordshire Ecological Record",
                                                                                                                                          "SWT",                                               "SWT",                                                                                                                                                                            "Scottish Wildlife Trust",
                                                                                                                                        "SxBRC",                                             "SxBRC",                                                                                                                                                                  "Sussex Biodiversity Record Centre",
                                                                                                                                         "TAAM",                                              "TAAM",                                                                                                   "Institute of Agricultural and Environmental Sciences of the Estonian University of Life Sciences",
                                                                                                                                          "TAM",                                               "TAM",                                                                                                                                                                 "Estonian Museum of Natural History",
                                                                                                                                         "TCNS",                                              "TCNS",                                                                                                                                                                       "Toyota city nature sanctuary",
                                                                                                                                         "TMAG",                                              "TMAG",                                                                                                                                                               "The Tasmanian Museum and Art Gallery",
                                                                                                                                         "TOYA",                                              "TOYA",                                                                                                                                                                              "Toyama Science Museum",
                                                                                                                                 "Trafikverket",                                      "Trafikverket",                                                                                                                                                                                       "Trafikverket",
                                                                                                                         "Royal Ontario Museum",                                              "TRTC",                                                                                                                                                                               "Royal Ontario Museum",
                                                                                                                                         "UTSC",                                              "TRTS",                                                                                                                                                         "Scarborough College, University of Toronto",
                                                                                                    "Tohoku university and Yamagata university",                                              "TUSG",                                                                                                                                                          "Tohoku University and Yamagata university",
                                                                                                                                         "TWIC",                                              "TWIC",                                                                                                                                                                           "Twickenham Girls' School",
                                                                                                                                         "UAAM",                                              "UAAM",                                                                                                                                                       "The Arthropod Museum, University of Arkansas",
                                                                                                                                         "UACJ",                                              "UACJ",                                                                                                                                                  "Universidad Aut\\u00f3noma de Ciudad Ju\\u00e1rez",
                                                                                                                                         "UBCZ",                                              "UBCZ",                                                                                                                                  "University of British Columbia - Spencer Entomological Collection",
                                                                                                                                          "UCH",                                               "UCH",                                                                                                                                                       "Universidad Aut\\u00f3noma de Chiriqu\\u00ed",
                                                                         "University of Colorado Museum of Natural History, Boulder, CO (UCMC)",                                              "UCMC",                                                                                                                               "University of Colorado Museum of Natural History, Boulder, CO (UCMC)",
                                                                                                                                         "UCMS",                                              "UCMS",                                                                                                                                        "University of Connecticut Biodiversity Research Collections",
                                                                                                                                         "UCRC",                                              "UCRC",                                                                                                                                                                "University of California, Riverside",
                                                                                                                                         "UCSD",                                              "UCSD",                                                                                                                                                                 "University of California San Diego",
                                                                                                                                        "UDLAP",                                             "UDLAP",                                                                                                                                                             "The Universidad de las Americas-Puebla",
                                                                                                                                         "UFCG",                                              "UFCG",                                                                                                                                                             "Universidade Federal de Campina Grande",
                                                                                                                                         "UFMS",                                              "UFMS",                                                                                                                                                         "Universidade Federal de Mato Grosso do Sul",
                                                                                                                                         "UFPR",                                              "UFPR",                                                                                                                                                               "Universidade Federal do Paran\\u00e1",
                                                                                                                                        "Ugent",                                             "Ugent",                                                                                                                                                                                   "Ghent University",
                                                                                                                                          "UIC",                                               "UIC",                                                                                                                                                                     "University of Illinois Chicago",
                                                                                                                                          "nef",                                               "UiO",                                                                                                                                                                                 "University of Oslo",
                                                                                                                                          "UiT",                                               "UiT",                                                                                                                                                                "UiT The Arctic University of Norway",
                                                                                                                                          "ULB",                                               "ULB",                                                                                                                                                                "Universit\\u00e9 Libre de Bruxelles",
                                                                                                                                          "UMH",                                               "UMH",                                                                                                                                                                 "Universidad Miguel Hern\\u00e1ndez",
                                                                                                                                         "UMIC",                                              "UMIC",                                                                                                                                                                          "University of Mississippi",
                                                                                                                                         "UMMZ",                                              "UMMZ",                                                                                                                                                                  "University of Michigan, Ann Arbor",
                                                                                                                                 "UMS PatriNat",                                      "UMS PatriNat",                                                                                                                                                                                       "UMS PatriNat",
                                                                                                                                         "UNHC",                                              "UNHC",                                                                                                                                                                                               "UNHC",
                                                                                                                  "University of Northern Iowa",                                               "UNI",                                                                                                                                                                        "University of Northern Iowa",
                                                                                                     "Universidad de la Amazonia (Uniamazonia)",                                       "Uniamazonia",                                                                                                                                                                         "Universidad de la Amazonia",
                                                                                                                                          "UOG",                                               "UOG",                                                                                                                                                                               "University of Guelph",
                                                                                                                                         "UPRM",                                              "UPRM",                                                                                                                               "University of Puerto Rico at Mayagueez, Rhizobium Culture Collection",
                                                                                                                               "UQUIND\\u00e5O",                                    "UQUIND\\u00e5O",                                                                                                                                                                                     "UQUIND\\u00e5O",
                                                                                                                      "University of Rochester",                                                "UR",                                                                                                                                                                            "University of Rochester",
                                                                                                                                          "USP",                                               "USP",                                                                                                                                                                          "Universidad San Pablo-CEU",
                                                                                                                                       "USP-RP",                                            "USP-RP",                                                                                                                                                 "University of S\\u00e3o Paulo Ribeir\\u00e3o Preto",
                                                                                                                                          "UTE",                                               "UTE",                                                                                                                                                         "University of Tartu Natural History Museum",
                                                                                                                                         "UTIC",                                              "UTIC",                                                                                                                                    "University of Texas, Biodiversity Center, Entomology Collection",
                                                                                                                "Ville de Trois-Rivi\\u00e8res",                                              "VdTR",                                                                                                                                                                      "Ville de Trois-Rivi\\u00e8res",
                                                                                                                                         "VTST",                                              "VTST",                                                                                                                                                                                               "VTST",
                                                                                                                                         "WASH",                                              "WASH",                                                                                                                                                                                "Washburn University",
                                                                                                                                 "WCS Colombia",                                              "WCSC",                                                                                                                                                       "Wildlife Conservation Society (WCS Colombia)",
                                                                                                 "Wildlife Conservation Society (WCS Colombia)",                                              "WCSC",                                                                                                                                                       "Wildlife Conservation Society (WCS Colombia)",
                                                                                                              "The Wildlife Information Centre",                                            "WildIC",                                                                                                                                                                    "The Wildlife Information Centre",
                                                                                                                             "Woodmeadow Trust",                                             "WoodT",                                                                                                                                                                                   "Woodmeadow Trust",
                                                                                                                                          "WPU",                                               "WPU",                                                                                                                                                                        "William Paterson University",
                                                                                                                                         "WSDA",                                              "WSDA",                                                                                                                                                         "Washington State Department of Agriculture",
                                                                                                                                          "WSL",                                               "WSL",                                                                                                                                    "Swiss Federal Institute for Forest, Snow and Landscape Research",
                                                                                                                                  "WUR-Alterra",                                               "WUR",                                                                                                                                                                  "Wageningen Universiy and Research",
                                                                                                                                        "WWBIC",                                             "WWBIC",                                                                                                                                                         "West Wales Biodiversity Information Centre",
                                                                                                                                         "YPYM",                                              "YPYM",                                                                                                                                                                       "Yamaguchi Prefectural Museum",
                                                                                                                                        "YorkU",                                              "YUTO",                                                                                                                                                                                    "York University",
                                                                                                                                         "YUTO",                                              "YUTO",                                                                                                                                                                                    "York University",
                                                                                                                                          "YWT",                                               "YWT",                                                                                                                                                                           "Yorkshire Wildlife Trust",
                                                                                                                                         "ZMBN",                                              "ZMBN",                                                                                                                                                    "Zoological Museum, University of Bergen, Norway",
                                                                                                                                          "ZMZ",                                               "ZMZ",                                                                                                                                                                   "Zoologisches Museum Z\\u00fcrich",
                                                                                                                                           "ZS",                                                "ZS",                                                                                                                                                                                                 "ZS",
                                                                                                                                         "USFQ",                                              "ZSFQ",                                                                                                                                        "Museo de Zoolog\\u00eda, Universidad San Francisco de Quito",
                                                                                                                                        "MMWVU",                                             "MMWVU",                                                                                                                                                         "Mathew McKinney\\u00eds Private Collection",
                                                                                                                                "891.780.111-8",                                            "UNIMAG",                                                                                                                                                                          "Universidad del Magdalena",
                                                                                                                                             NA,                                                  NA,                                                                                                                                                                                            "UNKNOWN",
                                                                                                                                       "Melica",                                            "Melica",                                                                                                                                                                                             "Melica",
                                                                                                                                         "JMSC",                                              "JMSC",                                                                                                                                                                                     "John M. Snider",
                                                                                                                                          "UMA",                                               "UMA",                                                                                                                                                     "University of Massachusetts, Museum of Zoology",
                                                                                                                                          "PGC",                                               "PGC",                                                                                                                                                                         "The Petersburg Garden Club",
                                                                                                                                       "TAMUIC",                                            "TAMUIC",                                                                                                                                                             "Texas A&M University Insect Collection",
                                                                                                                                          "WSU",                                               "WSU",                                                                                                                                                                        "Washington State University",
                                                                                                                    "Atlas of Living Australia",                                               "ALA",                                                                                                                                                                          "Atlas of Living Australia",
                                                                                                                         "North Sydney Council",                                             "NSydC",                                                                                                                                                                               "North Sydney Council",
                                                                                                                                         "BDRS",                                              "BDRS",                                                                                                                                                                                               "BDRS",
                                                                                                                                         "ANIC",                                              "ANIC",                                                                                                                                                              "Australian National Insect Collection",
                                                                                                                                         "UFMG",                                              "UFMG",                                                                                                                                                                 "Federal University of Minas Gerais",
                                                                                                                                         "FSCA",                                              "FSCA",                                                                                                                                   "Florida State Collection of Arthropods, The Museum of Entomology",
                                                                                                                                          "DAR",                                               "DAR",                                                                                                                                                                      "Orange Agricultural Institute",
                                                                                                                                        "TKIPM",                                             "TKIPM",                                                                                                                                                                                              "TKIPM",
                                                                                                       "Pontificia Universidad Javeriana (PUJ)",                                               "PUJ",                                                                                                                                                                   "Pontificia Universidad Javeriana",
                                                                                                                                         "ICMI", "Insect collection of Itami City Museum of Insects",                                                                                                                                                                                               "ICMI",
                                                                                                                                        "Sweco",                                             "Sweco",                                                                                                                                                                                              "Sweco",
                                                                                                                                     "CTALA_LB",                                          "CTALA_LB",                                                                                                                                                             "Ministerio del Medio Ambiente de Chile",
                                                                                                    "L\\u00e4nsstyrelsen J\\u00e4nk\\u00e4ping",         "L\\u00e4nsstyrelsen J\\u00e4nk\\u00e4ping",                                                                                                                                                          "L\\u00e4nsstyrelsen J\\u00e4nk\\u00e4ping",
                                                                                                     "Svenska Botaniska F\\u00e4reningen (SBF)",                                               "SBF",                                                                                                                                                                 "Svenska Botaniska F\\u00e4reningen",
                                                                                                                      "Falk\\u00e4pings kommun",                           "Falk\\u00e4pings kommun",                                                                                                                                                                            "Falk\\u00e4pings kommun",
                                                                                                                      "Link\\u00e4pings kommun",                           "Link\\u00e4pings kommun",                                                                                                                                                                            "Link\\u00e4pings kommun",
                                                                                                                "L\\u00e4nsstyrelsen Stockholm",                     "L\\u00e4nsstyrelsen Stockholm",                                                                                                                                                                      "L\\u00e4nsstyrelsen Stockholm",
                                                                                                 "G\\u00e4teborgs naturhistoriska museum (GNM)",                                               "GNM",                                                                                                                                                             "G\\u00e4teborgs naturhistoriska museum",
                                                                                                                       "IAS (invasivaarter.nu)",                                            "IAS.nu",                                                                                                                                                                                   "invasivaarter.nu",
                                                                                                           "L\\u00e4nsstyrelsen J\\u00e4mtland",                "L\\u00e4nsstyrelsen J\\u00e4mtland",                                                                                                                                                                 "L\\u00e4nsstyrelsen J\\u00e4mtland",
                                                                                                                                         "MCNB",                                              "MCNB",                                                                                                                                                      "Museu de Ci\\u00e8ncies Naturals de Barcelona",
                                                                                             "East Ayrshire Leisure Countryside Ranger Service",                                            "EALCRS",                                                                                                                                                   "East Ayrshire Leisure Countryside Ranger Service",
                                                                                                                                          "CBG",                                              "CANB",                                                                                                                                                                      "Australian National Herbarium",
                                                                                                                           "University of Mons",                                             "UMons",                                                                                                                                                                                 "University of Mons",
                                                                                                                                        "BDBCV",                                             "BDBCV",                                                                                                                                                  "Biodiversity Data Bank of the Valencian Community",
                                                                                                                                          "MSB",                                               "MSB",                                                                                                                                                                     "Museum of Southwestern Biology",
                                                                                                     "Universidad del Magdalena (UniMagdalena)",                                            "UNIMAG",                                                                                                                                                                          "Universidad del Magdalena",
                                                                                                                      "Agropecuaria Aliar S.A.",                                             "Aliar",                                                                                                                                                                            "Agropecuaria Aliar S.A.",
                                                                                                                                        "MnhnL",                                             "MNHNL",                                                                                                                                                     "National Museum of Natural History, Luxembourg",
                                                                                                                        "IB FRC Komi SC UB RAS",                             "IB FRC Komi SC UB RAS",                                                                                                  "Institute of Biology of Komi Science Centre of the Ural Branch of the Russian Academy of Sciences",
                                                                                                                                          "RCM",                                               "RCM",                                                                                                                                                                                                "RCM",
                                                                                                                                         "NHMW",                                              "NHMW",                                                                                                                                                                      "Naturhistorisches Museum Wien",
                                                                                                                                          "UAc",                                               "UAC",                                                                                                                                                                              "University of Calgary",
                                                                          "Universidad Pedag\\u00f3gica y Tecnol\\u00f3gica de Colombia (UPTC)",                                              "UPTC",                                                                                                                                       "Universidad Pedog\\u00f3gica y Tecnol\\u00f3gica de Colombia",
                                                     "Corporaci\\u00f3n Colombiana de Investigaci\\u00f3n Agropecuaria - AGROSAVIA (Agrosavia)",                                         "Agrosavia",                                                                                                                       "Corporaci\\u00f3n Colombiana de Investigaci\\u00f3n Agropecuaria - AGROSAVIA",
                                                                                                       "Aguas de Bogot\\u00e1 S.A. E.S.P. (AB)",                                             "SAESP",                                                                                                                                                                       "Aguas de Bogot\\u00e1 SA ESP",
                                                                                                                                       "UNICAL",                                            "UNICAL",                                                                                                                                                                    "Universit\\u00e0 Della Calabria",
                                                                                                                                          "MoJ",                                               "MoJ",                                                                                                                                                                                                "MoJ",
                                                                                                                                         "AKPM",                                              "AKPM",                                                                                                                                                               "Herbarium - Akita Prefectural Museum",
                                                                                                                                         "KCMN",                                              "KCMN",                                                                                                                                                             "Kaizuka City Museum of Natural History",
                                                                                                     "Dalarnas Botaniska S\\u00e4llskap (DABS)",                                              "DABS",                                                                                                                                                                  "Dalarnas Botaniska S\\u00e4llskap",
                                                                                                                                          "SBP",                                               "SBP",                                                                                                                                                                                                "SBP",
                                                                                                                                         "DRIB",                                              "DRIB",                                                                                                                                                                                               "DRIB",
                                                                        "Museo Javeriano de Historia Natural, Pontificia Universidad Javeriana",                                              "MPUJ",                                                                                                                              "Museo Javeriano de Historia Natural, Pontificia Universidad Javeriana",
                                                                                                                                          "ULL",                                               "ULL",                                                                                                                                                                                                "ULL",
                                                                                                                                          "ARM",                                               "ARM",                                                                                                                                                                                      "County Museum",
                                                                                                                                         "KCMH",                                              "KCMH",                                                                                                                                                                                "Kushiro City Museum",
                                                                                                                                   "FACT(TFRI)",                                              "TFRI",                                                                                                                                                                 "Taiwan Forestry Research Institute",
                                                                                                                                      "CardObs",                                           "CardObs",                                                                                                                                                                                            "CardObs",
                                                                                                                                          "GNM",                                               "GNM",                                                                                                                            "Gothenburg Museum of Natural History (Goteborgs Naturhistoriska Museum)",
                                                                                                                                          "UIB",                                               "UIB",                                                                                                                                                                               "University of Bergen",
                                                                                                                                         "NCMG",                                              "NCMG",                                                                                                                                                                "Nottingham City Museums & Galleries",
                                                                                                         "The Conservation Volunteers Scotland",                                               "CVS",                                                                                                                                                               "The Conservation Volunteers Scotland",
                                                                                                                                          "KOM",                                               "KOM",                                                                                                                                                                                "Komatsu City Museum",
                                                                                                  "ODDB-0NG/Direction de la Biodiversit\\u00e9",                                          "ODDB ONG",                                                                                                                          "Organisation pour le D\\u00e9veloppement Durable et la Biodiversit\\u00e9",
                                                                                                                                         "NMNL",                                              "NMNL",                                                                                                                                                                                   "Museum De Bastei",
                                                                                                                                         "CELP",                                              "CELP",                                                                                                                                                          "Catalina Environmental Leadership Program",
                                                                                                                                         "UACH",                                              "UACH",                                                                                                                                                                      "Universidad Autonoma Chapingo",
                                                                                                                                          "SAG",                                               "SAG",                                                                                                                                                "Sammlung von Algenkulturen at Universitat Gottingen",
                                                                                                           "Universidad de La Salle (La Salle)",                                               "BOG",                                                                                                                                                                            "Universidad de La Salle",
                                                                                                                                    "InDRE-SSA",                                             "INDRE",                                                                                                                                             "Institute for Epidemiological Diagnosis and References",
                                                                                                                                         "BGBM",                                              "BGBM",                                                                                                                                                         "Botanic Garden and Botanical Museum Berlin",
                                                                                                                                  "Anymals.org",                                       "Anymals.org",                                                                                                                                                                                        "Anymals.org",
                                                                      "CRQ - Corporaci\\u00f3n Aut\\u00f3noma Regional del Quind\\u00edo (CRQ)",                                               "CRQ",                                                                                                                                        "Corporaci\\u00f3n Aut\\u00f3noma Regional del Quind\\u00edo",
                                                                                                                          "Sk\\u00e4vde kommun",                               "Sk\\u00e4vde kommun",                                                                                                                                                                                "Sk\\u00e4vde kommun",
                                                                                                  "Universidade Federal do Paran\\u00e1 (UFPR)",                                              "UPCB",                                                                                                                                                               "Universidade Federal do Paran\\u00e1",
                                                                                         "Universidad Tecnol\\u00f3gica del Choc\\u00f3 (UTCH)",                                             "CHOCO",                                                                                                                                                      "Universidad Tecnol\\u00f3gica del Choc\\u00f3",
                                                                                                                                         "DIBA",                                              "DIBA",                                                                                                                                                                                               "DIBA",
                                                                                                                                         "GUAM",                                              "GUAM",                                                                                                                                                                    "University of Guam, UOG Station",
                                                                                                                                          "MBA",                                               "MBA",                                                                                                                                                                    "Environmental Protection Agency",
                                                                                                                       "CardObs - UMS PatriNat",                                      "UMS PatriNat",                                                                                                                                                                                       "UMS PatriNat",
                                                                                                                                         "ZMUO",              "University of Oulu Zoological Museum",                                                                                                                                                                                               "ZMUO",
                                                                                                                                     "DFFW2018",                                          "DFFW2018",                                                                                                                                                                                           "DFFW2018",
                                                                                                          "L\\u00e4nsstyrelsen G\\u00e4vleborg",               "L\\u00e4nsstyrelsen G\\u00e4vleborg",                                                                                                                                                                "L\\u00e4nsstyrelsen G\\u00e4vleborg",
                                                                                                                                         "UCLA",                                              "UCLA",                                                                                                                                                               "University of California Los Angeles",
                                                                                                                       "Finsp\\u00e5ngs kommun",                            "Finsp\\u00e5ngs kommun",                                                                                                                                                                             "Finsp\\u00e5ngs kommun",
                                                                                                              "L\\u00e4nsstyrelsen Sk\\u00e5ne",                   "L\\u00e4nsstyrelsen Sk\\u00e5ne",                                                                                                                                                                    "L\\u00e4nsstyrelsen Sk\\u00e5ne",
                                                                                                                                         "BMEC",                                              "UCLA",                                                                                                                                                               "University of California Los Angeles",
                                                                                                                                         "PNMH",                                              "PNMH",                                                                                                                                                                "Palestine Museum of Natural History",
                                                                                                                                        "NIBIO",                                             "NIBIO",                                                                                                                                                         "Norwegian Institute of Bioeconomy Research",
                                                                                                                                          "UVG",                                               "UVG",                                                                                                                                                                              "Universidad del Valle",
                                                                                                                                 "Marks kommun",                                      "Marks kommun",                                                                                                                                                                                       "Marks kommun",
                                                                                                                          "Florav\\u00e4ktarna",                               "Florav\\u00e4ktarna",                                                                                                                                                                                "Florav\\u00e4ktarna",
                                                                                                                             "Bor\\u00e5s Stad",                                  "Bor\\u00e5s Stad",                                                                                                                                                                                   "Bor\\u00e5s Stad",
                                                                                                                                         "OAFS",                                              "OAFS",                                                                                                                                                                                               "OAFS",
                                                                                                                                           "CU",                                               "UCM",                                                                                                                                                   "University of Colorado Museum of Natural History",
                                                                                                                              "Stockholms Stad",                                   "Stockholms Stad",                                                                                                                                                                                    "Stockholms Stad",
                                                                                                                            "Svenljunga kommun",                                 "Svenljunga kommun",                                                                                                                                                                                  "Svenljunga kommun",
                                                                                                                                          "PSU",                                              "PSUC",                                                                                                                                                  "Frost Entomological Museum, Penn State University",
                                                                                                                                          "WAM",                                               "WAM",                                                                                                                                                                          "Western Australian Museum",
                                                                                                                                        "PaDIL",                                             "PaDIL",                                                                                                                                                                   "Pests and Diseases Image Library",
                                                                                      "Chulalongkorn University Natural History Museum (CUNHM)",                                             "CUNHM",                                                                                                                                                    "Chulalongkorn University Natural History Museum",
                                                                                                                                       "Flickr",                                            "Flickr",                                                                                                                                                                                             "Flickr",
                                                                                                                                   "QuestaGame",                                        "QuestaGame",                                                                                                                                                                                         "QuestaGame",
                                                                                                                                    "bowerbird",                                         "bowerbird",                                                                                                                                                                                          "bowerbird",
                                                                                        "South Australia, Department for Environment and Water",                                            "DEW_SA",                                                                                                                                              "South Australia, Department for Environment and Water",
                                                                                                                                  "NatureShare",                                       "NatureShare",                                                                                                                                                                                        "NatureShare",
                                                                                                                           "Wildlife Watch NSC",                                             "WWNSC",                                                                                                                                                                                 "Wildlife Watch NSC",
                                                                                                                  "European Nucleotide Archive",                                               "ENA",                                                                                                                                                                        "European Nucleotide Archive",
                                                                                                                                         "BOLD",                                              "BOLD",                                                                                                                                                                       "Barcode of Life Data Systems",
                                                                                                                    "Biological Records Centre",                                               "BRC",                                                                                                                                                                          "Biological Records Centre",
                                                                                                                              "Observation.org",                                   "Observation.org",                                                                                                                                                                                    "Observation.org",
                                                                                                                                       "FinBIF",                                            "FinBIF",                                                                                                                                                          "Finnish Biodiversity Information Facility",
                                                                                                   "Ukrainian Nature Conservation Group (UNCG)",                                              "UNCG",                                                                                                                                                                "Ukrainian Nature Conservation Group",
                                                                                                                  "Station d'Ecologie de Lamto",                                               "SEL",                                                                                                                                                                        "Station d'Ecologie de Lamto",
                                                                                                                                "OFB-CNRS-MNHN",                                      "UMS PatriNat",                                                                                                                                                                                       "UMS PatriNat",
                                                                                                              "Riiklik keskkonnaseire programm",                   "Riiklik keskkonnaseire programm",                                                                                                                                                                    "Riiklik keskkonnaseire programm",
                                                                                                                                       "Plotuf",                                            "Plotuf",                                                                                                                                                                                             "Plotuf",
                                                                                                     "Suffolk Biodiversity Information Service",                                              "SBIS",                                                                                                                                                           "Suffolk Biodiversity Information Service",
                                                                                                                                      "SPIPOLL",                                           "SPIPOLL",                                                                                                                                                   "Suivi Photographique des Insectes POLLinisateurs",
                                                                                               "National Museum of Natural History, Luxembourg",                                             "MNHNL",                                                                                                                                                     "National Museum of Natural History, Luxembourg",
                                                                                                                                         "CSCF",                                              "CSCF",                                                                                                                                                          "Centre Suisse de Cartographie de la Faune",
                                                                                                                  "Royal Horticultural Society",                                               "RHS",                                                                                                                                                                        "Royal Horticultural Society",
                                                                                                                "Belgian Biodiversity Platform",                                               "BBP",                                                                                                                                                                      "Belgian Biodiversity Platform",
                                                                                                                                   "NatureSpot",                                        "NatureSpot",                                                                                                                                                                                         "NatureSpot",
                                                                                                                "Naturalis Biodiversity Center",                                         "Naturalis",                                                                                                                                                                      "Naturalis Biodiversity Center",
                                                                                                            "Natural History Museum of Denmark",                                              "NHMD",                                                                                                                                                                  "Natural History Museum of Denmark",
                                                                                                                                      "WildNet",                                           "WildNet",                                                                                                                                                                                            "WildNet",
                                                                                                                          "Chronicle of Nature",                               "Chronicle of Nature",                                                                                                                                                                                "Chronicle of Nature",
                                                                                                                                      "ECNRUFC",                                           "ECNRUFC",                                                                                                                                                                                            "ECNRUFC",
                                                                                                                        "Queensland Government",                                            "QLDGov",                                                                                                                                                                              "Queensland Government",
                                                                                                         "Prioksko-Terrasnyi Biosphere Reserve",                                              "PTBR",                                                                                                                                                               "Prioksko-Terrasnyi Biosphere Reserve",
                                                                                                 "Norwegian University of Life Sciences (NMBU)",                                              "NMBU",                                                                                                                                                       "Norwegian University of Life Sciences (NMBU)",
                                                                                                                      "Ministry of Justice, UK",                                             "MoJUK",                                                                                                                                                                            "Ministry of Justice, UK",
                                                                  "INSTITUTO DE INVESTIGA\\u00c3\\u00e1\\u00c3fO AGRON\\u00c3\\u201cMICA - IIA",                                               "IIA",                                                                                                                                             "Instituto de Investiga\\u00e7\\u00e3o Agron\\u00f3mica",
                                                                                             "Gloucestershire Centre for Environmental Records",                                              "GCER",                                                                                                                                                   "Gloucestershire Centre for Environmental Records",
                                                                                                                        "IndiaBiodiversity.org",                             "IndiaBiodiversity.org",                                                                                                                                                                              "IndiaBiodiversity.org",
                                                                                                       "Sheffield and Rotherham Wildlife Trust",                                              "SRWT",                                                                                                                                                             "Sheffield and Rotherham Wildlife Trust",
                                                                                                           "Isle of Wight Local Records Centre",                                             "IWLRC",                                                                                                                                                                 "Isle of Wight Local Records Centre",
                                                                                                                                         "WINC",                                              "WINC",                                                                                                                                                                 "Waite Insect & Nematode Collection",
                                                                  "Western Australia, Department of Biodiversity, Conservation and Attractions",                                           "WA_DBCA",                                                                                                                        "Western Australia, Department of Biodiversity, Conservation and Attractions",
                                                                                                                              "Mohonk Preserve",                                              "DSRC",                                                                                                                                                                                    "Mohonk Preserve",
                                                                                   "Cambridgeshire & Peterborough Environmental Records Centre",                                             "CPERC",                                                                                                                                         "Cambridgeshire & Peterborough Environmental Records Centre"
                 )


  
  
  #### 3.0 Table ####
  ##### 3.1 occCount ####
    # Get a count of the number of occurrences per institutionCode
  occCount <- data %>%
      # Group by institutionCode
    dplyr::group_by(institutionCode) %>%
      # Get a tally of occurrences for each institutionCode
    dplyr::add_tally() %>%
      # Select only institutionCode and the tally and then keep one of each
    dplyr::select(c(institutionCode, n)) %>%
    dplyr::distinct(institutionCode, .keep_all = TRUE) %>%
    # Rename the count column
    dplyr::rename("occurrenceCount" = n)
  
  ##### 3.2 spCount ####
    # Get a count of the number of species per institutionCode
  spCount <- data %>%
    # Keep only distinct institutionCode, scientificName combinations 
    dplyr::distinct(institutionCode, scientificName,.keep_all = TRUE) %>%
    # Group by institutionCode
    dplyr::group_by(institutionCode) %>%
    # Get a tally of occurrences for each institutionCode
    dplyr::add_tally() %>%
    # Select only institutionCode and the tally and then keep one of each
    dplyr::select(c(institutionCode, n)) %>%
    dplyr::distinct(institutionCode, .keep_all = TRUE) %>%
      # Rename the count column
    dplyr::rename("speciesCount" = n)
  
  ##### 3.3 Merge ####
  counts <- occCount %>%
    dplyr::left_join(spCount, by = "institutionCode") %>%
    dplyr::arrange(occurrenceCount)
  
  
  #### 4.0 Save and return ####
  # If user provided an outPath then save the file
  if(!is.null(outPath)){
  readr::write_excel_csv(counts, 
                   paste(outPath, fileName, sep = "/"))
  }

  
  return(counts)
  
  
} # END dataProvTables
