#' @title main function of the package
#' @description
#' Main function of the package. Fits a range of models introduced in the package vignette \code{BayesSUR.pdf}. Returns an object of S3 class BayesSUR. 
#' There are three options for the prior on the residual covariance matrix (i.e., independent inverse-Gamma, inverse-Wishart and hyper-inverse Wishart) 
#' and three options for the prior on the latent indicator variable (i.e., independent Bernoulli, hotspot and Markov random field). 
#' So there are nine models in total. See details for their combinations.
#' 
#' @docType package
#' @useDynLib BayesSUR
#' @importFrom utils head tail read.table write.table
#' @importFrom Rcpp sourceCpp
#' @importFrom xml2 as_xml_document write_xml
#' 
#' @name BayesSUR
#' @param data a data frame if using \code{formula}. If not using \code{formula}, it is either a matrix/dataframe or the path to (a plain text) data file with variables on the columns and observations on the rows 
#' @param Y,X,X_0 vectors of indexes (with respect to the data matrix) for the outcomes, the covariates to select and the fixed covariates respectively if data is either a path to a file or a matrix;
#' if the 'data' argument is not provided, these needs to be matrices containing the data instead.
#' @param outFilePath path to where the output files are to be written. The default path is the currect working directory.
#' @param nIter number of iterations for the MCMC procedure
#' @param burnin number of iterations (or fraction of iterations) to discard at the start of the chain Default = 0
#' @param nChains number of parallel chains to run
#' @param covariancePrior string indicating the prior for the covariance $C$; it has to be either "HIW" for the hyper-inverse-Wishar (which will result in a sparse covariance matrix),
#' "IW" for the inverse-Wishart prior ( dense covariance ) or "IG" for independent inverse-Gamma on all the diagonal elements and 0 otherwise. See the details for the model specification
#' @param gammaPrior string indicating the gamma prior to use, either "hotspot" for the Hotspot prior of Bottolo (2011), "MRF" for the Markov Random Field prior or "hierarchical" for a simpler hierarchical prior. See the details for the model specification
#' @param gammaSampler string indicating the type of sampler for gamma, either "bandit" for the Thompson sampling inspired samper or "MC3" for the usual $MC^3$ sampler
#' @param gammaInit gamma initialisation to either all-zeros ("0"), all ones ("1"), randomly ("R") or (default) MLE-informed ("MLE").
#' @param mrfG either a matrix or a path to the file containing the G matrix for the MRF prior on gamma (if necessary)
#' @param standardize Logical flag for X variable standardization. Default is standardize=TRUE. The coefficients are returned on the standardized scale.
#' @param standardize.response Standardization for the response variables. Default is standardize.response=TRUE.
#' @param hyperpar a list of named hypeparameters to use instead of the default values. Valid names are mrf_d, mrf_e, a_sigma, b_sigma, a_tau, b_tau, nu, a_eta, b_eta, a_o, b_o, a_pi, b_pi, a_w and b_w. 
#' Their default values are a_w=2, b_w=5, a_o=p-2, b_o=0.005, a_pi=2 (hotspot) or 1 (hierarchical), b_pi=1 (hotspot) or s-1 (hierarchical), nu=s+2, a_tau=0.1, b_tau=10, a_eta=0.1, b_eta=1, a_sigma=1, b_sigma=1, mrf_d=-3 and mrf_e=0.001. See the vignette for more information.
#' @param output_gamma allow ( \code{TRUE} ) or suppress ( \code{FALSE} ) the output for  gamma. See the return value below for more information.
#' @param output_beta allow ( \code{TRUE} ) or suppress ( \code{FALSE} ) the output for beta. See the return value below for more information.
#' @param output_G allow ( \code{TRUE} ) or suppress ( \code{FALSE} ) the output for G. See the return value below for more information.
#' @param output_sigmaRho allow ( \code{TRUE} ) or suppress ( \code{FALSE} ) the output for sigmaRho. See the return value below for more information.
#' @param output_pi allow ( \code{TRUE} ) or suppress ( \code{FALSE} ) the output for pi. See the return value below for more information.
#' @param output_tail allow ( \code{TRUE} ) or suppress ( \code{FALSE} ) the output for tail (hotspot tail probability). See the return value below for more information.
#' @param output_model_size allow ( \code{TRUE} ) or suppress ( \code{FALSE} ) the output for model_size. See the return value below for more information.
#' @param output_CPO allow ( \code{TRUE} ) or suppress ( \code{FALSE} ) the output for *; possible outputs are gamma, G, beta, sigmaRho, pi, tail (hotspot tail probability), model_size, CPO. See the return value below for more information.
#' @param output_Y allow ( \code{TRUE} ) or suppress ( \code{FALSE} ) the output for responses dataset Y.
#' @param output_X allow ( \code{TRUE} ) or suppress ( \code{FALSE} ) the output for predictors dataset X.
#' @param tmpFolder the path to a temporary folder where intermediate data files are stored (will be erased at the end of the chain) default to local tmpFolder
#' 
#' @details The arguments \code{covariancePrior} and \code{gammaPrior} specify the model HRR, dSUR or SSUR with different gamma prior. Let \eqn{\gamma_{jk}} be latent indicator variable of each coefficient and \eqn{C} be covariance matrix of response variables.
#' The nine models specified through the arguments \code{covariancePrior} and \code{gammaPrior} are as follows.
#' \tabular{cccc}{
#'                 \tab \eqn{\gamma_{jk}}~Bernoulli \tab \eqn{\gamma_{jk}}~hotspot \tab \eqn{\gamma_{jk}}~MRF \cr
#'   \eqn{C}~indep \tab HRR-B                       \tab HRR-H                     \tab HRR-M           \cr
#'   \eqn{C}~IW    \tab dSUR-B                      \tab dSUR-H                    \tab dSUR-M          \cr
#'   \eqn{C}~HIW   \tab SSUR-B                      \tab SSUR-H                    \tab SSUR-M           
#' }
#'
#' @return An object of class "BayesSUR":
#' \itemize{
#' \item status - the running status
#' \item input - a list of all input parameters by the user
#' \item output - a list of the all output filenames: 
#' \itemize{
#' \item "\code{*_logP_out.txt}" - contains each row for the \eqn{1000t}-th iteration's log-likelihoods of parameters, i.e., Tau, Eta, JunctionTree, SigmaRho, O, Pi, Gamma, W, Beta and data conditional log-likelihood depending on the models.
#' \item "\code{*_gamma_out.txt}" - posterior mean of the latent indicator matrix. 
#' \item "\code{*_pi_out.txt}" - posterior mean of the predictor effects (prospensity) by decomposing the probability of the latent indicator.
#' \item "\code{*_hotspot_tail_p_out.txt}" - posterior mean of the hotspot tail probability. Only available for the hotspot prior on the gamma.
#' \item "\code{*_beta_out.txt}" - posterior mean of the coefficients matrix.
#' \item "\code{*_G_out.txt}" - posterior mean of the response graph. Only available for the HIW prior on the covariance. 
#' \item "\code{*_sigmaRho_out.txt}" - posterior mean of the transformed parameters. Not available for the IG prior on the covariance.
#' \item "\code{*_model_size.txt}" - contains each row for the\eqn{1000t}-th iteration's model sizes of the multiple response variables.
#' \item "\code{*_CPO_out.txt}" - the (scaled) conditional predictive ordinates (CPO). 
#' \item "\code{*_CPOsumy_out.txt}" - the (scaled) conditional predictive ordinates (CPO) with joint posterior predictive of the response variables.
#' \item "\code{*_WAIC_out.txt}" - the widely applicable information criterion (WAIC). 
#' \item "\code{*_Y.txt}" - responses dataset. 
#' \item "\code{*_X.txt}" - predictors dataset.
#' \item "\code{*_X0.txt}" - fixed predictors dataset.
#' }
#' \item call - the matched call.
#' }
#' 
#' @references Banterle M, Bottolo L, Richardson S, Ala-Korpela M, Jarvelin MR, Lewin A (2018). \emph{Sparse variable and covariance selection for high-dimensional seemingly unrelated Bayesian regression.} bioRxiv: 467019.
#' @references Banterle M#, Zhao Z#, Bottolo L, Richardson S, Lewin A\*, Zucknick M\* (2019). \emph{BayesSUR: An R package for high-dimensional multivariate Bayesian variable and covariance selection in linear regression.} URL: https://github.com/mbant/BayesSUR/tree/master/BayesSUR/vignettes/vignettes.pdf
#' 
#' @examples
#' data("example_eQTL", package = "BayesSUR")
#' hyperpar <- list( a_w = 2 , b_w = 5 )
#' 
#' fit <- BayesSUR(Y = example_eQTL[["blockList"]][[1]], 
#'                 X = example_eQTL[["blockList"]][[2]],
#'                 data = example_eQTL[["data"]], outFilePath = tempdir(),
#'                 nIter = 100, burnin = 50, nChains = 2, gammaPrior = "hotspot",
#'                 hyperpar = hyperpar, tmpFolder = "tmp/" )
#' 
#' ## check output
#' # show the summary information
#' summary(fit)
#' 
#' # show the estimated beta, gamma and graph of responeses Gy
#' plotEstimator(fit)
#' 
#' \donttest{
#' plotEstimator(fit, fig.tex = TRUE)
#' system(paste(getOption("pdfviewer"), "ParamEstimator.pdf"))
#' }
#' 
#' @export
BayesSUR <- function(Y, X, X_0 = NULL, data = NULL, 
                     outFilePath = "", nIter = 10000, burnin = 5000, nChains = 2, 
                     covariancePrior = "HIW", gammaPrior = "",
                     gammaSampler = "bandit", gammaInit = "MLE", mrfG = NULL,
                     standardize = TRUE, standardize.response = TRUE,
                     output_gamma = TRUE, output_beta = TRUE, output_G = TRUE, output_sigmaRho = TRUE,
                     output_pi = TRUE, output_tail = TRUE, output_model_size = TRUE, output_CPO = TRUE,
                     output_Y = TRUE, output_X = TRUE, hyperpar = list(), tmpFolder = "tmp/")
{
  
  # Check the directory for the output files
  if(outFilePath == "")
    stop("Warning: Please specify a directory to save all output files!")
  
  outFilePathLength = nchar(outFilePath)
  if( substr(outFilePath,outFilePathLength,outFilePathLength) != "/" )
    outFilePath = paste( outFilePath , "/" , sep="" )
  dir.create(outFilePath)
  
  # Create temporary directory
  tmpFolderLength = nchar(tmpFolder)
  if( substr(tmpFolder,tmpFolderLength,tmpFolderLength) != "/" )
    tmpFolder = paste( tmpFolder , "/" , sep="" )
  tmpFolder = paste(outFilePath, tmpFolder, sep="")
  dir.create(tmpFolder)
  

  ## Check the input: reasoning is that the user provides either
  # a data matrix or a data path-to-file
  #     - in this case Y, X (and X_0) need to be provided as vectors of indexes 
  # if the data matrix is not provided, the user will give 2/3 matrices for Y, X (and X_0)
  #     - in which case we write those in order into a new joint file
  # everything else throws an error
  
  # check the formula
  cl <- match.call()
  
  # we'll check in reverse order, is data NULL?
  if ( is.null( data ) )# | m[1] )
  {

    # Y,X (and if there X_0) need to be valid numeric matrices then
    # check Y and X have comfortable number of observations
    if( !is.numeric(Y) | is.null(dim(Y)) & !is.data.frame(Y) )
      my_stop("Y needs to be a valid matrix or data.frame with > 1 column", tmpFolder)
    
    nObservations = nrow(Y)
    if( !is.numeric(X) | is.null(dim(X)) | nrow(X) != nObservations & !is.data.frame(X) )
      my_stop("X needs to be a valid matrix or data.frame with >= 1 column and the same number of rows of Y", tmpFolder)
    
    if ( is.null ( X_0 ) ){
      X_0 = matrix(NA,nrow=nObservations,ncol=0)
    }else{
      if( !is.numeric(X_0) | is.null(dim(X_0)) | nrow(X_0) != nObservations & !is.data.frame(X_0) )
        my_stop("if provided, X_0 needs to be a valid matrix or data.frame with >= 1 column and the same number of rows of Y", tmpFolder)
    }
      
    #if( !is.numeric(X_0) | is.null(dim(X_0)) | nrow(X_0) != nObservations )
    #  my_stop("if provided, X_0 needs to be a valid matrix or data.frame with >= 1 column and the same number of rows of Y")

    # Standarize the data
    if( standardize ){
      X = scale(X)[,]
      X_0 = scale(X_0)[,]
    } 
    if( standardize.response ) Y = scale(Y)[,]

    # Write the three down in a single data file
    write.table(cbind(Y,X,X_0),paste(sep="",tmpFolder,"data.txt"), row.names = FALSE, col.names = FALSE)
    data = paste(sep="",tmpFolder,"data.txt")

    blockLabels = c( rep(0,ncol(Y)) , rep(1,ncol(X)) , rep(2,ncol(X_0)) )

    # Write the data in a output file
    write.table(Y,paste(sep="",outFilePath,"data_Y.txt"), row.names = FALSE, col.names = TRUE)
    write.table(X,paste(sep="",outFilePath,"data_X.txt"), row.names = FALSE, col.names = TRUE)
    write.table(X_0,paste(sep="",outFilePath,"data_X0.txt"), row.names = FALSE, col.names = TRUE)

  }else{ # data is not null, so the user wants to use it to input the data

    # is the data given as matrix?
    ## If it's valid matrix, simply write it and re-assign the variable data to hold its path
    if( (is.numeric(data) | is.data.frame(data)) & !is.null(dim(data))  )
    {
      # Standarize the data
      if( standardize ){
        data[,X]= scale(data[,X])[,]
        data[,X_0] = scale(data[,X_0])[,]
      } 
      if( standardize.response ) data[,Y] = scale(data[,Y])[,]
      
      # Write the Y and X data in a output file
      write.table(data[,Y],paste(sep="",outFilePath,"data_Y.txt"), row.names = FALSE, col.names = TRUE)
      write.table(data[,X],paste(sep="",outFilePath,"data_X.txt"), row.names = FALSE, col.names = TRUE)
      write.table(data[,X_0],paste(sep="",outFilePath,"data_X0.txt"), row.names = FALSE, col.names = TRUE)
      
      write.table(data,paste(sep="",tmpFolder,"data.txt"), row.names = FALSE, col.names = FALSE)
      data = paste(sep="",tmpFolder,"data.txt")
    }

    # is the data given as a string?
    if( is.character(data) & length(data) == 1 )
    {    
      if( substr(data,1,1) == "~" )
        data = path.expand(data)

      # now try and read from given data file
      if( !file.exists( data ) )
        my_stop("Input file doesn't exists!",tmpFolder)
    }

    ## at this point data contains the path to a file that exists
    # try and read one line to check dimensions
    dataHeader = read.table(data,header = FALSE,nrows = 1)
    nVariables = ncol(dataHeader)

    ## Y, X (and X_0) should be some fixed variables that needs to be included in the model
    if ( is.null(X_0) )
      X_0 = c()

    # be sure they can be converted to numeric
    Y = as.numeric(Y)
    X = as.numeric(X)
    X_0 = as.numeric(X_0)

    # be sure that they are vectors
    if ( !( is.vector(Y,"numeric") & is.vector(X,"numeric") & is.vector(X_0,"numeric") ) )
      my_stop("When the `data` argument is set, Y,X and X_0 need to be corresponding index vectors", tmpFolder)

    # check thay do not overlap
    if ( length( c( intersect(Y,X) , intersect(Y,X_0) , intersect(X_0,X) ) ) != 0 )
      my_stop("Y,X and X_0 need to be distinct index vectors", tmpFolder)

    # check if dimensions correspond -- higher dimensions gets an error
    if( length( c(Y,X,X_0) ) > nVariables )
      my_stop("When the `data` argument is set, Y,X and X_0 need to be corresponding index vectors", tmpFolder)
    # equal dimensions are ok, but lower dimensions means some columns of the data will be disregarded ( set to -1  )

    # We can now init the blockList
    blockLabels = rep(NA,nVariables)
    blockLabels[Y] = 0
    blockLabels[X] = 1
    if ( length ( X_0 ) > 0 ) 
      blockLabels[X_0] = 2

    blockLabels[ is.na ( blockLabels ) ] = -1

  }

  ## Then init the structure graph
  # Consider that the indexes are written so that Y is 0 , X is 1 and (if there) X_0 is 2
  if ( length ( X_0 ) > 0 ){
    structureGraph = structureGraph = matrix(c(0,0,0,1,0,0,2,0,0),3,3,byrow=TRUE)
  }else structureGraph = structureGraph = matrix(c(0,0,1,0),2,2,byrow=TRUE)

  ## Finally write blockLabels and structureGraph to a file
  write.table(blockLabels,paste(sep="",tmpFolder,"blockLabels.txt"), row.names = FALSE, col.names = FALSE)
  blockList = paste(sep="",tmpFolder,"blockLabels.txt")
  write.table(structureGraph, paste(sep="",tmpFolder,"structureGraph.txt"), row.names = FALSE, col.names = FALSE)
  structureGraph = paste(sep="",tmpFolder,"structureGraph.txt")


  # cleanup file PATHS
  dataLength = nchar(data)
  if( dataLength == 0 )
    my_stop("Please provide a correct path to a plain-text (.txt) file", tmpFolder)
  
  # magicly strip '/' from the start and '.txt' from the end of the data file name
  dataString = head( strsplit( tail( strsplit(data,split = c("/"))[[1]] , 1 ) , ".txt" )[[1]] , 1 ) 


  # check how burnin was given
  if ( burnin < 0 ){
    my_stop("Burnin must be positive or 0",tmpFolder)
  }else{ if ( burnin > nIter ){
    my_stop("Burnin might not be greater than nIter",tmpFolder)
  }else{ if ( burnin < 1 ){ # given as a fraction
    burnin = ceiling(nIter * burnin) # the zero case is taken into account here as well
  }}} # else assume is given as an absolute number

  ###############################
  # prepare the print of hyperparameters corresponding the specified model
  hyperpar.all <- list(a_w=2, b_w=5, a_o=sum(blockLabels==1)-2, b_o=0.005, a_pi=NA, b_pi=NA, nu=sum(blockLabels==0)+2, a_tau=0.1, b_tau=10, a_eta=0.1, b_eta=1, a_sigma=1, b_sigma=1, mrf_d=-3, mrf_e=0.001)
  if(toupper(gammaPrior) %in% c("HOTSPOT", "HOTSPOTS", "HS")){
    hyperpar.all$a_pi <- 2
    hyperpar.all$b_pi <- 1
    
    if(toupper(covariancePrior) %in% c("INDEPENDENT", "INDEP", "IG"))
      hyperpar.all <- hyperpar.all[-c(7:11,14:15)]
    if(toupper(covariancePrior) %in% c("DENSE", "IW"))
      hyperpar.all <- hyperpar.all[-c(10:11,12:15)]
    if(toupper(covariancePrior) %in% c("SPARSE", "HIW"))
      hyperpar.all <- hyperpar.all[-c(12:15)]
  }
  if( toupper(gammaPrior) %in% c("HIERARCHICAL", "H")){
    #hyperpar.all$a_pi <- 1
    #hyperpar.all$b_pi <- sum(blockLabels==0) - 1
    
    if(toupper(covariancePrior) %in% c("INDEPENDENT", "INDEP", "IG"))
      hyperpar.all <- hyperpar.all[-c(3:6,7:11,14:15)]
    if(toupper(covariancePrior) %in% c("DENSE", "IW"))
      hyperpar.all <- hyperpar.all[-c(3:6,10:11,12:15)]
    if(toupper(covariancePrior) %in% c("SPARSE", "HIW"))
      hyperpar.all <- hyperpar.all[-c(3:6,12:15)]
  }
  if( toupper(gammaPrior) %in% c("MRF", "MARKOV RANDOM FIELD")){
    
    if(toupper(covariancePrior) %in% c("INDEPENDENT", "INDEP", "IG"))
      hyperpar.all <- hyperpar.all[-c(3:6,7:13)]
    if(toupper(covariancePrior) %in% c("DENSE", "IW"))
      hyperpar.all <- hyperpar.all[-c(3:6,10:13)]
    if(toupper(covariancePrior) %in% c("SPARSE", "HIW"))
      hyperpar.all <- hyperpar.all[-c(3:6,12:13)]
  }
  
  if(length(hyperpar)>0)
    for( i in 1:length(hyperpar)){
      if(names(hyperpar)[[i]] %in% names(hyperpar.all))
        hyperpar.all[[which(names(hyperpar.all)==names(hyperpar)[[i]])]] <- hyperpar[[i]]
    }
  
  # method to use
  if ( toupper(covariancePrior) %in% c("SPARSE", "HIW") ){
    covariancePrior <- "HIW"
  }else if ( toupper(covariancePrior) %in% c("DENSE", "IW") ){
    covariancePrior <- "IW"
  }else if ( toupper(covariancePrior) %in% c("INDEPENDENT", "INDEP", "IG") ){
    covariancePrior <- "IG"
  }else
    my_stop("Unknown covariancePrior argument: only sparse (HIW), dense(IW) or independent (IG) are available",tmpFolder)
  
  # mrfG and gammaPrior
  if( gammaPrior == "" )
	{
		if ( is.null(mrfG) )
		{
			cat( "Using default prior for Gamma - hotspot prior\n")
			#mrfG=""
			gammaPrior = "hotspot"
		}
		else
		{
			cat( "No value for gammaPrior was specified, but mrfG was given - choosing MRF prior\n")
			gammaPrior = "MRF"
		}
    
	}else{

    if ( toupper(gammaPrior) %in% c("HOTSPOT", "HOTSPOTS", "HS") )
      gammaPrior = "hotspot"
    else if ( toupper(gammaPrior) %in% c("MRF", "MARKOV RANDOM FIELD") ) 
      gammaPrior = "MRF"
    else if ( toupper(gammaPrior) %in% c("HIERARCHICAL", "H") ) 
      gammaPrior = "hierarchical"
    else
      my_stop("Unknown gammaPrior argument: only hotspot, MRF or hierarchical are available",tmpFolder)
  }

  # if mrfG is not a string
  if( !(is.character(mrfG) & length(mrfG) == 1) )
  {
    # if it's a matrix
    if( (is.numeric(mrfG)  | is.data.frame(mrfG)) & !is.null(dim(mrfG))  )
    {
      write.table(mrfG,paste(sep="",tmpFolder,"mrfG.txt"), row.names = FALSE, col.names = FALSE)
      mrfG = paste(sep="",tmpFolder,"mrfG.txt")
    }else if( is.null( mrfG )){
      # save a meaningless mrfG.txt file to pass the parameter to C++ 
      mrfG = matrix(c(0,0),ncol=2)
      write.table(mrfG, paste(sep="",tmpFolder,"mrfG.txt"), row.names = FALSE, col.names = FALSE)
      mrfG = paste(sep="",tmpFolder,"mrfG.txt")
    }else
      my_stop("Unknown mrfG argument: check the help function for possibile values",tmpFolder)
  }
  
  ## Set up the XML file for hyperparameters
  xml  = as_xml_document(
    list( hyperparameters = list(
      lapply(hyperpar,function(x) list(x)) # every element in the list should be a list
    )))
  hyperParFile = paste(sep="",tmpFolder,"hyperpar.xml")
  write_xml(xml,file = hyperParFile)
  
  ## Create the return object
  ret = list( status=1, input=list(), output = list() )
  class(ret) = "BayesSUR"
  
  # Copy the inputs
  ret$input["nIter"] = nIter
  ret$input["burnin"] = burnin
  ret$input["nChains"] = nChains 
  ret$input["covariancePrior"] = covariancePrior
  ret$input["gammaPrior"] = gammaPrior
  ret$input["gammaSampler"] = gammaSampler
  ret$input["gammaInit"] = gammaInit
  ret$input["mrfG"] = mrfG
  
  ret$input$hyperParameters = hyperpar.all

  methodString = 
    switch( covariancePrior,
      "HIW" = "SSUR" ,
      "IW"  = "dSUR" ,
      "IG"  = "HRR" )

  ret$call = cl
  
  # Prepare path to outputs
  ret$output["outFilePath"] = outFilePath
  
  ret$output["logP"] = paste(sep="", dataString , "_",  methodString , "_logP_out.txt")

  if ( output_gamma )
    ret$output["gamma"] = paste(sep="", dataString , "_",  methodString , "_gamma_out.txt")
  
  if( gammaPrior %in% c("hierarchical","hotspot") & output_pi )
    ret$output["pi"] = paste(sep="", dataString , "_",  methodString , "_pi_out.txt")

  if( gammaPrior == "hotspot" & output_tail )
    ret$output["tail"] = paste(sep="", dataString , "_",  methodString , "_hotspot_tail_p_out.txt")
  
  if ( output_beta )
    ret$output["beta"] = paste(sep="", dataString , "_",  methodString , "_beta_out.txt")
  
  if ( covariancePrior == "HIW" & output_G )
    ret$output["G"] = paste(sep="", dataString , "_",  methodString , "_G_out.txt")
    
  if ( covariancePrior %in% c("HIW","IW") & output_sigmaRho )
    ret$output["sigmaRho"] = paste(sep="", dataString , "_",  methodString , "_sigmaRho_out.txt")

  if ( output_model_size )
    ret$output["model_size"] = paste(sep="", dataString , "_",  methodString , "_model_size_out.txt")
  
  if ( output_CPO ){
    ret$output["CPO"] = paste(sep="", dataString , "_",  methodString , "_CPO_out.txt")
    ret$output["CPOsumy"] = paste(sep="", dataString , "_",  methodString , "_CPOsumy_out.txt")
    ret$output["WAIC"] = paste(sep="", dataString , "_",  methodString , "_WAIC_out.txt")
  }
  
  if ( output_Y )
    ret$output["Y"] = paste(sep="", "data_Y.txt")
  
  if ( output_X )
    ret$output["X"] = paste(sep="", "data_X.txt")

  betaPrior="independent"
  ret$status = BayesSUR_internal(data, mrfG, blockList, structureGraph, hyperParFile, outFilePath, 
            nIter, burnin, nChains, 
            covariancePrior, gammaPrior, gammaSampler, gammaInit, betaPrior,
            output_gamma, output_beta, output_G, output_sigmaRho, output_pi, output_tail, output_model_size, output_CPO)

  if(outFilePath != tmpFolder)
    unlink(tmpFolder,recursive = TRUE)
   
  class(print) <- c(class(print), "BayesSUR")
  class(summary) <- c(class(summary), "BayesSUR")
  class(plot) <- c(class(plot), "BayesSUR")
  class(fitted) <- c(class(fitted), "BayesSUR")
  class(predict) <- c(class(predict), "BayesSUR")
  class(coef) <- c(class(coef), "BayesSUR")
  return(ret)
}


my_stop = function( msg , tmpFolder )
{
  unlink(tmpFolder,recursive = TRUE)
  stop(msg)
}

