% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/irf.R
\name{irf}
\alias{irf}
\alias{irf.bgvar}
\title{Impulse Response Function}
\usage{
irf(x, n.ahead=24, ident="chol", shockinfo=NULL, quantiles=NULL, 
    expert=NULL, verbose=TRUE)
}
\arguments{
\item{x}{Object of class \code{bgvar}.}

\item{n.ahead}{Forecasting horizon.}

\item{ident}{Character defining type of identification scheme. Default \code{chol} refers to zero restrictions via the Cholesky decomposition while \code{sign} refers to sign-restrictions and \code{girf} to generalized impulse responses.  Details of preferred shock are given via \code{shockinfo}.}

\item{shockinfo}{Dataframe with additional information about the nature of shocks. Depending on the \code{ident} argument, the dataframe has to be specified differently. In order to get a dummy version for each identification scheme use \code{\link{get_shockinfo}}.}

\item{quantiles}{Numeric vector with posterior quantiles. Default is set to compute median along with 68\%/80\%/90\% confidence intervals.}

\item{expert}{Expert settings, must be provided as list. Default is set to \code{NULL}.\itemize{
\item{\code{MaxTries}}{ Numeric specifying maximal number of tries for finding a rotation matrix with sign-restrictions. Attention: setting this number very large may results in very long computational times.}
\item{\code{save.store}}{ If set to \code{TRUE} the full posterior of both, impulses responses and rotation matrices, are returned. Default is set to \code{FALSE} in order to save storage.}
\item{\code{use_R}}{ Boolean whether IRF computation should fall back on \code{R} version, otherwise \code{Rcpp} version is used.}
\item{\code{applyfun}}{ In case \code{use_R=TRUE}, this allows for user-specific apply function, which has to have the same interface than \code{lapply}. If \code{cores=NULL} then \code{lapply} is used, if set to a numeric either \code{parallel::parLapply()} is used on Windows platforms and \code{parallel::mclapply()} on non-Windows platforms.}
\item{\code{cores}}{ Numeric specifying the number of cores which should be used, also \code{all} and \code{half} is possible. By default only one core is used.}
}}

\item{verbose}{If set to \code{FALSE} it suppresses printing messages to the console.}
}
\value{
Returns a list of class \code{bgvar.irf} with the following elements: \itemize{
\item{\code{posterior}}{ Four-dimensional array (K times n.ahead times number of shocks times Q) that contains Q quantiles of the posterior distribution of the impulse response functions.}
\item{\code{ident}}{ Character showing the chosen identification scheme.}
\item{\code{shockinfo}}{ Dataframe with details on identification specification.}
\item{\code{rot.nr}}{ In case identification is based on sign restrictions (i.e., \code{ident="sign"}), this provides the number of rotation matrices found for the number of posterior draws (save*save_thin).}
\item{\code{struc.obj}}{ List object that contains posterior quantitites needed when calculating historical decomposition and structural errors via \code{hd.decomp}.\itemize{
\item{\code{A}}{ Median posterior of global coefficient matrix.}
\item{\code{Ginv}}{ Median posterior of matrix \code{Ginv}, which describes contemporaneous relationships between countries.}
\item{\code{S}}{ Posterior median of matrix with country variance-covariance matrices on the main diagonal.}
\item{\code{Rmed}}{ Posterior rotation matrix if \code{ident="sign"}.}
}}
\item{\code{model.obj}}{ List object that contains model-specific information, in particular\itemize{
\item{\code{xglobal}}{ Data of the model.}
\item{\code{plag}}{ Lag specification of the model.}
}}
\item{\code{IRF_store}}{ Four-dimensional array (K times n.ahead times number of shock times draws) which stores the whole posterior distribution. Exists only if \code{save.store=TRUE}.}
\item{\code{R_store}}{ Three-dimensional array (K times K times draws) which stores all rotation matrices. Exists only if \code{save.store=TRUE}.}
}
}
\description{
This function calculates three alternative ways of dynamic responses, namely generalized impulse response functions (GIRFs) as in Pesaran and Shin (1998), orthogonalized impulse response functions using a Cholesky decomposition and finally impulse response functions given a set of user-specified sign restrictions.
}
\examples{
oldpar <- par(no.readonly = TRUE)
# First example, a US monetary policy shock, quarterly data
library(BGVAR)
data(eerDatasmall)
# US monetary policy shock
model.eer<-bgvar(Data=eerDatasmall, W=W.trade0012.small, draws=100, burnin=100, 
                 plag=1, prior="SSVS", eigen=TRUE)

# define shock
shockinfo <- get_shockinfo("chol")
shockinfo$shock <- "US.stir"; shockinfo$scale <- -100

# generalized impulse responses
irf.girf.us.mp<-irf(model.eer, n.ahead=24, ident="girf", shockinfo=shockinfo)

# cholesky identification
irf.chol.us.mp<-irf(model.eer, n.ahead=24, ident="chol", shockinfo=shockinfo)

# sign restrictions
shockinfo <- get_shockinfo("sign")
shockinfo <- add_shockinfo(shockinfo, shock="US.stir", restriction=c("US.y","US.Dp"), 
sign=c("<","<"), horizon=c(1,1), scale=1, prob=1)
irf.sign.us.mp<-irf(model.eer, n.ahead=24, ident="sign", shockinfo=shockinfo)

# sign restrictions with relaxed cross-country restrictions
shockinfo <- get_shockinfo("sign")
# restriction for other countries holds to 75\\%
shockinfo <- add_shockinfo(shockinfo, shock="US.stir", restriction=c("US.y","EA.y","UK.y"), 
                           sign=c("<","<","<"), horizon=1, scale=1, prob=c(1,0.75,0.75))
shockinfo <- add_shockinfo(shockinfo, shock="US.stir", restriction=c("US.Dp","EA.Dp","UK.Dp"),
                           sign=c("<","<","<"), horizon=1, scale=1, prob=c(1,0.75,0.75))
irf.sign.us.mp<-irf(model.eer, n.ahead=24, ident="sign", shockinfo=shockinfo)
\donttest{
# Example with zero restriction (Arias et al., 2018) and 
# rationality conditions (D'Amico and King, 2017).
data("eerDataspf")
model.eer<-bgvar(Data=eerDataspf, W=W.trade0012.spf, draws=300, burnin=300,
                 plag=1, prior="SSVS", eigen=TRUE)
shockinfo <- get_shockinfo("sign")
shockinfo <- add_shockinfo(shockinfo, shock="US.stir_t+4", 
                           restriction=c("US.Dp_t+4","US.stir","US.y_t+4"),
                           sign=c("<","0","<"), horizon=1, prob=1, scale=1)
# rationality condition: US.stir_t+4 on impact is equal to average of 
# IRF of US.stir between horizon 1 to 4
shockinfo <- add_shockinfo(shockinfo, shock="US.stir_t+4", restriction="US.stir_t+4",
                           sign="ratio.avg", horizon=5, prob=1, scale=1)
# rationality condition: US.Dp_t+4 on impact is equal to IRF of US.Dp at horizon 4
shockinfo <- add_shockinfo(shockinfo, shock="US.stir_t+4", restriction="US.Dp_t+4",
                           sign="ratio.H", horizon=5, prob=1, scale=1)
# rationality condition: US.y_t+4 on impact is equal to IRF of US.y at horizon 4
shockinfo <- add_shockinfo(shockinfo, shock="US.stir_t+4", restriction="US.y_t+4",
                           sign="ratio.H", horizon=5, prob=1, scale=1)
# regulate maximum number of tries with expert settings
irf.ratio <- irf(model.eer, n.ahead=20, ident="sign", shockinfo=shockinfo,
                 expert=list(MaxTries=10))
}
}
\references{
Arias, J.E., Rubio-Ramirez, J.F, and D.F. Waggoner (2018) \emph{Inference Based on SVARs Identified with Sign and Zero Restrictions: Theory and Applications.} Econometrica Vol. 86(2), pp. 685-720.

D'Amico, S. and T. B. King (2017) \emph{What Does Anticipated Monetary Policy Do?} Federal Reserve Bank of Chicago Working paper series, Nr. 2015-10.

Pesaran, H.M. and Y. Shin (1998) \emph{Generalized impulse response analysis in linear multivariate models.} Economics Letters, Volume 58, Issue 1, p. 17-29.
}
\seealso{
\code{\link{bgvar}}, \code{\link{get_shockinfo}}, \code{\link{add_shockinfo}}
}
\author{
Maximilian Boeck, Martin Feldkircher, Florian Huber
}
