% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/pat_createAirSensor.R
\name{pat_createAirSensor}
\alias{pat_createAirSensor}
\title{Create an Air Sensor object}
\usage{
pat_createAirSensor(
  pat = NULL,
  parameter = "pm25",
  FUN = PurpleAirQC_hourly_AB_01,
  ...
)
}
\arguments{
\item{pat}{PurpleAir Timeseries \emph{pat} object.}

\item{parameter}{Parameter for which to create an univariate \emph{airsensor} 
object. See details.}

\item{FUN}{Algorithm applied to \emph{pat} object for hourly aggregation and 
quality control. See details.}

\item{...}{(optional) Additional parameters passed into \code{FUN}.}
}
\value{
An "airsensor" object of aggregated PurpleAir Timeseries data.
}
\description{
Converts data from a \emph{pat} object with an irregular time 
axis to an \emph{airsensor} object where the numeric data has been aggregated 
along a standardized hourly time axis, as well as adding additional required 
metadata for compatibility with the *PWFSLSmoke* package.
}
\details{
\code{FUN} allows users to provide custom aggregation and 
quality-control functions that are used to create an \emph{airsensor} object. 
The \code{FUN} must accept a \emph{pat} object as the first argument and 
return a dataframe with a regular hourly datetime axis. \code{FUN} 
can access and utilize any component of a standard \emph{pat} object 
(e.g pm25_A, temperature, etc.) as well as define new variables in the 
\emph{pat} data. See examples. 

\code{parameter} allows user to select which variable to use for the 
univariate \emph{airsensor} object (e.g 'pm25_A', 'humidity', etc.). 
Furthermore the \code{parameter} can be a new variable created via \code{FUN} 
evaluation. See examples.

Additional named parameters can be be passed to \code{FUN} through \code{...}.
}
\examples{
\donttest{
library(AirSensor)

# Default FUN = PurpleAirQC_hourly_AB_00
sensor <- pat_createAirSensor(example_pat)

PWFSLSmoke::monitor_timeseriesPlot(sensor, shadedNight = TRUE)

# Try out other package QC functions
example_pat \%>\%
  pat_createAirSensor(FUN = PurpleAirQC_hourly_AB_01) \%>\%
  PWFSLSmoke::monitor_timeseriesPlot(shadedNight = TRUE)
  
example_pat \%>\%
  pat_createAirSensor(FUN = PurpleAirQC_hourly_AB_01) \%>\%
  PWFSLSmoke::monitor_timeseriesPlot(shadedNight = TRUE)
  
# Custom FUN
humidity_correction <- function(pat, z = 0) {

  # Default hourly aggregation
  hourlyData <- 
    pat \%>\%
    pat_aggregate() \%>\%
    pat_extractData()
    
  # Create custom_pm variable 
  pm25 <- (hourlyData$pm25_A + hourlyData$pm25_B) / 2
  hum <- hourlyData$humidity
  temp <- hourlyData$temperature
  hourlyData$custom_pm <- pm25 - (pm25 * hum * z)
    
  return(hourlyData)
  
} 

# Evaluate custom FUN 
sensor <- pat_createAirSensor(
  example_pat, 
  parameter = "custom_pm", 
  FUN = humidity_correction,
  z = .005
)

PWFSLSmoke::monitor_timeseriesPlot(sensor, shadedNight = TRUE)
}

}
\seealso{
\link{PurpleAirQC_hourly_AB_01}

\link{pat_aggregate}
}
