# AlgeriAPIs - Access Algerian Data via Public APIs
# Version 0.1.0
# Copyright (c) 2025 Renzo Caceres Rossi
# Licensed under the MIT License.
# See the LICENSE file in the root directory for full license text.

# get_algeria_literacy_rate


library(testthat)

test_that("get_algeria_literacy_rate() returns a tibble with the correct structure and content", {
  skip_on_cran()  # Skip test on CRAN if the API is unavailable

  result <- get_algeria_literacy_rate()

  # Check that the result is not NULL
  expect_false(is.null(result))

  # Check that the result is a data.frame/tibble
  expect_s3_class(result, "data.frame")

  # Check that the column names are exactly as expected
  expect_named(result, c("indicator", "country", "year", "value"))

  # Check data types of each column
  expect_type(result$indicator, "character")
  expect_type(result$country, "character")
  expect_type(result$year, "integer")
  expect_type(result$value, "double")

  # Check that the indicator column contains only the expected value
  expect_true(all(result$indicator == "Literacy rate, adult total (% of people ages 15 and above)"))

  # Check that the country column contains only "Algeria"
  expect_true(all(result$country == "Algeria"))

  # Check that the year range is correct
  expect_true(all(result$year >= 2010 & result$year <= 2022))

  # Check that the number of rows is 13 (2010–2022)
  expect_equal(nrow(result), 13)

  # Check that the number of columns is 4
  expect_equal(ncol(result), 4)

  # Check that all non-NA values are positive numbers
  non_na_values <- result$value[!is.na(result$value)]
  if (length(non_na_values) > 0) {
    expect_true(all(non_na_values > 0))
  }

  # Check that all non-NA values are within valid percentage range (0-100)
  if (length(non_na_values) > 0) {
    expect_true(all(non_na_values >= 0 & non_na_values <= 100))
  }
})
