% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/prob-classification_cost.R
\name{classification_cost}
\alias{classification_cost}
\alias{classification_cost.data.frame}
\alias{classification_cost_vec}
\title{Costs function for poor classification}
\usage{
classification_cost(data, ...)

\method{classification_cost}{data.frame}(
  data,
  truth,
  ...,
  costs = NULL,
  na_rm = TRUE,
  event_level = yardstick_event_level(),
  case_weights = NULL
)

classification_cost_vec(
  truth,
  estimate,
  costs = NULL,
  na_rm = TRUE,
  event_level = yardstick_event_level(),
  case_weights = NULL,
  ...
)
}
\arguments{
\item{data}{A \code{data.frame} containing the columns specified by \code{truth} and
\code{...}.}

\item{...}{A set of unquoted column names or one or more
\code{dplyr} selector functions to choose which variables contain the
class probabilities. If \code{truth} is binary, only 1 column should be selected.
Otherwise, there should be as many columns as factor levels of \code{truth}.}

\item{truth}{The column identifier for the true class results
(that is a \code{factor}). This should be an unquoted column name although
this argument is passed by expression and supports
\link[rlang:topic-inject]{quasiquotation} (you can unquote column
names). For \verb{_vec()} functions, a \code{factor} vector.}

\item{costs}{A data frame with columns \code{"truth"}, \code{"estimate"}, and \code{"cost"}.

\code{"truth"} and \code{"estimate"} should be character columns containing unique
combinations of the levels of the \code{truth} factor.

\code{"costs"} should be a numeric column representing the cost that should
be applied when the \code{"estimate"} is predicted, but the true result is
\code{"truth"}.

It is often the case that when \code{"truth" == "estimate"}, the cost is zero
(no penalty for correct predictions).

If any combinations of the levels of \code{truth} are missing, their costs are
assumed to be zero.

If \code{NULL}, equal costs are used, applying a cost of \code{0} to correct
predictions, and a cost of \code{1} to incorrect predictions.}

\item{na_rm}{A \code{logical} value indicating whether \code{NA}
values should be stripped before the computation proceeds.}

\item{event_level}{A single string. Either \code{"first"} or \code{"second"} to specify
which level of \code{truth} to consider as the "event". This argument is only
applicable when \code{estimator = "binary"}. The default uses an
internal helper that generally defaults to \code{"first"}, however, if the
deprecated global option \code{yardstick.event_first} is set, that will be
used instead with a warning.}

\item{case_weights}{The optional column identifier for case weights.
This should be an unquoted column name that evaluates to a numeric column
in \code{data}. For \verb{_vec()} functions, a numeric vector.}

\item{estimate}{If \code{truth} is binary, a numeric vector of class probabilities
corresponding to the "relevant" class. Otherwise, a matrix with as many
columns as factor levels of \code{truth}. \emph{It is assumed that these are in the
same order as the levels of \code{truth}.}}
}
\value{
A \code{tibble} with columns \code{.metric}, \code{.estimator},
and \code{.estimate} and 1 row of values.

For grouped data frames, the number of rows returned will be the same as
the number of groups.

For \code{class_cost_vec()}, a single \code{numeric} value (or \code{NA}).
}
\description{
\code{classification_cost()} calculates the cost of a poor prediction based on
user-defined costs. The costs are multiplied by the estimated class
probabilities and the mean cost is returned.
}
\details{
As an example, suppose that there are three classes: \code{"A"}, \code{"B"}, and \code{"C"}.
Suppose there is a truly \code{"A"} observation with class probabilities \code{A = 0.3 / B = 0.3 / C = 0.4}. Suppose that, when the true result is class \code{"A"}, the
costs for each class were \code{A = 0 / B = 5 / C = 10}, penalizing the
probability of incorrectly predicting \code{"C"} more than predicting \code{"B"}. The
cost for this prediction would be \code{0.3 * 0 + 0.3 * 5 + 0.4 * 10}. This
calculation is done for each sample and the individual costs are averaged.
}
\examples{
library(dplyr)

# ---------------------------------------------------------------------------
# Two class example
data(two_class_example)

# Assuming `Class1` is our "event", this penalizes false positives heavily
costs1 <- tribble(
  ~truth,   ~estimate, ~cost,
  "Class1", "Class2",  1,
  "Class2", "Class1",  2
)

# Assuming `Class1` is our "event", this penalizes false negatives heavily
costs2 <- tribble(
  ~truth,   ~estimate, ~cost,
  "Class1", "Class2",  2,
  "Class2", "Class1",  1
)

classification_cost(two_class_example, truth, Class1, costs = costs1)

classification_cost(two_class_example, truth, Class1, costs = costs2)

# ---------------------------------------------------------------------------
# Multiclass
data(hpc_cv)

# Define cost matrix from Kuhn and Johnson (2013)
hpc_costs <- tribble(
   ~estimate, ~truth, ~cost,
   "VF",   "VF",     0,
   "VF",    "F",     1,
   "VF",    "M",     5,
   "VF",    "L",    10,
   "F",    "VF",     1,
   "F",     "F",     0,
   "F",     "M",     5,
   "F",     "L",     5,
   "M",    "VF",     1,
   "M",     "F",     1,
   "M",     "M",     0,
   "M",     "L",     1,
   "L",    "VF",     1,
   "L",     "F",     1,
   "L",     "M",     1,
   "L",     "L",     0
)

# You can use the col1:colN tidyselect syntax
hpc_cv \%>\%
  filter(Resample == "Fold01") \%>\%
  classification_cost(obs, VF:L, costs = hpc_costs)

# Groups are respected
hpc_cv \%>\%
  group_by(Resample) \%>\%
  classification_cost(obs, VF:L, costs = hpc_costs)
}
\seealso{
Other class probability metrics: 
\code{\link{average_precision}()},
\code{\link{gain_capture}()},
\code{\link{mn_log_loss}()},
\code{\link{pr_auc}()},
\code{\link{roc_auc}()},
\code{\link{roc_aunp}()},
\code{\link{roc_aunu}()}
}
\author{
Max Kuhn
}
\concept{class probability metrics}
