% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/cross_correlation.R
\name{cross_correlation}
\alias{cross_correlation}
\title{Time-frequency cross-correlation}
\usage{
cross_correlation(
  X = NULL,
  wl = 512,
  bp = "pairwise.freq.range",
  ovlp = 70,
  wn = "hanning",
  cor.method = "pearson",
  parallel = 1,
  path = NULL,
  pb = TRUE,
  na.rm = FALSE,
  output = "cor.mat",
  templates = NULL,
  surveys = NULL,
  compare.matrix = NULL,
  type = "fourier",
  nbands = 40,
  method = 1
)
}
\arguments{
\item{X}{'selection_table', 'extended_selection_table' or data frame containing columns for sound files (sound.files),
selection number (selec), and start and end time of signal (start and end).
All selections must have the same sampling rate.}

\item{wl}{A numeric vector of length 1 specifying the window length of the spectrogram, default
is 512.}

\item{bp}{A numeric vector of length 2 for the lower and upper limits of a
frequency bandpass filter (in kHz). If columns for bottom and top frequency ('bottom.freq' and 'top.freq') are supplied "pairwise.freq.range" ca be used (default). If so, the lowest values in 'bottom.freq'
and the highest values in 'top.freq' for the selections involved in a pairwise comparison will be used as bandpass limits.}

\item{ovlp}{Numeric vector of length 1 specifying \% of overlap between two
consecutive windows, as in \code{\link[seewave]{spectro}}. Default is 70. High values of ovlp
slow down the function but produce more accurate results.}

\item{wn}{A character vector of length 1 specifying the window name as in \code{\link[seewave]{ftwindow}}.}

\item{cor.method}{A character vector of length 1 specifying the correlation method as in \code{\link[stats]{cor}}.}

\item{parallel}{Numeric. Controls whether parallel computing is applied.
It specifies the number of cores to be used. Default is 1 (i.e. no parallel computing).}

\item{path}{Character string containing the directory path where the sound files are located.
If \code{NULL} (default) then the current working directory is used.}

\item{pb}{Logical argument to control progress bar. Default is \code{TRUE}.}

\item{na.rm}{Logical. If \code{TRUE} all NAs produced when pairwise cross-correlations failed are removed from the
results. This means that all selections with at least 1 cross-correlation that failed are excluded.}

\item{output}{Character vector of length 1 to determine if only the correlation matrix is returned ('cor.mat', default) or a list ('list') containing 1) the correlation matrix and 2) a data frame with correlation values at each sliding step for each comparison. The list, which is also of class 'xcorr.output', can be used to graphically explore detections using \code{\link{full_spectrograms}}.}

\item{templates}{Character vector with the selections in 'X' to be used as templates for cross-correlation detection. To refer to specific selections in 'X' the user must use the format "sound.file-selec" (e.g. "file1.wav-1"). If only the sound file name is included then the entire sound file is used as template.}

\item{surveys}{Character vector with the selections in 'X' to be used as surveys for cross-correlation detection. To refer to specific selections in 'X' the user must use the format "sound.file-selec" (e.g. "file1.wav-1"). If only the sound file name is included then the entire sound file is used as survey.}

\item{compare.matrix}{A character matrix with 2 columns indicating the selections to be compared (column 1 vs column 2). The columns must contained the ID of the selection, which is given by combining the 'sound.files' and 'selec' columns of 'X',  separated by '-' (i.e. \code{paste(X$sound.files, X$selec, sep = "-")}). Default is \code{NULL}. If the 'selec' ID is not supplied then the entire sound file is used instead. If 'compare.matrix' is supplied only those comparisons will be calculated (as opposed to all pairwise comparisons as the default behavior) and the output will be a data frame composed of the supplied matrix and the correspondent cross-correlation values. Note that 'method' is automatically set to 2 (create spectrograms on the fly) when 'compare.matrix' is supplied but can be set back to 1. When supplied 'output' is forced to be 'list'. Note that the use of this function for signal detection (with 'compare.matrix') will be deprecated in future warbleR versions. Look at the ohun package for automatic signal detection functions.}

\item{type}{A character vector of length 1 specifying the type of cross-correlation; either "fourier" (i.e. spectrographic cross-correlation using Fourier transform; internally using \code{\link[seewave]{spectro}}; default) or "mfcc" (Mel cepstral coefficient spectrogram cross-correlation; internally using \code{\link[tuneR]{melfcc}}).}

\item{nbands}{Numeric vector of length 1 controlling the number of warped spectral bands to calculate when using \code{type = "mfcc"} (see \code{\link[tuneR]{melfcc}}). Default is 40.}

\item{method}{Numeric vector of length 1 to control the method used to create spectrogram matrices. Two option are available:
\itemize{
   \item \code{1}:  matrices are created first (keeping them internally as a list) and cross-correlation is calculated on a second step. Note that this method may require lots of memory if selection and or sound files are large.
   \item \code{2}: matrices are created "on the fly" (i.e. at the same time that cross-correlation is calculated). More memory efficient but may require extracting the same matrix several times, which will affect performance. Note that when using this method the function does not check if sound files have the same sampling rate which if not, may produce an error.
   }}
}
\value{
If \code{output = "cor.mat"} the function returns a matrix with
the maximum (peak) correlation for each pairwise comparison (if 'compare.matrix' is not supplied) or the peak correlation for each comparison in the supplied 'compare.matrix'. Otherwise it will return a list that includes 1) a matrix with
the maximum correlation for each pairwise comparison ('max.xcorr.matrix') and 2) a data frame with the correlation statistic for each "sliding" step ('scores').
}
\description{
\code{cross_correlation} estimates the similarity of two sound waves by means of time-frequency cross-correlation
}
\details{
This function calculates the pairwise similarity of multiple signals by means of time-frequency cross-correlation. Fourier or Mel frequency ceptral coefficients spectrograms can be used as time-frequency representations of sound.
This method "slides" the spectrogram of the shortest selection over the longest one calculating a correlation of the amplitude values at each step.
The function runs pairwise cross-correlations on several signals and returns a list including the correlation statistic
for each "sliding" step as well as the maximum (peak) correlation for each pairwise comparison.
The correlation matrix could have NA's if some of the pairwise correlation did not work (common when sound files have been modified by band-pass filters). Make sure all sound files have the same sampling rate (can be checked with \code{\link{check_sels}} or \code{\link{check_sound_files}}).
}
\examples{
{
  # load data
  data(list = c("Phae.long1", "Phae.long2", "Phae.long3", "Phae.long4", "lbh_selec_table"))

  # save sound files
  writeWave(Phae.long1, file.path(tempdir(), "Phae.long1.wav"))
  writeWave(Phae.long2, file.path(tempdir(), "Phae.long2.wav"))
  writeWave(Phae.long3, file.path(tempdir(), "Phae.long3.wav"))
  writeWave(Phae.long4, file.path(tempdir(), "Phae.long4.wav"))

  # run cross correlation on spectrograms (SPCC)
  xcor <- cross_correlation(X = lbh_selec_table, wl = 300, ovlp = 90, path = tempdir())

  # run cross correlation on Mel cepstral coefficients (mfccs)
  xcor <- cross_correlation(
    X = lbh_selec_table, wl = 300, ovlp = 90, path = tempdir(),
    type = "mfcc"
  )

  # using the 'compare.matrix' argument to specify pairwise comparisons
  # create matrix with ID of signals to compare
  cmp.mt <- cbind(
    paste(lbh_selec_table$sound.files[1:10], lbh_selec_table$selec[1:10], sep = "-"),
    paste(lbh_selec_table$sound.files[2:11], lbh_selec_table$selec[2:11], sep = "-")
  )

  # run cross-correlation on the selected pairwise comparisongs
  xcor <- cross_correlation(
    X = lbh_selec_table, compare.matrix = cmp.mt,
    wl = 300, ovlp = 90, path = tempdir()
  )
}
}
\references{
{
Araya-Salas, M., & Smith-Vidaurre, G. (2017). warbleR: An R package to streamline analysis of animal acoustic signals. Methods in Ecology and Evolution, 8(2), 184-191.

H. Khanna, S.L.L. Gaunt & D.A. McCallum (1997). Digital spectrographic cross-correlation: tests of sensitivity. Bioacoustics 7(3): 209-234

Lyon, R. H., & Ordubadi, A. (1982). Use of cepstra in acoustical signal analysis. Journal of Mechanical Design, 104(2), 303-306.
}
}
\seealso{
\code{\link{mfcc_stats}}, \code{\link{spectro_analysis}}, \code{\link{freq_DTW}}
}
\author{
Marcelo Araya-Salas \email{marcelo.araya@ucr.ac.cr})
}
