% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/vmgeom.R
\name{vmgeom}
\alias{vmgeom}
\alias{dvmgeom}
\alias{pvmgeom}
\alias{qvmgeom}
\alias{rvmgeom}
\title{Visual magnitude distribution of geometric distributed meteor magnitudes}
\usage{
dvmgeom(m, lm, r, log = FALSE, perception.fun = NULL)

pvmgeom(m, lm, r, lower.tail = TRUE, log = FALSE, perception.fun = NULL)

qvmgeom(p, lm, r, lower.tail = TRUE, perception.fun = NULL)

rvmgeom(n, lm, r, perception.fun = NULL)
}
\arguments{
\item{m}{numeric; the meteor magnitude.}

\item{lm}{numeric; limiting magnitude.}

\item{r}{numeric; the population index. It is the only parameter of the distribution.}

\item{log}{logical; if \code{TRUE}, probabilities p are given as \code{log(p)}.}

\item{perception.fun}{function; perception probability function (optional).
Default is \link{vmperception}.}

\item{lower.tail}{logical; if \code{TRUE} (default) probabilities are
\eqn{P[M < m]}, otherwise, \eqn{P[M \ge m]}.}

\item{p}{numeric; probability.}

\item{n}{numeric; count of meteor magnitudes.}
}
\value{
\code{dvmgeom} gives the density, \code{pvmgeom} gives the distribution function,
\code{qvmgeom} gives the quantile function, and \code{rvmgeom} generates random deviates.

The length of the result is determined by \code{n} for \code{rvmgeom}, and is the maximum
of the lengths of the numerical vector arguments for the other functions.

Since the distribution is discrete, \code{qvmgeom} and \code{rvmgeom} always return integer values.
\code{qvmgeom} can return \code{NaN} value with a warning.
}
\description{
Density, distribution function, quantile function and random generation for the
visual magnitude distribution of geometric distributed meteor magnitudes.
}
\details{
In visual meteor observation, it is common to estimate meteor magnitudes in integer values.
Hence, this distribution is discrete and has the density
\deqn{
    {\displaystyle P[X = x] \sim f(x) \, \mathrm r^{-x}} \,\mathrm{,}
}
where \eqn{x \ge -0.5} is the difference between the limiting magnitude \code{lm}
and the meteor magnitude \code{m} and \eqn{f(x)} is the perception probability function.
This distribution is thus a product of the
\link[=vmperception]{perception probabilities} and the
actual \link[stats:Geometric]{geometric distribution} of the meteor magnitudes.
Therefore, the parameter \code{p} of the geometric distribution is \code{p = 1 - 1/r}.

The parameter \code{lm} indicate what the parameter \code{m} refers to.
\code{m} must be an integer meteor magnitude.
The length of the vector \code{lm} must then be equal to the length of the vector \code{m}
or \code{lm} is a scalar value.
In case of \code{rvmgeom}, the length of the vector \code{lm} must be \code{n} or \code{lm} is a scalar value.

If the perception probabilities function \code{perception.fun} is given,
it must have the signature \verb{function(x)} and must return the perception probabilities of
the difference \code{x} between the limiting magnitude and the meteor magnitude.
If \code{x >= 15.0}, the \code{perception.fun} function should return the perception probability of \code{1.0}.
If \code{log = TRUE} is given, the logarithm value of the perception probabilities
must be returned. \code{perception.fun} is resolved using \link{match.fun}.
}
\examples{
N <- 100
r <- 2.0
limmag <- 6.5
(m <- seq(6, -7))

# discrete density of `N` meteor magnitudes
(freq <- round(N * dvmgeom(m, limmag, r)))

# log likelihood function
lld <- function(r) {
    -sum(freq * dvmgeom(m, limmag, r, log=TRUE))
}

# maximum likelihood estimation (MLE) of r
est <- optim(2, lld, method='Brent', lower=1.1, upper=4)

# estimations
est$par # mean of r

# generate random meteor magnitudes
m <- rvmgeom(N, r, lm=limmag)

# log likelihood function
llr <- function(r) {
    -sum(dvmgeom(m, limmag, r, log=TRUE))
}

# maximum likelihood estimation (MLE) of r
est <- optim(2, llr, method='Brent', lower=1.1, upper=4, hessian=TRUE)

# estimations
est$par # mean of r
sqrt(1/est$hessian[1][1]) # standard deviation of r

m <- seq(6, -4, -1)
p <- vismeteor::dvmgeom(m, limmag, r)
barplot(
    p,
    names.arg = m,
    main = paste0('Density (r = ', r, ', limmag = ', limmag, ')'),
    col = "blue",
    xlab = 'm',
    ylab = 'p',
    border = "blue",
    space = 0.5
)
axis(side = 2, at = pretty(p))
}
\seealso{
\link{vmperception}
\link[stats:Geometric]{stats::Geometric}
}
