% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/visibility.R
\name{visibility}
\alias{visibility}
\title{Computes the visibility from a single location in a 3D point cloud}
\usage{
visibility(data, position, angular_res, scene_radius, store_points)
}
\arguments{
\item{data}{LAS class object containing the xyz coordinates of a 3D point
cloud}

\item{position}{vector of length 3 containing the xyz coordinates of the
animal location. Default = c(0,0,0).}

\item{angular_res}{numeric. The angular resolution of a single sightline.
Default = 1.}

\item{scene_radius}{(optional) numeric. Defines the radius of the scene
relative to the animal position. Can be used to apply a cut-off distance to
visibility analyses.}

\item{store_points}{logical. If \code{TRUE}, the 3D point cloud is returned
with visible and not visible points classified (see details).}
}
\value{
If \code{store_points = FALSE}, a list containing the data.table of the visibility
(Visibility) as a function of distance to the animal location (r) is
returned and the viewshed coefficient (the area under the curve of visibility
as function of distance). If \code{store_points = TRUE}, a LAS containing
the point cloud (X, Y, Z), the distance of each point to the animal position
(r) and the class of each point: visible or not visible from the animal
position (Visibility = 2 or 1, respectively) is added to the output.
}
\description{
Computes visibility from a user-defined location with
user-defined sightline angles and returns the visibility as function of
distance and, optionally, the 3D point cloud classified as visible
and non-visible points.
}
\details{
Sightline directions are computed from the method described by
Malkin (2016). This ensures that every sightline explores
a similar portion of the 3d space.
}
\note{
In most cases, a ground reconstruction should be performed before
visibility computation to avoid sightlines passing through the ground. This
can be done with the \code{\link{reconstruct_ground}} function.
}
\examples{
\donttest{
#- import the tree_line_plot dataset
file <- system.file("extdata", "tree_line_plot.laz", package="viewshed3d")
tls <- lidR::readLAS(file)

center <- c(0,0,2) # defines the scene center for the entire process
angle <- 1 # defines the angular resolution for the entire process

#- remove noise to avoid visibility estimates error
tls_clean <- viewshed3d::denoise_scene(tls,method="sd",
                                       filter=6)


#- class ground and vegetation points
class <- lidR::classify_ground(tls_clean, lidR::csf(rigidness = 1L,
                                              class_threshold = 0.2,
                                              sloop_smooth = FALSE))

#- reconstruct the ground with the optimal resolution
recons <- viewshed3d::reconstruct_ground(data=class,
                                         position = center,
                                         ground_res = 0.05,
                                         angular_res = angle,
                                         method="knnidw",
                                         full_raster = TRUE)

#- compute the visibility and store the output point cloud.
#- As the input file is a LAS object, the output
#- point cloud is also stored in a LAS file.
view.data <- viewshed3d::visibility(data = recons,
                                    position = center,
                                    angular_res = angle,
                                    scene_radius = 17, # apply cut_oof distance
                                    store_points = TRUE)

#- viewshed coefficient
view.data$viewshed_coeffecient

#- 3D plot with visible points in white and non-visible points in darkgrey
x=lidR::plot(view.data$points,color="Visibility",colorPalette = c("grey24","white"))

#- add animal position to the plot
position=data.frame(X=center[1],Y=center[2],Z=center[3])
lidR::add_treetops3d(x,sp::SpatialPointsDataFrame(position,position),
                     radius=0.2,col="red")

#- plot the visibility as function of distance
plot(view.data$visibility$r,view.data$visibility$visibility,
     type="l",ylim=c(0,100),lwd=4)
}

}
\references{
Malkin, Z. (2016). A new method to subdivide a spherical surface
into equal-area cells. arXiv:1612.03467.
}
