% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/weighted_DA.R
\name{vlda}
\alias{vlda}
\title{Weighted Linear Discriminant Analysis}
\usage{
vlda(x, uncertainties, grouping, prior)
}
\arguments{
\item{x}{frame or matrix containing the data to be discriminated}

\item{uncertainties}{data frame or matrix containing the values for uncertainties per cell. Uncertainties should be relative errors, e.g. the relative standard deviation of the  measurand}

\item{grouping}{a factor or character vector specifying the group for each observation (row).}

\item{prior}{the prior probabilities of class membership. If unspecified, the class proportions for the training set are used. If present, the probabilities should be specified in the order of the factor levels.}
}
\value{
object of class 'vlda' containing the following components:
\code{prior} the prior probabilities used.
\code{counts} counts per group.
\code{means} the group means.
\code{generalizedMeans} the group means calculated by the function \code{\link{generalized_mean}}
\code{groupVarCorrected} the group variances calculated by the function \code{\link{calc_estimate_true_var}}
\code{lev} the levels of the grouping factor.
\code{grouping} the factor specifying the class for each observation.
}
\description{
Extension of the qda() of package 'MASS' (not the lda() function) to calculate a LDA incorporating individual, cell-wise uncertainties,
e.g. if the uncertainties are expressed as individual variances for each measurand.
}
\details{
Uncertainties can be considered in a statistical analysis either by each measured variable, by each observation or by using the individual, cell-wise uncertainties.
There are several methods for incorporating variable-wise or observation-wise uncertainties into a QDA, most of them using the uncertainties as weights for the variables or observations of the data set.
The term 'cell-wise uncertainties' describe a data set of $d$ analysed variables where each observation has an individual uncertainty for each of the $d$ variables conforming it.
Hence, a data set of $n \\times d$ data values has associated a data set of $n \\times d$ individual uncertainties.
Instead of weighting the columns or rows of the data set, the vlda() function uses uncertainties to recalculate better estimates of the group variances and group means.
It is internally very similar to the \code{\link{vqda}} function, but with an averaged group variance for all groups.
If the presence of uncertainties is not accounted for, the decision rules  are based on the group variances calculated by the given data set.
But this observed group variance might deviate notably from the group variance, which can be estimated including the uncertainties.
This methodological framework does not only allow to incorporate cell-wise uncertainties, but also would largely be valid if the information about the co-dependency between uncertainties within each observation would be reported.
}
\examples{
# for non-compositional data:
data("dataobs")
data("uncertainties")
mylda = vlda(x = dataobs[, 1:2], uncertainties = uncertainties[, 1:2], grouping = dataobs$Group)
mypred = predict(mylda, newdata = dataobs[, 1:2], newerror = uncertainties[, 1:2])
forplot = cbind(dataobs, LG1 = mypred$posterior[,1])
if (require("ggplot2")) {
  scatter_plot = ggplot(data = forplot, aes(x = Var1, y = Var2)) +
    geom_point(aes(shape = Group, color = LG1))
  if (require("ggthemes")) {
    scatter_plot = scatter_plot +
        scale_color_gradientn(colours = colorblind_pal()(5))
  }
  scatter_plot
}

# for compositional data
data("dataobs_coda")
data("uncertainties_coda")
require(compositions)
# generate ilr-transformation (from package 'compositions')
data_ilr = ilr(dataobs_coda[, 1:3])
uncert_ilr = t(simplify2array(apply(uncertainties_coda[, 1:3],1,
                       function(Delta) clrvar2ilr(diag(Delta)))))
uncert_ilr = rmult(uncert_ilr) # change class into rmult from package 'compositions'
mylda_coda = vlda(x = data_ilr, uncertainties = uncert_ilr, grouping = dataobs_coda$Group)
mypred_coda = predict(mylda_coda, newdata = data_ilr, newerror = uncert_ilr)
forplot_coda = cbind(dataobs_coda, LG1 = mypred_coda$posterior[,1])
# if 'ggtern' is installed, you can plot via ggtern:
# if (require("ggtern")) {
#   ternary_plot = ggtern(data = forplot_coda, aes(x = Var1, y = Var2, z = Var3)) +
#     geom_point(aes(shape = Group, color = LG1))
#   if (require("ggthemes")) {
#     ternary_plot = ternary_plot +
#         scale_color_gradientn(colours = colorblind_pal()(5))
#   }
#   ternary_plot
# }

}
\references{
Pospiech, S., R. Tolosana-Delgado and K.G. van den Boogaart (2020) Discriminant Analysis for Compositional Data Incorporating Cell-Wise Uncertainties, Mathematical Geosciences
}
\author{
Solveig Pospiech, package 'MASS'
}
