if (!isGeneric('pc3D_dsm')) {
  setGeneric('pc3D_dsm', function(x, ...)
    standardGeneric('pc3D_dsm'))
}

#'@name pc3D_dsm
#'@title Create a Digital Surface Model from a UAV generated point cloud
#'
#'@description
#' Create a Digital Surface Model from a UAV generated point cloud. Basically returns a DSM.
#' It uses the 'r.in.lidar' function to calculate LiDAR derived raster grids.
#' It creates a \code{raster*} object.
#' @note For using 'PDAL'(since GRASS7.4.x) you have to install the PDAL binaries and python bindings if not bundled with GRASS. running Linux \code{sudo apt-get install libpdal-base5 libpdal-dev libpdal-plugin-python pdal}
#' @seealso \href{https://grass.osgeo.org/grass70/manuals/r.in.lidar.html}{r.in.lidar help}
#'
#'@author Chris Reudenbach
#'
#'@param lasDir  character. default is \code{NULL} path  to the laz/las file(s)
#'@param gisdbasePath character. gisdbase will be linked or created depending on \code{gisdbase_exist}
#'@param GRASSlocation character. location will be linked or created depending on \code{gisdbase_exist}
#'@param projsubFolders character. subfolders that will be created/linked for R related GRASS processing
#'@param gridSize numeric. resolution for raster operations
#'@param smoothFilter  character. default is \code{gauss} alternatives are \code{spline} or for no smoothing at all \code{no}
#'@param splineNumber numeric. default is 4 number ob spline iterations
#'@param otb_gauss_radius numeric. default is \code{0.5} radius of otb smoothing filter in meter
#'@param dsm_minalt numeric. default is \code{0}, minimum DTM altitude accepted
#'@param dsm_area numeric. default \code{FALSE} generate polygon of valid DSM data
#'@param proj4  character. default is EPSG \code{32632} any valid proj4 string that is assumingly the correct one
#'@param gisdbase_exist logical. default is  \code{FALSE} switch if gisdbase is created or linked only
#'@param giLinks  list. of GI tools cli paths as generated by a full call of link2GI
#'@param grass_lidar_method character. statistical method to sample the Lidar data, see also \href{https://grass.osgeo.org/grass70/manuals/r.in.lidar.html}{r.in.lidar help}.
#'@param grass_lidar_pth numeric. grass lidar aggregation column percentile, see also \href{https://grass.osgeo.org/grass70/manuals/r.in.lidar.html}{r.in.lidar help}.
#'@param dsm_maxalt  numeric. dsm maximum altitude
#'@param pathLastools character. folder containing the Windows binary files of the lastools
#'@param verbose logical. to be quiet FALSE
#'@param cutExtent numerical. clip area c(mix,miny,maxx,maxy)
#'@param MP character mounting point / drive letter default is "~"
#'@param grassVersion numeric. version of GRASS as derived by findGRASS() default is 1 (=oldest/only version) please note GRASS version later than 7.4 is not working with r.inlidar
#'@param searchPath path to look for grass


#'@export pc3D_dsm
#'@examples
#' \dontrun{
#'require(uavRst)
#'require(link2GI)
#'
#' # create and check the links to the GI software
#' giLinks<-uavRst::linkAll()
#' (giLinks$saga$exist & giLinks$otb$exist & giLinks$grass$exist) 
#'{
#'# proj subfolders
#'projRootDir<-tempdir()
#'setwd(projRootDir)
#'unlink(paste0(projRootDir,"*"), force = TRUE)
#'
#'projsubFolders<-c("data/","data/ref/","output/","run/","las/")
#'paths<-link2GI::initProj(projRootDir = projRootDir,
#'                         projFolders = projsubFolders,
#'                         global = TRUE,
#'                         path_prefix = "path_")
#'# get some colors
#'pal = mapview::mapviewPalette("mapviewTopoColors")
#'
#'# get the data
#'utils::download.file(url="https://github.com/gisma/gismaData/raw/master/uavRst/data/lidar.las",
#'                     destfile="lasdata.las")
#'
#'# create a DSM  based on a uav point cloud
#'pc3DSM<-pc3D_dsm(lasDir =  "lasdata.las",
#'         gisdbasePath = projRootDir,
#'         projsubFolders = projsubFolders,
#'         gridSize = "0.5",
#'         giLinks = giLinks)
#'mapview::mapview(pc3DSM[[1]])
#'}
#'##+}
#'

pc3D_dsm <- function(lasDir = NULL,
                   gisdbasePath = NULL,
                   GRASSlocation = "tmp/",
                   grassVersion=1,
                   searchPath =NULL,
                   projsubFolders = c("data/","output/","run/","las/"),
                   gridSize = "0.25",
                   grass_lidar_method = "mean",
                   grass_lidar_pth = 90,
                   splineNumber = "4" ,
                   otb_gauss_radius = "0.5",
                   smoothFilter = "gauss",
                   dsm_minalt = 0,
                   dsm_maxalt = 4000,
                   dsm_area = FALSE,
                   cutExtent = NULL,
                   proj4 = "+proj=utm +zone=32 +datum=WGS84 +units=m +no_defs",
                   gisdbase_exist = FALSE,
                   pathLastools = NULL,
                   giLinks =NULL,
                   MP ="~",
                   verbose = FALSE) {

  if (is.null(searchPath)){
    if(Sys.info()["sysname"]=="Windows") {
      searchPath="C:"
      MP<- search
    }
    else {
      searchPath <- "/usr"
      MP<-"~"
      }
    }

  LASbin<-searchLastools(MP=MP)
  if (length(LASbin)<1) stop("\n At ",MP," no LAStool binaries found")
  else lasbin<- as.character(LASbin[[1]][,])

  if (is.null(giLinks)){
    giLinks <- linkAll()
  }

  gdal <- giLinks$gdal
  saga <- giLinks$saga
  otb <- giLinks$otb
  sagaCmd<-saga$sagaCmd
  path_OTB <- otb$pathOTB
  if (!verbose){
  GV <- Sys.getenv("GRASS_VERBOSE")
  Sys.setenv("GRASS_VERBOSE"=0)
  ois <- rgrass7::get.ignore.stderrOption()
  rgrass7::set.ignore.stderrOption(TRUE)}

  # get/map the las binary folder and create the base command line
  if (is.null(lasDir)) stop("no directory containing las/laz files provided...\n")
  lasDir <- path.expand(lasDir)
  if (Sys.info()["sysname"] == "Windows") {
    #cmd <- pathLastools <- paste(system.file(package = "uavRst"), "lastools", sep="/")/lastools/bin
    if (is.null(pathLastools)) {cmd <- pathLastools <- lasbin
    if (verbose) cat("\n You did not provide a path to your lastool binary folder. Assuming C:/lastools/bin\n")}
  } else {
    #cmd <- paste("wine ",pathLastools <- paste(system.file(package = "uavRst"), "lastools",  sep="/"))
    if (is.null(pathLastools)) {cmd <- pathLastools <- paste0("wine ",lasbin)
    if (verbose) cat("\n You did not provide a path to your lastool binary folder.\n
                         Assumed to be somewher in your home directory")}

  }

  #create cmd strings
  lasmerge      <- paste(cmd,"lasmerge.exe",sep = "/")
  lasinfo       <- paste(cmd,"lasinfo.exe",sep = "/")
  las2dem       <- paste(cmd,"las2dem.exe",sep = "/")
  # delete content in run directory


  # some basic checks

  if(raster::extension(basename(lasDir)) !=".las" & raster::extension(basename(lasDir)) !=".laz") {
    # check las / laz files laz will be preferred
    lasFileNames <- list.files(pattern = "[.]las$", path = lasDir, full.names = TRUE)
    lazFileNames <- list.files(pattern = "[.]laz$", path = lasDir, full.names = TRUE)
    if (length(lazFileNames) > 0 ) {
      extFN <- substr(raster::extension(basename(lazFileNames[1])),2,4)
      noF <- length(lazFileNames)
    }
    else if (length(lasFileNames) > 0) {
      extFN <- substr(raster::extension(basename(lasFileNames[1])),2,4)
      noF <- length(lasFileNames)
    }
    else stop("no valid las or laz files found...\n")
    # merge all files
    cat("\nNOTE: You are dealing with a huge UAV generated point cloud data set.\n      so take time and keep relaxed... :-)\n")
    cat(":: merge and decompress ",noF," point cloud files...\n")
    ret <- system(paste0(lasmerge,
                         " -i ",lasDir,"*.",extFN,
                         " -olas",
                         " -o ",path_run,"full_point_cloud.las"),
                  intern = TRUE,
                  ignore.stderr = TRUE

    )
    name<-"full_point_cloud.las"
  }
   else {
     name<-basename(lasDir)
     if (!file.exists(paste0(path_run,name)))
       file.copy(from = lasDir,
                 to = paste0(path_run,name),
                 overwrite = TRUE)

   }

  if (!is.null(cutExtent)){
  #lastool(tool = "lasclip",lasFile = lasfile,cutExtent = cutExtent)
  las = lidR::readLAS(paste0(path_run,name))
  las_clip<-lidR::lasclipRectangle(las, as.numeric(cutExtent[1]), as.numeric(cutExtent[3]), as.numeric(cutExtent[2]), as.numeric(cutExtent[4]))
  lidR::writeLAS(las_clip ,paste0(path_run,"cut_point_cloud.las"))
  name<-"cut_point_cloud.las"
  }

  # get extent of merged file
  sp_param <-lastool(lasFile= paste0(path_run,name))
  # rename output file according to the extent
  fn<- paste(sp_param ,collapse=" ")
  tmp <- gsub(paste(sp_param ,collapse=" "),pattern = " ",replacement = "_")
  fn<-gsub(tmp,pattern = "[.]",replacement = "_")
  # copy it to the output folder
  file.rename(from = paste0(path_run,name),
            to = paste0(path_run,fn,".las"))
  # add proj4 string manually
  sp_param[5] <- proj4
  # create and export globally project folder structure
  paths<-link2GI::initProj(projRootDir = gisdbasePath,
                           GRASSlocation = GRASSlocation,
                           projFolders =  projsubFolders)

  # create GRASS7 connection according gisdbase_exist (permanent or temporary)
  if (gisdbase_exist)
    paths<-link2GI::linkGRASS7(gisdbase = gisdbasePath,
                               location = GRASSlocation,
                               gisdbase_exist = TRUE,
                               ver_select = grassVersion,
                               search_path = searchPath)
  else
    paths<-link2GI::linkGRASS7(gisdbase = gisdbasePath,
                               location = GRASSlocation,
                               spatial_params = sp_param,
                               resolution = gridSize,
                               ver_select = grassVersion,
                               search_path = searchPath)

  # create raw DSM using r.in.lidar
  cat(":: calculate DSM...\n")
  # #if (!grepl(system("g.extension -l",ignore.stdout = TRUE),pattern = "r.in.lidar"))
  #   ret <- rgrass7::execGRASS("r.in.pdal",
  #                             flags  = c("overwrite","quiet"),
  #                             input  = paste0(path_run,fn,".las"),
  #                             output = fn,
  #                             method = grass_lidar_method,
  #                             pth = grass_lidar_pth,
  #                             proj_in = sp_param[5],
  #                             resolution = as.numeric(gridSize),
  #                             intern = TRUE,
  #                             ignore.stderr = TRUE
  #   )
  # else
  ret <- rgrass7::execGRASS("r.in.lidar",
                            flags  = c("overwrite","quiet","o"),
                            input  = paste0(path_run,fn,".las"),
                            output = fn,
                            method = grass_lidar_method,
                            pth = grass_lidar_pth,
                            resolution = as.numeric(gridSize),
                            intern = TRUE,
                            ignore.stderr = TRUE
  )



  cat(":: convert raw DSM to GeoTiff \n")
  #h_grass2tif(runDir = path_run, layerName = "point_cloud_dsm")
  raster::writeRaster(raster::raster(rgrass7::readRAST(fn)),paste0(path_run,fn), overwrite=TRUE,format="GTiff")
  cat(":: fill no data... \n")
  #uavRst:::fillGaps(path_run,paste0("point_cloud_dsm.tif "))
  ret <- system(paste0("gdal_fillnodata.py ",
                       path_run,fn,".tif ",
                       path_run,fn,".tif"),intern = TRUE)
  dsm <- raster::raster(paste0(path_run,fn,".tif"))

  cat(":: smoothing the gap filled DSM... \n")
  # otb gaussian smooth
  if (smoothFilter != "no") {
    if (smoothFilter == "gauss") {
      otb_gauss_radius <- as.character(as.numeric(otb_gauss_radius)/as.numeric(gridSize))
      module  <- "otbcli_Smoothing"
      command <- paste0(path_OTB, module)
      command <- paste0(command, " -in ",path_run,fn,".tif")
      command <- paste0(command, " -out ", path_run,"dsm_",fn,".tif")
      command <- paste0(command, " -ram 4096")
      command <- paste0(command, " -type gaussian")
      command <- paste0(command, " -type.gaussian.radius ",otb_gauss_radius) #(in pixel)
      ret <- system(command, intern = TRUE,ignore.stdout = TRUE,ignore.stderr = TRUE)
      dsm <- raster::raster(paste0(path_run,"dsm_",fn,".tif"))
    }
    if (smoothFilter == "spline") {
      #
      gdalUtils::gdalwarp(paste0(path_run,fn,".tif"),
                          paste0(path_run,fn,".sdat"),
                          overwrite = TRUE,
                          of = 'SAGA',
                          verbose = FALSE)
      # saga spline interpolation of the alt values
      cat(":: iterative spline smooth of dsm... \n")
      ret <- system(paste0(sagaCmd,' grid_spline 5 ',
                           ' -GRID ', path_run,fn,'.sgrd',
                           ' -TARGET_DEFINITION 0',
                           ' -TARGET_OUT_GRID ',path_run,"dsm_",fn,".sgrd",
                           ' -TARGET_USER_SIZE ',gridSize,
                           ' -METHOD 1',
                           ' -EPSILON 0.0001',
                           ' -LEVEL_MAX ',splineNumber),
                    intern = TRUE,
                    ignore.stdout =  TRUE,
                    ignore.stderr = TRUE)
      dsm <- gdalUtils::gdalwarp(paste0(path_run,"dsm_",fn,".sdat"),
                                 paste0(path_run,"dsm_",fn,".tif"),
                                 t_srs = proj4,
                                 output_Raster = TRUE,
                                 overwrite = TRUE,
                                 verbose = FALSE)
    }
  }
  cat(":: calculate metadata ... \n")
  dsm[dsm <= dsm_minalt] <- NA
  dsm[dsm > dsm_maxalt] <- NA
  raster::writeRaster(dsm, paste0(path_run,"dsm_",fn,".tif"), overwrite = TRUE)
  e <- extent(dsm)
  dsmA <- methods::as(e, 'SpatialPolygons')
  if (dsm_area) {
    dsm2 <- dsm > -Inf
    tmp <- raster::aggregate(dsm2,fact = 1 / gridSize)
    dsmdA  <- rasterToPolygons(tmp)
    dsmdA  <- rgeos::gUnaryUnion(dsmdA)
    #dsmdA <- rasterToPolygons(dsm2, dissolve=TRUE)
  }
  else dsmdA <- NULL
  if (!verbose)  {
    Sys.setenv("GRASS_VERBOSE"=GV)
    rgrass7::set.ignore.stderrOption(ois)
    }

  return(list(dsm,dsmA,dsmdA,paste0(fn,".las")))
}
