% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/TARMA.fit2.R
\name{TARMA.fit2}
\alias{TARMA.fit2}
\title{TARMA Modelling of Time Series}
\usage{
TARMA.fit2(
  x,
  ar.lags = NULL,
  tar1.lags = c(1),
  tar2.lags = c(1),
  ma.ord = 1,
  sma.ord = 0L,
  period = NA,
  estimate.thd = TRUE,
  threshold,
  d = 1,
  pa = 0.25,
  pb = 0.75,
  thd.var = NULL,
  include.int = TRUE,
  x.reg = NULL,
  optim.control = list(),
  ...
)
}
\arguments{
\item{x}{A univariate time series.}

\item{ar.lags}{Vector of common AR lags. Defaults to \code{NULL}. It can be a subset of lags.}

\item{tar1.lags}{Vector of AR lags for the lower regime. It can be a subset of \code{1 ... p1 = max(tar1.lags)}.}

\item{tar2.lags}{Vector of AR lags for the upper regime. It can be a subset of \code{1 ... p2 = max(tar2.lags)}.}

\item{ma.ord}{Order of the MA part (also called \code{q} below).}

\item{sma.ord}{Order of the seasonal MA part (also called \code{Q} below).}

\item{period}{Period of the seasonal MA part (also called \code{s} below).}

\item{estimate.thd}{Logical. If \code{TRUE} estimates the threshold over the threshold range specified by \code{pa} and \code{pb}.}

\item{threshold}{Threshold parameter. Used only if \code{estimate.thd = FALSE}.}

\item{d}{Delay parameter. Defaults to \code{1}.}

\item{pa}{Real number in \code{[0,1]}. Sets the lower limit for the threshold search to the \code{100*pa}-th sample percentile.
The default is \code{0.25}}

\item{pb}{Real number in \code{[0,1]}. Sets the upper limit for the threshold search to the \code{100*pb}-th sample percentile.
The default is \code{0.75}}

\item{thd.var}{Optional exogenous threshold variable. If \code{NULL} it is set to \code{lag(x,-d)}.
If not \code{NULL} it has to be a \code{ts} object.}

\item{include.int}{Logical. If \code{TRUE} includes the intercept terms in both regimes, a common intercept is included otherwise.}

\item{x.reg}{Covariates to be included in the model. These are passed to \code{arima}. If they are not \code{ts} objects they must have the same length as \code{x}.}

\item{optim.control}{List of control parameters for the optimization method.}

\item{\dots}{Additional arguments passed to \code{arima}.}
}
\value{
A list of class \code{TARMA} with components:
\itemize{
\item \code{fit}    - The output of the fit. It is a \code{arima} object.
\item \code{aic}    - Value of the AIC for the minimised least squares criterion over the threshold range.
\item \code{bic}    - Value of the BIC for the minimised least squares criterion over the threshold range.
\item \code{aic.v}  - Vector of values of the AIC over the threshold range.
\item \code{thd.range} - Vector of values of the threshold range.
\item \code{d}      - Delay parameter.
\item \code{thd}    - Estimated threshold.
\item \code{phi1}   - Estimated AR parameters for the lower regime.
\item \code{phi2}   - Estimated AR parameters for the upper regime.
\item \code{theta1} - Estimated MA parameters for the lower regime.
\item \code{theta2} - Estimated MA parameters for the upper regime.
\item \code{delta}  - Estimated parameters for the covariates \code{x.reg}.
\item \code{tlag1}  - TAR lags for the lower regime
\item \code{tlag2}  - TAR lags for the upper regime
\item \code{mlag1}  - TMA lags for the lower regime
\item \code{mlag2}  - TMA lags for the upper regime
\item \code{arlag}  -  Same as the input slot \code{ar.lags}
\item \code{include.int} -  Same as the input slot \code{include.int}
\item \code{se}     - Standard errors for the parameters. Note that they are computed conditionally upon the threshold so that they are generally smaller than the true ones.
\item \code{rss}    - Minimised residual sum of squares.
\item \code{method} - Estimation method.
\item \code{call}   - The matched call.
}
}
\description{
\loadmathjax
Maximum Likelihood fit of a two-regime \code{TARMA(p1,p2,q,q)}
model with common MA parameters, possible common AR parameters and possible covariates.
}
\details{
Fits the following two-regime \code{TARMA} process with optional components: linear \code{AR} part, seasonal \code{MA} and covariates. \cr
\mjdeqn{X_{t} = \phi_{0} + \sum_{h \in I} \phi_{h} X_{t-h} + \sum_{l=1}^Q \Theta_{l} \epsilon_{t-ls} + \sum_{j=1}^q \theta_{j} \epsilon_{t-j} + \sum_{k=1}^K \delta_{k} Z_{k} + \epsilon_{t} + \left\lbrace
    \begin{array}{ll}
\phi_{1,0} + \sum_{i \in I_1} \phi_{1,i} X_{t-i}   & \mathrm{if } X_{t-d} \leq \mathrm{thd} \\\\\\
 &\\\\\\
\phi_{2,0} + \sum_{i \in I_2} \phi_{2,i} X_{t-i}   & \mathrm{if } X_{t-d} > \mathrm{thd}
\end{array}
\right. }{X[t] = \phi[0] + \Sigma_{h in I}  \phi[h] X[t-h] + \Sigma_{j = 1,..,q} \theta[j] \epsilon[t-j] + \Sigma_{j = 1,..,Q} \Theta[j] \epsilon[t-js] + \Sigma_{k = 1,..,K} \delta[k] Z[k] + \epsilon[t] +
+ \phi[1,0] + \Sigma_{i  in I_1} \phi[1,i] X[t-i] --  if X[t-d] <= thd
+ \phi[2,0] + \Sigma_{i in I_2} \phi[2,i] X[t-i]  --  if X[t-d] > thd}

where \mjeqn{\phi_h}{\phi[h]} are the common AR parameters and \mjseqn{h} ranges in \code{I = ar.lags}. \mjeqn{\theta_j}{\theta[j]} are the common MA parameters and \mjeqn{j = 1,\dots,q}{j = 1,...,q}
(\code{q = ma.ord}), \mjeqn{\Theta_l}{\Theta[l]} are the common seasonal MA parameters and \mjeqn{l = 1,\dots,Q}{l = 1,...,Q} (\code{Q = sma.ord})
\mjeqn{\delta_k}{\delta[k]} are the parameters for the covariates. Finally, \mjeqn{\phi_{1,i}}{\phi[1,i]} and \mjeqn{\phi_{2,i}}{\phi[2,i]} are the TAR parameters
for the lower and upper regime, respectively and \code{I1 = tar1.lags} \code{I2 = tar2.lags} are the vector of TAR lags.
}
\examples{
## a TARMA(1,1,1,1)
set.seed(127)
x    <- TARMA.sim(n=100, phi1=c(0.5,-0.5), phi2=c(0,0.8), theta1=0.5, theta2=0.5, d=1, thd=0.2)
fit1 <- TARMA.fit2(x, tar1.lags=1, tar2.lags=1, ma.ord=1, d=1)

\donttest{
## Showcase of the fit with covariates ---
## simulates from a TARMA(3,3,1,1) model with common MA parameter
## and common AR(1) and AR(2) parameters. Only the lag 3 parameter varies across regimes
set.seed(212)
n <- 300
x <- TARMA.sim(n=n, phi1=c(0.5,0.3,0.2,0.4), phi2=c(0.5,0.3,0.2,-0.2), theta1=0.4, theta2=0.4,
     d=1, thd=0.2, s1=1, s2=1)

## FIT 1: estimates lags 1,2,3 as threshold lags ---
fit1 <- TARMA.fit2(x, ma.ord=1, tar1.lags=c(1,2,3), tar2.lags=c(1,2,3), d=1)

## FIT 2: estimates lags 1 and 2 as fixed AR and lag 3 as the threshold lag
fit2 <- TARMA.fit2(x, ma.ord=1, tar1.lags=c(3),  tar2.lags=c(3), ar.lags=c(1,2), d=1)

## FIT 3: creates lag 1 and 2 and fits them as covariates ---
z1   <- lag(x,-1)
z2   <- lag(x,-2)
fit3 <- TARMA.fit2(x, ma.ord=1,  tar1.lags=c(3), tar2.lags=c(3), x.reg=ts.intersect(z1,z2), d=1)

## FIT 4: estimates lag 1 as a covariate, lag 2 as fixed AR and lag 3 as the threshold lag
fit4 <- TARMA.fit2(x, ma.ord = 1,  tar1.lags=c(3), tar2.lags=c(3), x.reg=z1, ar.lags=2, d=1)

}
}
\references{
\itemize{
\item \insertRef{Gia21}{tseriesTARMA}
\item \insertRef{Cha19}{tseriesTARMA}
}
}
\seealso{
\code{\link{TARMA.fit}} for Least Square estimation of full subset TARMA
models. \code{\link{print.TARMA}} for print methods for \code{TARMA} fits.
\code{\link{predict.TARMA}} for prediction and forecasting.
}
\author{
Simone Giannerini, \email{simone.giannerini@unibo.it}

Greta Goracci, \email{greta.goracci@unibo.it}
}
