% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/tq_get.R
\name{tq_get}
\alias{tq_get}
\alias{tq_get_options}
\alias{tq_get_stock_index_options}
\title{Get quantitative data in \code{tibble} format}
\usage{
tq_get(x, get = "stock.prices", ...)

tq_get_options()

tq_get_stock_index_options()
}
\arguments{
\item{x}{A character string representing a single stock index,
stock symbol, metal symbol, currency combination, FRED code, etc.}

\item{get}{A character string representing the type of data to get
for \code{x}. Options include:
\itemize{
  \item \code{"stock.index"}: Get all stocks in an index or exchange
  from \href{http://www.marketvolume.com/indexes_exchanges/}{marketvolume.com}.
  \item \code{"stock.prices"}: Get the stock prices for a stock symbol from
  \href{https://finance.yahoo.com/}{Yahoo Finance}.
  \item \code{"dividends"}: Get the dividends for a stock symbol
  from \href{https://finance.yahoo.com/}{Yahoo Finance}.
  \item \code{"splits"}: Get the splits for a stock symbol
  from \href{https://finance.yahoo.com/}{Yahoo Finance}.
  \item \code{"financials"}: Get the income, balance sheet, and cash flow
  financial statements for a stock symbol from
  \href{https://www.google.com/finance}{Google Finance}.
  \item \code{"key.ratios"}: Get historical growth, profitablity, financial health,
  efficiency, and valuation ratios over 10-years for a stock symbol from
  \href{https://www.morningstar.com}{morningstar.com}.
  \item \code{"economic.data"}: Get economic data from
  \href{https://fred.stlouisfed.org/}{FRED}.
  \item \code{"metal.prices"}: Get the metal prices from
  \href{https://www.oanda.com/}{Oanda}.
  \item \code{"exchange.rates"}: Get exchange rates from
  \href{https://www.oanda.com/currency/converter/}{Oanda}.
}}

\item{...}{Additional parameters passed to the appropriate \code{quantmod}
function. Common optional parameters include:
\itemize{
  \item \code{from}: Optional. A character string representing a start date in
  YYYY-MM-DD format. No effect on \code{get = "stock.index"},
  \code{"financials"}, or \code{"key.ratios"}.
  \item \code{to}: A character string representing a end date in
  YYYY-MM-DD format. No effect on \code{get = "stock.index"},
  \code{get = "financials"}, or \code{"key.ratios"}.
  \item \code{use_fallback}: Used with \code{get = "stock.index"} only.
  Set to \code{FALSE} by default. A boolean
  representing whether to use the fall back data set for a stock index. Useful
  if the data cannot be fetched from the website. The fallback data returned is
  accurate as of the date the package was last updated.
}}
}
\value{
Returns data in the form of a \code{tibble} object.
}
\description{
Get quantitative data in \code{tibble} format
}
\details{
\code{tq_get()} is a consolidated function that gets data from various
web sources. The function is a wrapper for several \code{quantmod}
functions. The results are always returned as a \code{tibble}. The advantages
are (1) only one function is needed for all data sources and (2) the function
can be seemlessly used with the tidyverse: \code{purrr}, \code{tidyr}, and
\code{dplyr} verbs.

\code{tq_get_options()} returns a list of valid `get` options you can
choose from.

\code{tq_get_stock_index_options()} returns a list of stock indexes you can
choose from. Alternatively \code{tq_get("options", get = "stock.index")}
can be used.
}
\examples{
# Load libraries
library(tidyverse)
library(tidyquant)

##### Basic Functionality

# Get the list of `get` options
tq_get_options()

# Get all stocks in a stock index from www.marketvolume.com
tq_get("SP500", get = "stock.index")

# Get the list of stock index options that can be used with tq_get(get = "stock.index")
tq_get_stock_index_options()

# Get stock prices for a stock from Yahoo
aapl_stock_prices <- tq_get("AAPL")
cvx_stock_prices  <- tq_get("CVX", get = "stock.prices",
                            from = "2014-01-01", to = "2015-01-01")

# Get dividends and splits for a stock from Yahoo
tq_get("AAPL", get = "dividends", from = "1990-01-01")
tq_get("AAPL", get = "splits", from = "1990-01-01")

# Get financial statement data for a stock from Google
appl_financials <- tq_get("AAPL", get = "financials")

# Get key ratios for a stock from Morningstar
appl_key_ratios <- tq_get("AAPL", get = "key.ratios")

# Get FRED economic data for a commodity code
tq_get("DCOILWTICO", get = "economic.data") # WTI crude oil spot prices

# Get exchange rate data from Oanda
eur_usd <- tq_get("EUR/USD", get = "exchange.rates")

# Get metal prices from Oanda
plat_price_usd <- tq_get("plat", get = "metal.prices")
gold_price_eur <- tq_get("gold", get = "metal.prices",
                                 base.currency = "EUR")

##### Tidyverse functionality

# Get a historical stock prices from multiple stocks
FANG <- tibble(symbol = c("FB", "AMZN", "NFLX", "GOOGL")) \%>\%
    mutate(stock.prices = map(.x = symbol,
                              ~ tq_get(.x, get = "stock.prices"))) \%>\%
    unnest()

}

