% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/cox_regression.R
\name{fit_coxreg}
\alias{fit_coxreg}
\alias{fit_coxreg_univar}
\alias{fit_coxreg_multivar}
\title{Fits for Cox Proportional Hazards Regression}
\usage{
fit_coxreg_univar(variables, data, at = list(), control = control_coxreg())

fit_coxreg_multivar(variables, data, control = control_coxreg())
}
\arguments{
\item{variables}{(\code{list})\cr a named list corresponds to the names of variables found in \code{data}, passed as a named
list and corresponding to \code{time}, \code{event}, \code{arm}, \code{strata}, and \code{covariates} terms. If \code{arm} is missing from
\code{variables}, then only Cox model(s) including the \code{covariates} will be fitted and the corresponding effect
estimates will be tabulated later.}

\item{data}{(\code{data.frame})\cr the dataset containing the variables to fit the models.}

\item{at}{(\code{list} of \code{numeric})\cr when the candidate covariate is a \code{numeric}, use \code{at} to specify
the value of the covariate at which the effect should be estimated.}

\item{control}{(\code{list})\cr a list of parameters as returned by the helper function \code{\link[=control_coxreg]{control_coxreg()}}.}
}
\value{
\itemize{
\item \code{fit_coxreg_univar()} returns a \code{coxreg.univar} class object which is a named \code{list}
with 5 elements:
\itemize{
\item \code{mod}: Cox regression models fitted by \code{\link[survival:coxph]{survival::coxph()}}.
\item \code{data}: The original data frame input.
\item \code{control}: The original control input.
\item \code{vars}: The variables used in the model.
\item \code{at}: Value of the covariate at which the effect should be estimated.
}
}

\itemize{
\item \code{fit_coxreg_multivar()} returns a \code{coxreg.multivar} class object which is a named list
with 4 elements:
\itemize{
\item \code{mod}: Cox regression model fitted by \code{\link[survival:coxph]{survival::coxph()}}.
\item \code{data}: The original data frame input.
\item \code{control}: The original control input.
\item \code{vars}: The variables used in the model.
}
}
}
\description{
\ifelse{html}{\href{https://lifecycle.r-lib.org/articles/stages.html#stable}{\figure{lifecycle-stable.svg}{options: alt='[Stable]'}}}{\strong{[Stable]}}

Fitting functions for univariate and multivariate Cox regression models.
}
\section{Functions}{
\itemize{
\item \code{fit_coxreg_univar()}: Fit a series of univariate Cox regression models given the inputs.

\item \code{fit_coxreg_multivar()}: Fit a multivariate Cox regression model.

}}
\note{
When using \code{fit_coxreg_univar} there should be two study arms.
}
\examples{
library(survival)

set.seed(1, kind = "Mersenne-Twister")

# Testing dataset [survival::bladder].
dta_bladder <- with(
  data = bladder[bladder$enum < 5, ],
  data.frame(
    time = stop,
    status = event,
    armcd = as.factor(rx),
    covar1 = as.factor(enum),
    covar2 = factor(
      sample(as.factor(enum)),
      levels = 1:4, labels = c("F", "F", "M", "M")
    )
  )
)
labels <- c("armcd" = "ARM", "covar1" = "A Covariate Label", "covar2" = "Sex (F/M)")
formatters::var_labels(dta_bladder)[names(labels)] <- labels
dta_bladder$age <- sample(20:60, size = nrow(dta_bladder), replace = TRUE)

plot(
  survfit(Surv(time, status) ~ armcd + covar1, data = dta_bladder),
  lty = 2:4,
  xlab = "Months",
  col = c("blue1", "blue2", "blue3", "blue4", "red1", "red2", "red3", "red4")
)

# fit_coxreg_univar

## Cox regression: arm + 1 covariate.
mod1 <- fit_coxreg_univar(
  variables = list(
    time = "time", event = "status", arm = "armcd",
    covariates = "covar1"
  ),
  data = dta_bladder,
  control = control_coxreg(conf_level = 0.91)
)

## Cox regression: arm + 1 covariate + interaction, 2 candidate covariates.
mod2 <- fit_coxreg_univar(
  variables = list(
    time = "time", event = "status", arm = "armcd",
    covariates = c("covar1", "covar2")
  ),
  data = dta_bladder,
  control = control_coxreg(conf_level = 0.91, interaction = TRUE)
)

## Cox regression: arm + 1 covariate, stratified analysis.
mod3 <- fit_coxreg_univar(
  variables = list(
    time = "time", event = "status", arm = "armcd", strata = "covar2",
    covariates = c("covar1")
  ),
  data = dta_bladder,
  control = control_coxreg(conf_level = 0.91)
)

## Cox regression: no arm, only covariates.
mod4 <- fit_coxreg_univar(
  variables = list(
    time = "time", event = "status",
    covariates = c("covar1", "covar2")
  ),
  data = dta_bladder
)

# fit_coxreg_multivar

## Cox regression: multivariate Cox regression.
multivar_model <- fit_coxreg_multivar(
  variables = list(
    time = "time", event = "status", arm = "armcd",
    covariates = c("covar1", "covar2")
  ),
  data = dta_bladder
)

# Example without treatment arm.
multivar_covs_model <- fit_coxreg_multivar(
  variables = list(
    time = "time", event = "status",
    covariates = c("covar1", "covar2")
  ),
  data = dta_bladder
)

}
\seealso{
\link{h_cox_regression} for relevant helper functions, \link{cox_regression}.
}
