% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/tar_repository_cas_local.R
\name{tar_repository_cas_local}
\alias{tar_repository_cas_local}
\title{Local content-addressable storage (CAS) repository
(an experimental feature).}
\usage{
tar_repository_cas_local(path = NULL, consistent = FALSE)
}
\arguments{
\item{path}{Character string, file path to the CAS repository
where all the data object files will be stored. \code{NULL} to default to
\code{file.path(tar_config_get("store"), "cas")} (usually \code{"_targets/cas/"}).}

\item{consistent}{Logical. Set to \code{TRUE} if the storage platform is
strongly read-after-write consistent. Set to \code{FALSE} if the platform
is not necessarily strongly read-after-write consistent.

A data storage system is said to have strong read-after-write consistency
if a new object is fully available for reading as soon as the write
operation finishes. Many modern cloud services like Amazon S3 and
Google Cloud Storage have strong read-after-write consistency,
meaning that if you upload an object with a PUT request, then a
GET request immediately afterwards will retrieve the precise
version of the object you just uploaded.

Some storage systems do not have strong read-after-write consistency.
One example is network file systems (NFS). On a computing cluster,
if one node creates a file on an NFS, then there is a delay before
other nodes can access the new file. \code{targets} handles this situation
by waiting for the new file to appear with the correct hash
before attempting to use it in downstream computations.
\code{consistent = FALSE} imposes a waiting period in which \code{targets}
repeatedly calls the \code{exists} method until the file becomes available
(at time intervals configurable with \code{\link[=tar_resources_network]{tar_resources_network()}}).
These extra calls to \code{exists} may come with a
little extra latency / computational burden,
but on systems which are not strongly read-after-write consistent,
it is the only way \code{targets} can safely use the current results
for downstream computations.}
}
\value{
A character string from \code{\link[=tar_repository_cas]{tar_repository_cas()}} which may be
passed to the \code{repository} argument of \code{\link[=tar_target]{tar_target()}} or
\code{\link[=tar_option_set]{tar_option_set()}} to use a local CAS system.
}
\description{
Local content-addressable storage (CAS) repository.
}
\details{
Pass to the \code{repository} argument of \code{\link[=tar_target]{tar_target()}} or
\code{\link[=tar_option_set]{tar_option_set()}} to use a local CAS system.
}
\section{Content-addressable storage}{

Normally, \code{targets} organizes output data
based on target names. For example,
if a pipeline has a single target \code{x} with default settings,
then \code{\link[=tar_make]{tar_make()}} saves the output data to the file
\verb{_targets/objects/x}. When the output of \code{x} changes, \code{\link[=tar_make]{tar_make()}}
overwrites \verb{_targets/objects/x}.
In other words, no matter how many changes happen to \code{x},
the data store always looks like this:

\if{html}{\out{<div class="sourceCode">}}\preformatted{_targets/
    meta/
        meta
    objects/
        x
}\if{html}{\out{</div>}}

By contrast, with content-addressable storage (CAS),
\code{targets} organizes outputs based on the hashes of their contents.
The name of each output file is its hash, and the
metadata maps these hashes to target names. For example, suppose
target \code{x} has \code{repository = tar_repository_cas_local("my_cas")}.
When the output of \code{x} changes, \code{\link[=tar_make]{tar_make()}} creates a new file
inside \verb{my_cas/} without overwriting or deleting any other files
in that folder. If you run \code{\link[=tar_make]{tar_make()}} three different times
with three different values of \code{x}, then storage will look like this:

\if{html}{\out{<div class="sourceCode">}}\preformatted{_targets/
    meta/
        meta
my_cas/
    1fffeb09ad36e84a
    68328d833e6361d3
    798af464fb2f6b30
}\if{html}{\out{</div>}}

The next call to \code{tar_read(x)} uses \code{tar_meta(x)$data}
to look up the current hash of \code{x}. If \code{tar_meta(x)$data} returns
\code{"1fffeb09ad36e84a"}, then \code{tar_read(x)} returns the data from
\verb{my_cas/1fffeb09ad36e84a}. Files \verb{my_cas/68328d833e6361d3} and
and \verb{my_cas/798af464fb2f6b30} are left over from previous values of \code{x}.

Because CAS accumulates historical data objects,
it is ideal for data versioning and collaboration.
If you commit the \verb{_targets/meta/meta} file to version control
alongside the source code,
then you can revert to a previous state of your pipeline with all your
targets up to date, and a colleague can leverage your hard-won
results using a fork of your code and metadata.

The downside of CAS is the cost of accumulating many data objects
over time. Most pipelines that use CAS
should have a garbage collection system or retention policy
to remove data objects when they no longer needed.

The \code{\link[=tar_repository_cas]{tar_repository_cas()}} function lets you create your own CAS system
for \code{targets}. You can supply arbitrary custom methods to upload,
download, and check for the existence of data objects. Your custom
CAS system can exist locally on a shared file system or remotely
on the cloud (e.g. in an AWS S3 bucket).
See the "Repository functions" section and the documentation
of individual arguments for advice on how
to write your own methods.

The \code{\link[=tar_repository_cas_local]{tar_repository_cas_local()}} function has an example
CAS system based on a local folder on disk.
It uses \code{\link[=tar_cas_u]{tar_cas_u()}} for uploads,
\code{\link[=tar_cas_d]{tar_cas_d()}} for downloads, and
\code{\link[=tar_cas_l]{tar_cas_l()}} for listing keys.
}

\examples{
if (identical(Sys.getenv("TAR_EXAMPLES"), "true")) { # for CRAN
tar_dir({ # tar_dir() runs code from a temp dir for CRAN.
tar_script({
  library(targets)
  library(tarchetypes)
  repository <- tar_repository_cas_local("cas")
  write_file <- function(object) {
    writeLines(as.character(object), "file.txt")
    "file.txt"
  }
  list(
    tar_target(x, c(2L, 4L), repository = repository),
    tar_target(
      y,
      x,
      pattern = map(x),
      format = "qs",
      repository = repository
    ),
    tar_target(z, write_file(y), format = "file", repository = repository)
  )
})
tar_make()
tar_read(y)
tar_read(z)
list.files("cas")
tar_meta(any_of(c("x", "z")), fields = any_of("data"))
})
}
}
\seealso{
Other content-addressable storage: 
\code{\link{tar_repository_cas}()},
\code{\link{tar_repository_cas_local_gc}()}
}
\concept{content-addressable storage}
