% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/tprob.R
\name{tprob}
\alias{tprob}
\title{Nonparametric estimation of transition probabilities in
the illness-death model}
\usage{
tprob(formula, s, method = "AJ", conf = FALSE, conf.level = 0.95,
  conf.type = "linear", n.boot = 199, data, z.value, bw = "dpik",
  window = "gaussian", method.weights = "NW", cluster = FALSE,
  ncores = NULL, na.rm = TRUE)
}
\arguments{
\item{formula}{A \code{formula} object, which must have a \code{survIDM}
object as the response on the left of the \code{~} operator and, if desired,
a term on the right. The term may be a qualitative or quantitative variable.
Without covariates, the right hand side should be \code{~ 1}.}

\item{s}{The first time for obtaining estimates for the transition
probabilities. If missing, 0 will be used.}

\item{method}{The method used to compute the transition probabilities.
Possible options are \code{"AJ"}, \code{"LIDA"} \code{"LDM"}, \code{"PLDM"} and
\code{"IPCW"}. Defaults to \code{"AJ"}. The \code{"IPCW"} method
is recommended to obtain conditional transition probabilities (i.e., with a
quantitative term on the right hand side of formula).}

\item{conf}{Provides pointwise confidence bands. Defaults to \code{FALSE}.}

\item{conf.level}{Level of confidence. Defaults to 0.95 (corresponding to 95\%).}

\item{conf.type}{Method to compute the confidence intervals.
Transformation applied to compute confidence intervals. Possible choices
are \code{"linear"}, \code{"log"}, \code{"log-log"} and \code{"bootstrap"}.
Default is \code{"linear"}.}

\item{n.boot}{The number of bootstrap replicates to compute the variance
of the non-Markovian estimator. Default is 199.}

\item{data}{A data.frame including at least four columns named
\code{time1}, \code{event1}, \code{Stime} and \code{event}, which correspond
to disease free survival time, disease free survival indicator, time to death
or censoring, and death indicator, respectively.}

\item{z.value}{The value of the covariate on the right hand side of formula
at which the transition probabilities are computed. For quantitative
covariates, i.e. of class integer and numeric.}

\item{bw}{A single numeric value to compute a kernel density bandwidth.
Use \code{"dpik"} for the \pkg{KernSmooth} package based selector or \code{"np"}
for the \code{'npudensbw'} function of the \pkg{np} package.}

\item{window}{A character string specifying the desired kernel.
See details below for possible options. Defaults to \code{"gaussian"}
where the gaussian density kernel will be used.}

\item{method.weights}{A character string specifying the desired weights method.
Possible options are \code{"NW"} for the Nadaraya-Watson weights and \code{"LL"}
for local linear weights. Defaults to \code{"NW"}.}

\item{cluster}{A logical value. If \code{TRUE} (default), the bootstrap procedure
for the confidence intervals is parallelized. Note that there are
cases (e.g., a low number of bootstrap repetitions) that \R will gain in
performance through serial computation. \R takes time to distribute tasks
across the processors also it will need time for binding them all together
later on. Therefore, if the time for distributing and gathering pieces
together is greater than the time need for single-thread computing,
it does not worth parallelize.}

\item{ncores}{An integer value specifying the number of cores to be used in
the parallelized procedure. If \code{NULL} (default), the number of cores
to be used is equal to the number of cores of the machine - 1.}

\item{na.rm}{A logical value indicating whether NA values should
be stripped in the computation.}
}
\value{
An object of class \code{"survIDM"} and one of the following
five classes: \code{"AJ"}, \code{"LIDA"}, \code{"LMD"}, \code{"PLDM"} and
\code{"tpIPCW"}. Objects are implemented as a list with elements:

\item{est}{data.frame with estimates of the transition probabilities.}
\item{CI}{data.frame with the confidence intervals of the transition probabilities.}
\item{conf.level}{Level of confidence.}
\item{s}{The first time for obtaining estimates for the transition probabilities.}
\item{t}{The time for obtaining the estimates of transition probabilities.}
\item{conf}{logical; if \code{FALSE} (default) the pointwise confidence
bands are not given.}
\item{conf.type}{Type of the confidence interval.}
\item{callp}{The expression of the estimated probability.}
\item{Nlevels}{The number of levels of the covariate. Provides important
information when the covariate at the right hand side of formula
is of class factor.}
\item{levels}{The levels of the qualitative covariate
(if it is of class factor) on the right hand side of formula.}
\item{formula}{A formula object.}
\item{call}{A call object.}
}
\description{
This function is used to obtain nonparametric estimates of
the transition probabilities in the illness-death model.
}
\details{
Possible options for argument window are \code{"gaussian"},
\code{"epanechnikov"}, \code{"tricube"}, \code{"boxcar"},
\code{"triangular"}, \code{"quartic"} or \code{"cosine"}.
}
\examples{
# Aalen-Johansen
# Occupation Probabilities Pj(t)=Pij(0,t)

res <- tprob(survIDM(time1, event1, Stime, event) ~ 1, s = 0,
method = "AJ", conf = FALSE, data = colonIDM)

summary(res, time=365*1:6)
plot(res)


# Transition Probabilities Pij(t)=Pij(365,t)

# LIDA
res1 <- tprob(survIDM(time1, event1, Stime, event) ~ 1, s = 365,
method = "LIDA", conf = FALSE, data = colonIDM)

summary(res1, time=365*1:6)
plot(res1)
plot(res1, trans="01", ylim=c(0,0.15))

# Landmark (LDM)
res2 <- tprob(survIDM(time1, event1, Stime, event) ~ 1, s = 365,
method = "LDM", conf = FALSE, data = colonIDM)

summary(res2, time=365*1:6)
plot(res2)

# Presmoothed LDM
res3 <- tprob(survIDM(time1, event1, Stime, event) ~ 1, s = 365,
method = "PLDM", conf = FALSE, data = colonIDM)

summary(res3, time=365*1:6)
plot(res3)



# Conditional transition probabilities

#with factor
res4 <- tprob(survIDM(time1, event1, Stime, event) ~ factor(sex), s = 365,
method = "AJ", conf = FALSE, data = colonIDM)

summary(res4, time=365*1:6)
plot(res4, trans="02", ylim=c(0,0.5))

# with continuous covariate (IPCW)
res5 <- tprob(survIDM(time1, event1, Stime, event) ~ age, s = 365,
method = "IPCW", z.value = 48, conf = FALSE, data = colonIDM,
bw = "dpik", window = "gaussian", method.weights = "NW")

summary(res5, time=365*1:6)
plot(res5)


# Confidence intervals
res6 <- tprob(survIDM(time1, event1, Stime, event) ~ 1, s = 365,
method = "AJ", conf = TRUE, conf.level = 0.95,
conf.type = "log", data = colonIDM)

summary(res6, time=365*1:7)
plot(res6)
}
\references{
Aalen O. O., Johansen S. (1978) An Empirical Transition Matrix for
Nonhomogeneous Markov Chains Based on Censored Observations. Scandinavian
Journal of Statistics 5(3), 141--150.

Meira-Machado L. F., de Una-Alvarez J. and Cadarso-Suarez C. (2006).
Nonparametric estimation of transition probabilities in a non-Markov
illness-death model. Lifetime Data Anal 12(3), 325--344.

de Una-Alvarez J. and Meira-Machado L. (2015). Nonparametric estimation
of transition probabilities in a non-Markov illness-death model:
a comparative study. Biometrics 71, 364--375.
}
\author{
Luis Meira-Machado and Marta Sestelo.
}
