% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/CoBC.R
\name{coBC}
\alias{coBC}
\title{Train the Co-bagging model}
\usage{
coBC(x, y, bclassif = bClassifOneNN(), dist = "matrix", N = 3,
  min.amount = ceiling(length(which(is.na(y))) * 0.3), u = 100,
  max.iter = 50)
}
\arguments{
\item{x}{A object that can be coerced as matrix.
This object have various interpretations depending on the value set in \code{dist} argument.
See \code{dist} argument.}

\item{y}{A vector with the labels of training instances. In this vector the unlabeled instances
are specified with the value \code{NA}.}

\item{bclassif}{Base classifier specification. Default is \code{bClassifOneNN()}. For defining new base classifiers
see \code{\link{bClassif}}.}

\item{dist}{Distance information. Valid options are:
\itemize{
   \item \code{"matrix"}: this string indicates that \code{x} is a distance matrix.
   \item \emph{string}: the name of a distance method available in \code{proxy} package.
   In this case \code{x} is interpreted as a matrix of instances.
   \item \emph{function}: a function defined by the user that computes the distance between two vectors.
   This function is called passing the vectors in the firsts two arguments. If the function have others arguments,
   those arguments must be have default values.
   In this case \code{x} is interpreted as a matrix of instances.
}}

\item{N}{The number of classifiers used as committee members. All these classifiers are defined using
the description provided by \code{bClassif}. Default is 3.}

\item{min.amount}{Minimum number of unlabeled instances to stop the training process. When the size
of unlabeled training instances reaches this number the self-labeling process is stopped. Default is
0.3 * <number of unlabeled instances>.}

\item{u}{Number of unlabeled instances in the pool. Default is 100.}

\item{max.iter}{Maximum number of iterations to execute in the self-labeling process. Default is 50.}
}
\value{
The trained model stored in a list with the following named values:
\describe{
  \item{models}{A list with the models (classifiers) trained. See argument \code{N} for details.}
  \item{bclassif}{The value of \code{bclassif} argument.}
  \item{dist}{The value of \code{dist} argument.
  This value is optionally returned when the argument \code{dist} is different from "matrix".}
  \item{included.insts}{The indexes of the training instances used.
  This set is the union of the indexes used to train each model in \code{models}.
  These indexes are relative to \code{x} argument.}
  \item{indexes}{A list of vectors. Each vector indicates the indexes of the instances
  used to train the corresponding model in \code{models}. For example, the instances used to train
  \code{models[1]} are \code{x[included.insts[indexes[1]], ]}.}
  \item{tr.insts}{The training instances corresponding to the indexes returned in \code{included.insts}.
   This value is optionally returned when the \code{dist} argument is different from "matrix".}
  \item{labels.map}{An internal map used for the labels.}
}
}
\description{
Builds and trains a model to predict the label of instances,
according to Co-bagging algorithm.
}
\examples{
# This example is part of CoBC demo.
# Use demo(CoBC) to see all the examples.

## Load Wine data set
data(wine)

x <- wine[, -14] # instances without classes
y <- wine[, 14] # the classes
x <- scale(x) # scale the attributes

## Prepare data
set.seed(20)
# Use 50\% of instances for training
tra.idx <- sample(x = length(y), size = ceiling(length(y) * 0.5))
xtrain <- x[tra.idx,] # training instances
ytrain <- y[tra.idx]  # classes of training instances
# Use 70\% of train instances as unlabeled set
tra.na.idx <- sample(x = length(tra.idx), size = ceiling(length(tra.idx) * 0.7))
ytrain[tra.na.idx] <- NA # remove class information of unlabeled instances

# Use the other 50\% of instances for inductive testing
tst.idx <- setdiff(1:length(y), tra.idx)
xitest <- x[tst.idx,] # testing instances
yitest <- y[tst.idx] # classes of testing instances

## Example: Using the Euclidean distance in proxy package.
m <- coBC(xtrain, ytrain, dist = "Euclidean")
pred <- predict(m, xitest)
caret::confusionMatrix(table(pred, yitest))

}
\references{
Avrim Blum and Tom Mitchell.\cr
\emph{Combining labeled and unlabeled data with co-training.}\cr
In Eleventh Annual Conference on Computational Learning Theory, COLT’ 98, pages 92–100, New York, NY, USA, 1998. ACM.
ISBN 1-58113-057-0. doi: 10.1145/279943.279962.
}

