% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/raster.change.R
\name{raster.change}
\alias{raster.change}
\title{Raster change between two nominal rasters}
\usage{
raster.change(
  x,
  y,
  s = 3,
  stat = c("kappa", "t.test", "cor", "entropy", "cross-entropy", "divergence"),
  ...
)
}
\arguments{
\item{x}{A terra SpatRaster}

\item{y}{A terra SpatRaster for comparison to x}

\item{s}{Integer or matrix for defining Kernel, 
must be odd but not necessarily square}

\item{stat}{Statistic to use in comparison, please see details for 
options.}

\item{...}{Additional arguments passed to terra::focalPairs}
}
\value{
A terra SpatRaster layer containing one of the following layers:
\itemize{ 
\item   kappa         Kappa or Weighted Kappa statistic (if stat = "kappa")
\item   correlation   Paired t.test statistic  (if stat = "cor")
\item   entropy       Local entropy  (if stat = "entropy")
\item   divergence    Kullback-Leibler divergence (if stat = "divergence")
\item   cross.entropy Local Cross-entropy (if stat = "cross.entropy")
\item   t.test        Paired t.test statistic  (if stat = "t.test")
\item   p.value       p-value of the paired t.test statistic (if stat = "t.test")
 }
}
\description{
Compares two categorical rasters with a variety of 
             statistical options

This function provides a various statistics for comparing two classified maps. 
Valid options are:
\itemize{ 
\item   kappa - Cohen's Kappa 
\item   t.test - Two-tailed paired t-test 
\item   cor - Persons Correlation 
\item   entropy - Delta entropy 
\item   cross-entropy - Cross-entropy loss function 
\item   divergence - Kullback-Leibler divergence (relative entropy) 
}
}
\note{
Kappa and t-test values < 0 are reported as 0. For a weighted kappa, a matrix must 
be provided that correspond to the pairwise weights for all values in both rasters. 
Delta entropy is derived by calculating Shannon's on each focal window then 
differencing  them (e(x) - e(y)). The s argument can be a single scalar, defining
a symmetrical kernel, two scalers defining the dimensions of the kernel eg., c(3,5)
or a matrix defining the kernel say, resulting from terra::focalMat
}
\examples{
\donttest{
 library(sf) 
 library(terra) 
  
 e <- ext(179407.8, 181087.9, 331134.4, 332332.1)
 r1 <- rast(e, resolution=20)
   r1[] <- sample(1:5, ncell(r1), replace=TRUE)
 r2 <- rast(e, resolution=20)
   r2[] <- sample(1:5, ncell(r2), replace=TRUE)
 	  
 d = 5 # kernel    
 ( r.kappa <- raster.change(r1, r2, s = d) )   
 ( r.ttest <- raster.change(r1, r2, s = d, stat="t.test") )
 ( r.ent <- raster.change(r1, r2, s = d, stat="entropy") )   
 ( r.cor <- raster.change(r1, r2, s = d, stat="cor") )
 ( r.ce <- raster.change(r1, r2, s = d, stat = "cross-entropy") )
 ( r.kl <- raster.change(r1, r2, s = d, stat = "divergence") )	
       
   opar <- par(no.readonly=TRUE)
   par(mfrow=c(3,2))
     plot(r.kappa, main="Kappa")
     plot(r.ttest[[1]], main="Paired t-test")
     plot(r.ent, main="Delta Entropy")
     plot(r.cor, main="Rank Correlation")
     plot(r.kl, main="Kullback-Leibler")
     plot(r.ce, main="cross-entropy")
   par(opar) 
}

}
\references{
Cohen, J. (1960). A coefficient of agreement for nominal scales. Educational  
  and Psychological Measurement, 20:37-46

McHugh M.L. (2012) Interrater reliability: the kappa statistic. 
  Biochemia medica, 22(3):276–282.

Kullback, S., R.A. Leibler (1951). On information and sufficiency. Annals of 
   Mathematical Statistics. 22(1):79–86
}
\author{
Jeffrey S. Evans  <jeffrey_evans@tnc.org>
}
