% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/spatial.select.R
\name{spatial.select}
\alias{spatial.select}
\title{Spatial Select}
\usage{
spatial.select(
  x,
  y = NULL,
  distance = NULL,
  predicate = c("intersection", "intersect", "contains", "covers", "touches",
    "proximity", "contingency"),
  neighbors = c("queen", "rook")
)
}
\arguments{
\item{x}{An sp or sf polygon(s) object that defines the spatial query}

\item{y}{A sp or sf feature class that will be subset by the query of x}

\item{distance}{A proximity distance of features to select (within distance)}

\item{predicate}{Spatial predicate for intersection}

\item{neighbors}{If predicate = "contingency" type of neighbors options are
c("queen", "rook")}
}
\value{
An sf object representing a subset of y based on the spatial query of x or,
if predicate = contingency a sparse matrix representing neighbor indexes
}
\description{
Performs a spatial select (feature subset) between a polygon(s) and other
feature class
}
\details{
\itemize{
\item intersection  Create a spatial intersection between two features
\item intersect     Boolean evaluation of features intersecting
\item contains      Boolean evaluation of x containing y
\item covers        Boolean evaluation of x covering y
\item touches       Boolean evaluation of x touching y
\item proximity     Evaluation of distance-based proximity of x to y (x and y can be the same)
\item contingency   Evaluation of polygon contingency (eg., 1st, 2nd order)
}
}
\note{
Performs a spatial select of features based on an overlay of a polygon (x),
which can represent multiple features, and a polygon, point or line feature
classes (y). User can specify a partial or complete intersection, using within
argument, or within a distance, using distance argument, predicated on the
query polygon. This function is similar to ArcGIS/Pro spatial select. Please note
that for point to point neighbor selections use the knn function.
Valid spatial predicates include: intersect, touches, covers, contains, proximity
and contingency. See DE-9IM topology model for detailed information on data predicates.
}
\examples{
library(sf)
 if(require(sp, quietly = TRUE)) {
   data(meuse, package = "sp")
   meuse <- st_as_sf(meuse, coords = c("x", "y"), crs = 28992, 
                     agr = "constant")

spolys <- hexagons(meuse, res=100)
  spolys$ID <- 1:nrow(spolys)
    p <- st_as_sf(st_sample(spolys, 500))
	  p$PTID <- 1:nrow(p) 
	  sf::st_geometry(p) <- "geometry"

  plot(st_geometry(spolys), main="all data")
    plot(st_geometry(p), pch=20, add=TRUE)
	
sub.int <- spatial.select(p, spolys, predicate = "intersect")
  plot(st_geometry(sub.int), main="intersects")
    plot(st_geometry(p), pch=20, add=TRUE)	

sub.prox <- spatial.select(p, spolys, distance=100, predicate = "proximity")	
  plot(st_geometry(sub.int), main="intersects")
    plot(st_geometry(p), pch=20, add=TRUE)

# For rook or queen polygon contingency 	
plot( spolys <- sf::st_make_grid(sf::st_sfc(sf::st_point(c(0,0)), 
                 sf::st_point(c(3,3))), n = c(3,3)) )
  
spatial.select(x=spolys, predicate = "contingency")
spatial.select(spolys, predicate = "contingency", neighbors = "rook") 

}
}
\seealso{
\code{\link[sf]{st_intersection}} for details on intersection predicate

\code{\link[sf]{st_intersects}} for details on intersect predicate

\code{\link[sf]{st_contains}} for details on contain predicate

\code{\link[sf]{st_covers}} for details on covers predicate

\code{\link[sf]{st_touches}} for details on touches predicate

\code{\link[sf]{st_is_within_distance}} for details on proximity predicate

\url{https://en.wikipedia.org/wiki/DE-9IM} for details on DE-9IM topology model
}
\author{
Jeffrey S. Evans  \href{mailto:jeffrey_evans@tnc.org}{jeffrey_evans@tnc.org}
}
