% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/eval_design_mc.R
\name{eval_design_mc}
\alias{eval_design_mc}
\title{Monte Carlo Power Evaluation for Experimental Designs}
\usage{
eval_design_mc(
  design,
  model = NULL,
  alpha = 0.05,
  blocking = NULL,
  nsim = 1000,
  glmfamily = "gaussian",
  calceffect = TRUE,
  effect_anova = TRUE,
  varianceratios = NULL,
  rfunction = NULL,
  anticoef = NULL,
  firth = FALSE,
  effectsize = 2,
  contrasts = contr.sum,
  parallel = FALSE,
  adjust_alpha_inflation = FALSE,
  detailedoutput = FALSE,
  progress = TRUE,
  advancedoptions = NULL,
  ...
)
}
\arguments{
\item{design}{The experimental design. Internally, \code{eval_design_mc} rescales each numeric column
to the range [-1, 1].}

\item{model}{The model used in evaluating the design. If this is missing and the design
was generated with skpr, the generating model will be used. It can be a subset of the model used to
generate the design, or include higher order effects not in the original design generation. It cannot include
factors that are not present in the experimental design.}

\item{alpha}{Default `0.05`. The type-I error. p-values less than this will be counted as significant.}

\item{blocking}{Default `NULL`. If `TRUE`, \code{eval_design_mc} will look at the rownames (or blocking columns) to determine blocking structure. Default FALSE.}

\item{nsim}{Default `1000`. The number of Monte Carlo simulations to perform.}

\item{glmfamily}{Default `gaussian`. String indicating the family of distribution for the `glm` function
("gaussian", "binomial", "poisson", or "exponential").}

\item{calceffect}{Default `TRUE`. Whether to calculate effect power. This calculation is more expensive than parameter power,
so turned off (if not needed) can greatly speed up calculation time.}

\item{effect_anova}{Default `TRUE`, whether to a Type-III Anova or a likelihood ratio test to calculate effect power.
If `TRUE`, effect power will be calculated using a Type-III Anova (using the car package) and a Wald test. If `FALSE`,
a likelihood ratio test (using a reduced model for each effect) will performed using the `lmtest` package. If `firth = TRUE`,
this will be set to `FALSE` automatically.}

\item{varianceratios}{Default `NULL`. The ratio of the whole plot variance to the run-to-run variance.
If not specified during design generation, this will default to 1. For designs with more than one subplot
this ratio can be a vector specifying the variance ratio for each subplot (comparing to the run-to-run variance).
Otherwise, it will use a single value for all strata.}

\item{rfunction}{Default `NULL`.Random number generator function for the response variable. Should be a function of the form f(X, b, delta), where X is the
model matrix, b are the anticipated coefficients, and delta is a vector of blocking errors. Typically something like rnorm(nrow(X), X * b + delta, 1).
You only need to specify this if you do not like the default behavior described below.}

\item{anticoef}{Default `NULL`.The anticipated coefficients for calculating the power. If missing, coefficients
will be automatically generated based on the \code{effectsize} argument.}

\item{firth}{Default `FALSE`. Whether to apply the firth correction (via the `mbest` package) to a logistic regression. This
setting also automatically sets `effect_lr = TRUE`.}

\item{effectsize}{Helper argument to generate anticipated coefficients. See details for more info.
If you specify \code{anticoef}, \code{effectsize} will be ignored.}

\item{contrasts}{Default \code{contr.sum}. The contrasts to use for categorical factors. If the user has specified their own contrasts
for a categorical factor using the contrasts function, those will be used. Otherwise, skpr will use contr.sum.}

\item{parallel}{Default `FALSE`. If `TRUE`, uses all cores available to speed up computation. WARNING: This can slow down computation if nonparallel time to complete the computation is less than a few seconds.}

\item{adjust_alpha_inflation}{Default `FALSE`. If `TRUE`, this will run the simulation twice:
first to calculate the empirical distribution of p-values under the null hypothesis and find
the true Type-I error cutoff that corresponds to the desired Type-I error rate,
and then again given effect size to calculate power values.}

\item{detailedoutput}{Default `FALSE`. If `TRUE`, return additional information about evaluation in results.}

\item{progress}{Default `TRUE`. Whether to include a progress bar.}

\item{advancedoptions}{Default `NULL`. Named list of advanced options. `advancedoptions$anovatype` specifies the Anova type in the car package (default type `III`),
user can change to type `II`). `advancedoptions$anovatest` specifies the test statistic if the user does not want a `Wald` test--other options are likelyhood-ratio `LR` and F-test `F`.
`advancedoptions$progressBarUpdater` is a function called in non-parallel simulations that can be used to update external progress bar.`advancedoptions$GUI` turns off some warning messages when in the GUI.
If `advancedoptions$save_simulated_responses = TRUE`, the dataframe will have an attribute `simulated_responses` that contains the simulated responses from the power evaluation.}

\item{...}{Additional arguments.}
}
\value{
A data frame consisting of the parameters and their powers, with supplementary information
stored in the data frame's attributes. The parameter estimates from the simulations are stored in the "estimates"
attribute. The "modelmatrix" attribute contains the model matrix that was used for power evaluation, and
also provides the encoding used for categorical factors. If you want to specify the anticipated
coefficients manually, do so in the order the parameters appear in the model matrix.
}
\description{
Evaluates the power of an experimental design, given the run matrix and the
statistical model to be fit to the data, using monte carlo simulation. Simulated data is fit using a
generalized linear model
and power is estimated by the fraction of times a parameter is significant.
Returns a data frame of parameter powers.
}
\details{
Evaluates the power of a design with Monte Carlo simulation. Data is simulated and then fit
with a generalized linear model, and the fraction of simulations in which a parameter
is significant (its p-value, according to the fit function used, is less than the specified \code{alpha})
is the estimate of power for that parameter.

First, if \code{blocking = TURE}, the random noise from blocking is generated with \code{rnorm}.
Each block gets a single sample of Gaussian random noise, with a variance as specified in
\code{varianceratios},
and that sample is copied to each run in the block. Then, \code{rfunction} is called to generate a simulated
response for each run of the design, and the data is fit using the appropriate fitting function.
The functions used to simulate the data and fit it are determined by the \code{glmfamily}
and \code{blocking} arguments
as follows. Below, X is the model matrix, b is the anticipated coefficients, and d
is a vector of blocking noise (if \code{blocking = FALSE} then d = 0):

\tabular{llrr}{
\bold{glmfamily}      \tab \bold{blocking} \tab \bold{rfunction} \tab \bold{fit} \cr
"gaussian"     \tab F        \tab \code{rnorm(mean = X \%*\% b + d, sd = 1)}        \tab \code{lm}         \cr
"gaussian"     \tab T        \tab \code{rnorm(mean = X \%*\% b + d, sd = 1)}        \tab \code{lme4::lmer} \cr
"binomial"     \tab F        \tab \code{rbinom(prob = 1/(1+exp(-(X \%*\% b + d))))} \tab \code{glm(family = "binomial")}  \cr
"binomial"     \tab T        \tab \code{rbinom(prob = 1/(1+exp(-(X \%*\% b + d))))} \tab \code{lme4::glmer(family = "binomial")} \cr
"poisson"      \tab F        \tab \code{rpois(lambda = exp((X \%*\% b + d)))}       \tab \code{glm(family = "poisson")}         \cr
"poisson"      \tab T        \tab \code{rpois(lambda = exp((X \%*\% b + d)))}       \tab \code{lme4::glmer(family = "poisson")} \cr
"exponential"  \tab F        \tab \code{rexp(rate = exp(-(X \%*\% b + d)))}            \tab \code{glm(family = Gamma(link = "log"))} \cr
"exponential"  \tab T        \tab \code{rexp(rate = exp(-(X \%*\% b + d)))}            \tab \code{lme4:glmer(family = Gamma(link = "log"))} \cr
}
Note that the exponential random generator uses the "rate" parameter, but \code{skpr} and \code{glm} use
the mean value parameterization (= 1 / rate), hence the minus sign above. Also note that
the gaussian model assumes a root-mean-square error of 1.

Power is dependent on the anticipated coefficients. You can specify those directly with the \code{anticoef}
argument, or you can use the \code{effectsize} argument to specify an effect size and \code{skpr} will auto-generate them.
You can provide either a length-1 or length-2 vector. If you provide a length-1 vector, the anticipated
coefficients will be half of \code{effectsize}; this is equivalent to saying that the \emph{linear predictor}
(for a gaussian model, the mean response; for a binomial model, the log odds ratio; for an exponential model,
the log of the mean value; for a poisson model, the log of the expected response)
changes by \code{effectsize} when a continuous factor goes from its lowest level to its highest level. If you provide a
length-2 vector, the anticipated coefficients will be set such that the \emph{mean response} (for
a gaussian model, the mean response; for a binomial model, the probability; for an exponential model, the mean
response; for a poisson model, the expected response) changes from
\code{effectsize[1]} to \code{effectsize[2]} when a factor goes from its lowest level to its highest level, assuming
that the other factors are inactive (their x-values are zero).

The effect of a length-2 \code{effectsize} depends on the \code{glmfamily} argument as follows:

For \code{glmfamily = 'gaussian'}, the coefficients are set to \code{(effectsize[2] - effectsize[1]) / 2}.

For \code{glmfamily = 'binomial'}, the intercept will be
\code{1/2 * log(effectsize[1] * effectsize[2] / (1 - effectsize[1]) / (1 - effectsize[2]))},
and the other coefficients will be
\code{1/2 * log(effectsize[2] * (1 - effectsize[1]) / (1 - effectsize[2]) / effectsize[1])}.

For \code{glmfamily = 'exponential'} or \code{'poisson'},
the intercept will be
\code{1 / 2 * (log(effectsize[2]) + log(effectsize[1]))},
and the other coefficients will be
\code{1 / 2 * (log(effectsize[2]) - log(effectsize[1]))}.
}
\examples{
#We first generate a full factorial design using expand.grid:
factorialcoffee = expand.grid(cost = c(-1, 1),
                              type = as.factor(c("Kona", "Colombian", "Ethiopian", "Sumatra")),
                              size = as.factor(c("Short", "Grande", "Venti")))
if(skpr:::run_documentation()) {
#And then generate the 21-run D-optimal design using gen_design.
designcoffee = gen_design(factorialcoffee,
                         model = ~cost + type + size, trials = 21, optimality = "D")
}
if(skpr:::run_documentation()) {
#To evaluate this design using a normal approximation, we just use eval_design
#(here using the default settings for contrasts, effectsize, and the anticipated coefficients):

eval_design(design = designcoffee, model = ~cost + type + size, 0.05)
}
if(skpr:::run_documentation()) {
#To evaluate this design with a Monte Carlo method, we enter the same information
#used in eval_design, with the addition of the number of simulations "nsim" and the distribution
#family used in fitting for the glm "glmfamily". For gaussian, binomial, exponential, and poisson
#families, a default random generating function (rfunction) will be supplied. If another glm
#family is used or the default random generating function is not adequate, a custom generating
#function can be supplied by the user. Like in `eval_design()`, if the model isn't entered, the
#model used in generating the design will be used.

eval_design_mc(designcoffee, nsim = 100, glmfamily = "gaussian")
}
if(skpr:::run_documentation()) {
#We see here we generate approximately the same parameter powers as we do
#using the normal approximation in eval_design. Like eval_design, we can also change
#effectsize to produce a different signal-to-noise ratio:

eval_design_mc(design = designcoffee, nsim = 100,
                       glmfamily = "gaussian", effectsize = 1)
}
if(skpr:::run_documentation()) {
#Like eval_design, we can also evaluate the design with a different model than
#the one that generated the design.
eval_design_mc(design = designcoffee, model = ~cost + type, alpha = 0.05,
              nsim = 100, glmfamily = "gaussian")
}
if(skpr:::run_documentation()) {
#And here it is evaluated with additional interactions included:
eval_design_mc(design = designcoffee, model = ~cost + type + size + cost * type, 0.05,
              nsim = 100, glmfamily = "gaussian")
}
if(skpr:::run_documentation()) {
#We can also set "parallel = TRUE" to use all the cores available to speed up
#computation.
eval_design_mc(design = designcoffee, nsim = 10000,
                       glmfamily = "gaussian", parallel = TRUE)
}
if(skpr:::run_documentation()) {
#We can also evaluate split-plot designs. First, let us generate the split-plot design:

factorialcoffee2 = expand.grid(Temp = c(1, -1),
                               Store = as.factor(c("A", "B")),
                               cost = c(-1, 1),
                               type = as.factor(c("Kona", "Colombian", "Ethiopian", "Sumatra")),
                               size = as.factor(c("Short", "Grande", "Venti")))

vhtcdesign = gen_design(factorialcoffee2,
                       model = ~Store, trials = 6, varianceratio = 1)
htcdesign = gen_design(factorialcoffee2, model = ~Store + Temp, trials = 18,
                       splitplotdesign = vhtcdesign, blocksizes = rep(3, 6), varianceratio = 1)
splitplotdesign = gen_design(factorialcoffee2,
                            model = ~Store + Temp + cost + type + size, trials = 54,
                            splitplotdesign = htcdesign, blocksizes = rep(3, 18),
                            varianceratio = 1)

#Each block has an additional noise term associated with it in addition to the normal error
#term in the model. This is specified by a vector specifying the additional variance for
#each split-plot level. This is equivalent to specifying a variance ratio of one between
#the whole plots and the run-to-run variance for gaussian models.

#Evaluate the design. Note the decreased power for the blocking factors.
eval_design_mc(splitplotdesign, blocking = TRUE, nsim = 100,
                       glmfamily = "gaussian", varianceratios = c(1, 1, 1))
}
if(skpr:::run_documentation()) {
#We can also use this method to evaluate designs that cannot be easily
#evaluated using normal approximations. Here, we evaluate a design with a binomial response and see
#whether we can detect the difference between each factor changing whether an event occurs
#70\% of the time or 90\% of the time.

factorialbinom = expand.grid(a = c(-1, 1), b = c(-1, 1))
designbinom = gen_design(factorialbinom, model = ~a + b, trials = 90, optimality = "D")

eval_design_mc(designbinom, ~a + b, alpha = 0.2, nsim = 100, effectsize = c(0.7, 0.9),
              glmfamily = "binomial")
}
if(skpr:::run_documentation()) {
#We can also use this method to determine power for poisson response variables.
#Generate the design:

factorialpois = expand.grid(a = as.numeric(c(-1, 0, 1)), b = c(-1, 0, 1))
designpois = gen_design(factorialpois, ~a + b, trials = 70, optimality = "D")

#Evaluate the power:

eval_design_mc(designpois, ~a + b, 0.05, nsim = 100, glmfamily = "poisson",
               anticoef = log(c(0.2, 2, 2)))
}

#The coefficients above set the nominal value -- that is, the expected count
#when all inputs = 0 -- to 0.2 (from the intercept), and say that each factor
#changes this count by a factor of 4 (multiplied by 2 when x= +1, and divided by 2 when x = -1).
#Note the use of log() in the anticipated coefficients.
}
