% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/simStateSpace-sim-ssm-ou-vary.R
\name{SimSSMOUVary}
\alias{SimSSMOUVary}
\title{Simulate Data from an Ornstein–Uhlenbeck Model
using a State Space Model Parameterization
for n > 1 Individuals (Varying Parameters)}
\usage{
SimSSMOUVary(
  n,
  mu0,
  sigma0_sqrt,
  mu,
  phi,
  sigma_sqrt,
  nu,
  lambda,
  theta_sqrt,
  delta_t,
  time,
  burn_in
)
}
\arguments{
\item{n}{Positive integer.
Number of individuals.}

\item{mu0}{Numeric vector.
Mean of initial latent variable values
(\eqn{\boldsymbol{\mu}_{\boldsymbol{\eta} \mid 0}}).}

\item{sigma0_sqrt}{Numeric matrix.
Cholesky decomposition of the covariance matrix
of initial latent variable values
(\eqn{\boldsymbol{\Sigma}_{\boldsymbol{\eta} \mid 0}}).}

\item{mu}{List of numeric vectors.
The long-term mean or equilibrium level
(\eqn{\boldsymbol{\mu}}).}

\item{phi}{List of numeric matrices.
The rate of mean reversion,
determining how quickly the variable returns to its mean
(\eqn{\boldsymbol{\Phi}}).}

\item{sigma_sqrt}{List of numeric matrices.
Cholesky decomposition of the matrix of volatility
or randomness in the process
(\eqn{\boldsymbol{\Sigma}}).}

\item{nu}{Numeric vector.
Vector of intercepts for the measurement model
(\eqn{\boldsymbol{\nu}}).}

\item{lambda}{Numeric matrix.
Factor loading matrix linking the latent variables
to the observed variables
(\eqn{\boldsymbol{\Lambda}}).}

\item{theta_sqrt}{Numeric matrix.
Cholesky decomposition of the measurement error covariance matrix
(\eqn{\boldsymbol{\Theta}}).}

\item{delta_t}{Numeric.
Time interval (\eqn{\delta_t}).}

\item{time}{Positive integer.
Number of time points to simulate.}

\item{burn_in}{Positive integer.
Number of burn-in points to exclude before returning the results.}
}
\value{
Returns a list of length \code{n}.
Each element is a list with the following elements:
\itemize{
\item \code{y}: A \code{t} by \code{k} matrix of values for the manifest variables.
\item \code{eta}: A \code{t} by \code{p} matrix of values for the latent variables.
\item \code{time}: A vector of discrete time points from 1 to \code{t}.
\item \code{id}: A vector of ID numbers of length \code{t}.
\item \code{n}: Number of individuals.
}
}
\description{
This function simulates data from an Ornstein–Uhlenbeck model
using a state space model parameterization
for \code{n > 1} individuals.
In this model,
the parameters can vary across individuals.
}
\details{
Parameters can vary across individuals
by providing a list of parameter values.
If the length of any of the parameters
(\code{mu0},
\code{sigma0_sqrt},
\code{mu},
\code{phi},
\code{sigma_sqrt},
\code{nu},
\code{lambda},
\code{theta_sqrt})
is less the \code{n},
the function will cycle through the available values.
}
\examples{
# prepare parameters
# In this example, phi varies across individuals
set.seed(42)
p <- k <- 2
iden <- diag(p)
iden_sqrt <- chol(iden)
n <- 5
mu0 <- list(c(-3.0, 1.5))
sigma0_sqrt <- list(iden_sqrt)
mu <- list(c(5.76, 5.18))
phi <- list(
  as.matrix(Matrix::expm(diag(x = -0.1, nrow = k))),
  as.matrix(Matrix::expm(diag(x = -0.2, nrow = k))),
  as.matrix(Matrix::expm(diag(x = -0.3, nrow = k))),
  as.matrix(Matrix::expm(diag(x = -0.4, nrow = k))),
  as.matrix(Matrix::expm(diag(x = -0.5, nrow = k)))
)
sigma_sqrt <- list(
  chol(
    matrix(data = c(2.79, 0.06, 0.06, 3.27), nrow = p)
  )
)
nu <- list(rep(x = 0, times = k))
lambda <- list(diag(k))
theta_sqrt <- list(chol(diag(x = 0.50, nrow = k)))
delta_t <- 0.10
time <- 50
burn_in <- 0

ssm <- SimSSMOUVary(
  n = n,
  mu0 = mu0,
  sigma0_sqrt = sigma0_sqrt,
  mu = mu,
  phi = phi,
  sigma_sqrt = sigma_sqrt,
  nu = nu,
  lambda = lambda,
  theta_sqrt = theta_sqrt,
  delta_t = delta_t,
  time = time,
  burn_in = burn_in
)

str(ssm)

}
\references{
Chow, S.-M., Losardo, D., Park, J., & Molenaar, P. C. M. (2023).
Continuous-time dynamic models: Connections to structural equation models
and other discrete-time models.
In R. H. Hoyle (Ed.), Handbook of structural equation modeling (2nd ed.).
The Guilford Press.

Uhlenbeck, G. E., & Ornstein, L. S. (1930).
On the theory of the brownian motion.
\emph{Physical Review}, \emph{36}(5), 823–841.
\doi{10.1103/physrev.36.823}
}
\seealso{
Other Simulation of State Space Models Data Functions: 
\code{\link{OU2SSM}()},
\code{\link{Sim2Matrix}()},
\code{\link{SimSSM0Fixed}()},
\code{\link{SimSSM0Vary}()},
\code{\link{SimSSM0}()},
\code{\link{SimSSMOUFixed}()},
\code{\link{SimSSMOU}()},
\code{\link{SimSSMVARFixed}()},
\code{\link{SimSSMVARVary}()},
\code{\link{SimSSMVAR}()}
}
\author{
Ivan Jacob Agaloos Pesigan
}
\concept{Simulation of State Space Models Data Functions}
\keyword{ou}
\keyword{sim}
\keyword{simStateSpace}
