% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/input-select.R
\name{varSelectInput}
\alias{varSelectInput}
\alias{varSelectizeInput}
\title{Select variables from a data frame}
\usage{
varSelectInput(
  inputId,
  label,
  data,
  selected = NULL,
  multiple = FALSE,
  selectize = TRUE,
  width = NULL,
  size = NULL
)

varSelectizeInput(inputId, ..., options = NULL, width = NULL)
}
\arguments{
\item{inputId}{The \code{input} slot that will be used to access the value.}

\item{label}{Display label for the control, or \code{NULL} for no label.}

\item{data}{A data frame. Used to retrieve the column names as choices for a \code{\link[=selectInput]{selectInput()}}}

\item{selected}{The initially selected value (or multiple values if \code{multiple = TRUE}). If not specified then defaults to the first value for
single-select lists and no values for multiple select lists.}

\item{multiple}{Is selection of multiple items allowed?}

\item{selectize}{Whether to use \pkg{selectize.js} or not.}

\item{width}{The width of the input, e.g. \code{'400px'}, or \code{'100\%'};
see \code{\link[=validateCssUnit]{validateCssUnit()}}.}

\item{size}{Number of items to show in the selection box; a larger number
will result in a taller box. Not compatible with \code{selectize=TRUE}.
Normally, when \code{multiple=FALSE}, a select input will be a drop-down list,
but when \code{size} is set, it will be a box instead.}

\item{...}{Arguments passed to \code{varSelectInput()}.}

\item{options}{A list of options. See the documentation of \pkg{selectize.js}(\url{https://selectize.dev/docs/usage})
for possible options (character option values inside \code{\link[base:AsIs]{base::I()}} will
be treated as literal JavaScript code; see \code{\link[=renderDataTable]{renderDataTable()}}
for details).}
}
\value{
A variable select list control that can be added to a UI definition.
}
\description{
Create a select list that can be used to choose a single or multiple items
from the column names of a data frame.
}
\details{
By default, \code{varSelectInput()} and \code{selectizeInput()} use the
JavaScript library \pkg{selectize.js}
(\url{https://selectize.dev/}) to instead of the basic
select input element. To use the standard HTML select input element, use
\code{selectInput()} with \code{selectize=FALSE}.
}
\note{
The variable selectize input created from \code{varSelectizeInput()} allows
deletion of the selected option even in a single select input, which will
return an empty string as its value. This is the default behavior of
\pkg{selectize.js}. However, the selectize input created from
\code{selectInput(..., selectize = TRUE)} will ignore the empty string
value when it is a single choice input and the empty string is not in the
\code{choices} argument. This is to keep compatibility with
\code{selectInput(..., selectize = FALSE)}.
}
\section{Server value}{

The resulting server \code{input} value will be returned as:
\itemize{
\item A symbol if \code{multiple = FALSE}. The \code{input} value should be
used with rlang's \code{\link[rlang:injection-operator]{rlang::!!()}}. For example,
\code{ggplot2::aes(!!input$variable)}.
\item A list of symbols if \code{multiple = TRUE}. The \code{input} value
should be used with rlang's \code{\link[rlang:splice-operator]{rlang::!!!()}} to expand
the symbol list as individual arguments. For example,
\code{dplyr::select(mtcars, !!!input$variabls)} which is
equivalent to \code{dplyr::select(mtcars, !!input$variabls[[1]], !!input$variabls[[2]], ..., !!input$variabls[[length(input$variabls)]])}.
}
}

\examples{

## Only run examples in interactive R sessions
if (interactive()) {

library(ggplot2)

# single selection
shinyApp(
  ui = fluidPage(
    varSelectInput("variable", "Variable:", mtcars),
    plotOutput("data")
  ),
  server = function(input, output) {
    output$data <- renderPlot({
      ggplot(mtcars, aes(!!input$variable)) + geom_histogram()
    })
  }
)


# multiple selections
\dontrun{
shinyApp(
 ui = fluidPage(
   varSelectInput("variables", "Variable:", mtcars, multiple = TRUE),
   tableOutput("data")
 ),
 server = function(input, output) {
   output$data <- renderTable({
      if (length(input$variables) == 0) return(mtcars)
      mtcars \%>\% dplyr::select(!!!input$variables)
   }, rownames = TRUE)
 }
)}

}
}
\seealso{
\code{\link[=updateSelectInput]{updateSelectInput()}}

Other input elements: 
\code{\link{actionButton}()},
\code{\link{checkboxGroupInput}()},
\code{\link{checkboxInput}()},
\code{\link{dateInput}()},
\code{\link{dateRangeInput}()},
\code{\link{fileInput}()},
\code{\link{numericInput}()},
\code{\link{passwordInput}()},
\code{\link{radioButtons}()},
\code{\link{selectInput}()},
\code{\link{sliderInput}()},
\code{\link{submitButton}()},
\code{\link{textAreaInput}()},
\code{\link{textInput}()}
}
\concept{input elements}
