% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/sghmccv.R
\name{sghmccv}
\alias{sghmccv}
\title{Stochastic Gradient Hamiltonian Monte Carlo with Control Variates}
\usage{
sghmccv(logLik, dataset, params, stepsize, optStepsize, logPrior = NULL,
  minibatchSize = 0.01, alpha = 0.01, L = 5L, nIters = 10^4L,
  nItersOpt = 10^4L, verbose = TRUE, seed = NULL)
}
\arguments{
\item{logLik}{function which takes parameters and dataset 
(list of TensorFlow variables and placeholders respectively) as input. 
It should return a TensorFlow expression which defines the log likelihood of the model.}

\item{dataset}{list of numeric R arrays which defines the datasets for the problem.
The names in the list should correspond to those referred to in the logLik and logPrior functions}

\item{params}{list of numeric R arrays which define the starting point of each parameter.
The names in the list should correspond to those referred to in the logLik and logPrior functions}

\item{stepsize}{list of numeric values corresponding to the SGLD stepsizes for each parameter
The names in the list should correspond to those in params.
Alternatively specify a single numeric value to use that stepsize for all parameters.}

\item{optStepsize}{numeric value specifying the stepsize for the optimization 
to find MAP estimates of parameters. The TensorFlow GradientDescentOptimizer is used.}

\item{logPrior}{optional. Default uninformative improper prior.
Function which takes parameters (list of TensorFlow variables) as input.
The function should return a TensorFlow tensor which defines the log prior of the model.}

\item{minibatchSize}{optional. Default 0.01.
Numeric or integer value that specifies amount of dataset to use at each iteration 
either as proportion of dataset size (if between 0 and 1) or actual magnitude (if an integer).}

\item{alpha}{optional. Default 0.01. 
List of numeric values corresponding to the SGHMC momentum tuning constants
(\eqn{\alpha} in the original paper). One value should be given 
for each parameter in params, the names should correspond to those in params.
Alternatively specify a single float to specify that value for all parameters.}

\item{L}{optional. Default 5L. Integer specifying the trajectory parameter of the simulation, 
as defined in the main reference.}

\item{nIters}{optional. Default 10^4L. Integer specifying number of iterations to perform.}

\item{nItersOpt}{optional. Default 10^4L. 
Integer specifying number of iterations of initial optimization to perform.}

\item{verbose}{optional. Default TRUE. Boolean specifying whether to print algorithm progress}

\item{seed}{optional. Default NULL. Numeric seed for random number generation. The default
does not declare a seed for the TensorFlow session.}
}
\value{
Returns list of arrays for each parameter containing the MCMC chain.
 Dimension of the form (nIters,paramDim1,paramDim2,...)
}
\description{
Simulates from the posterior defined by the functions logLik and logPrior using 
 stochastic gradient Hamiltonian Monte Carlo with an improved gradient estimate 
 that is calculated using control variates. 
 Currently we use the approximation \eqn{\hat \beta = 0}, 
 as used in the simulations by the original reference. 
 This will be changed in future implementations.
}
\examples{
\dontrun{
# Simulate from a Normal Distribution with uninformative prior
dataset = list("x" = rnorm(1000))
params = list("theta" = 0)
logLik = function(params, dataset) { 
    distn = tf$distributions$Normal(params$theta, 1)
    return(tf$reduce_sum(distn$log_prob(dataset$x)))
}
stepsize = list("theta" = 1e-5)
optStepsize = 1e-1
output = sghmccv(logLik, dataset, params, stepsize, optStepsize)
}
}
\references{
\itemize{
 \item \href{https://arxiv.org/pdf/1706.05439.pdf}{
 Baker, J., Fearnhead, P., Fox, E. B., and Nemeth, C. (2017).
 Control variates for stochastic gradient MCMC. ArXiv preprint arXiv:1706.05439.}
 \item \href{https://arxiv.org/pdf/1402.4102v2.pdf}{
 Chen, T., Fox, E. B., and Guestrin, C. (2014). Stochastic gradient Hamiltonian Monte Carlo. 
 In ICML (pp. 1683-1691).}}
}
