#' Interaction Function
#'
#' \code{interaction_term} creates interaction measurement items by applying orthogonal, product indicator, or two stage approach.
#'
#' This function automatically generates interaction measurement items for a PLS SEM.
#'
#' @param iv The independent variable that is subject to moderation.
#' @param moderator The moderator variable.
#' @param method The method to generate the estimated interaction term with a default of `two_stage`.
#' @param weights The weighting mode of the interaction items with default of `modeA`.
#'
#' Interaction Combinations as generated by the \code{interaction} or
#' \code{interaction_term} methods.
#'
#' @usage
#'
#' interaction_term(iv, moderator, method, weights)
#'
#' @examples
#' data(mobi)
#'
#' # seminr syntax for creating measurement model
#' mobi_mm <- constructs(
#'   composite("Image",        multi_items("IMAG", 1:5)),
#'   composite("Expectation",  multi_items("CUEX", 1:3)),
#'   composite("Value",        multi_items("PERV", 1:2)),
#'   composite("Satisfaction", multi_items("CUSA", 1:3)),
#'   interaction_term(iv = "Image", moderator = "Expectation", method = orthogonal),
#'   interaction_term(iv = "Image", moderator = "Value", method = product_indicator)
#' )
#'
#' #  structural model: note that name of the interactions construct should be
#' #  the names of its two main constructs joined by a '*' in between.
#' mobi_sm <- relationships(
#'   paths(to = "Satisfaction",
#'         from = c("Image", "Expectation", "Value",
#'                  "Image*Expectation", "Image*Value"))
#' )
#'
#' mobi_pls <- estimate_pls(mobi, mobi_mm, mobi_sm)
#' summary(mobi_pls)
#'
#' @export
interaction_term <- function(iv, moderator, method = two_stage, weights = mode_A) {
  int <- method(iv, moderator, weights)
  class(int) <- class(method())
  return(int)
}

#' \code{orthogonal} creates interaction measurement items by using the orthogonalized approach..
#'
#' This function automatically generates interaction measurement items for a PLS SEM using the orthogonalized approach..
#'
#' @param iv The independent variable that is subject to moderation.
#' @param moderator The moderator variable.
#' @param weights is the relationship between the items and the interaction terms. This can be
#' specified as \code{correlation_weights} or \code{mode_A} for correlation weights (Mode A) or as
#' \code{regression_weights} or \code{mode_B} for regression weights (Mode B). Default is correlation weights.
#'
#' @usage
#'  # orthogonalization approach as per Henseler & Chin (2010):
#'  orthogonal(iv, moderator, weights)
#'
#' @references Henseler & Chin (2010), A comparison of approaches for the analysis of interaction effects
#' between latent variables using partial least squares path modeling. Structural Equation Modeling, 17(1),82-109.
#'
#' @examples
#' data(mobi)
#'
#' # seminr syntax for creating measurement model
#' mobi_mm <- constructs(
#'   composite("Image",        multi_items("IMAG", 1:5)),
#'   composite("Expectation",  multi_items("CUEX", 1:3)),
#'   composite("Value",        multi_items("PERV", 1:2)),
#'   composite("Satisfaction", multi_items("CUSA", 1:3)),
#'   interaction_term(iv = "Image", moderator = "Expectation", method = orthogonal),
#'   interaction_term(iv = "Image", moderator = "Value", method = orthogonal)
#' )
#'
#' #  structural model: note that name of the interactions construct should be
#' #  the names of its two main constructs joined by a '*' in between.
#' mobi_sm <- relationships(
#'   paths(to = "Satisfaction",
#'         from = c("Image", "Expectation", "Value",
#'                  "Image*Expectation", "Image*Value"))
#' )
#'
#' mobi_pls <- estimate_pls(mobi, mobi_mm, mobi_sm)
#' summary(mobi_pls)
#'
#' @export
orthogonal <- function(iv, moderator, weights) {
  ortho_construct <- function(data, measurement_model, structural_model, ints, inner_weights) {
    interaction_name <- paste(iv, moderator, sep = "*")
    iv1_items <- measurement_model[measurement_model[, "construct"] == iv, "measurement"]
    iv2_items <- measurement_model[measurement_model[, "construct"] == moderator, "measurement"]

    iv1_data <- as.data.frame(scale(data[iv1_items]))
    iv2_data <- as.data.frame(scale(data[iv2_items]))

    multiples_list <- lapply(iv1_data, mult, iv2_data)
    interaction_data <- do.call("cbind", multiples_list)
    colnames(interaction_data) <- as.vector(sapply(iv1_items, name_items, iv2_items))
    # Create formula
    frmla <- stats::as.formula(paste("interaction_data[,i]",paste(as.vector(c(iv1_items,iv2_items)), collapse ="+"), sep = " ~ "))

    # iterate and orthogonalize
    for(i in 1:ncol(interaction_data)) {
      interaction_data[, i] <- stats::lm(formula = frmla, data = data)$residuals
    }
    intxn_mm <- matrix(measure_interaction(interaction_name, interaction_data, weights), ncol = 3, byrow = TRUE)
    return(list(name = interaction_name,
                data = interaction_data,
                mm = intxn_mm))
  }
  class(ortho_construct) <- append(class(ortho_construct), c("construct", "orthogonal_interaction"))
  return(ortho_construct)
}

#' \code{product_indicator} creates interaction measurement items by scaled product indicator approach.
#'
#' This function automatically generates interaction measurement items for a PLS SEM using scaled product indicator approach.
#'
#' @param iv The independent variable that is subject to moderation.
#' @param moderator The moderator variable.
#' @param weights is the relationship between the items and the interaction terms. This can be
#' specified as \code{correlation_weights} or \code{mode_A} for correlation weights (Mode A) or as
#' \code{regression_weights} or \code{mode_B} for regression weights (Mode B). Default is correlation weights.
#'
#' @usage
#'  # standardized product indicator approach as per Henseler & Chin (2010):
#'  product_indicator(iv, moderator, weights)
#'
#' @references Henseler & Chin (2010), A comparison of approaches for the analysis of interaction effects
#' between latent variables using partial least squares path modeling. Structural Equation Modeling, 17(1),82-109.
#'
#' @examples
#' data(mobi)
#'
#' # seminr syntax for creating measurement model
#' mobi_mm <- constructs(
#'  composite("Image",        multi_items("IMAG", 1:5),weights = mode_A),
#'  composite("Expectation",  multi_items("CUEX", 1:3),weights = mode_A),
#'  composite("Value",        multi_items("PERV", 1:2),weights = mode_A),
#'  composite("Satisfaction", multi_items("CUSA", 1:3),weights = mode_A),
#'  interaction_term(iv = "Image",
#'                   moderator = "Expectation",
#'                   method = product_indicator,
#'                   weights = mode_A),
#'  interaction_term(iv = "Image",
#'                   moderator = "Value",
#'                   method = product_indicator,
#'                   weights = mode_A)
#' )
#'
#' # structural model: note that name of the interactions construct should be
#' #  the names of its two main constructs joined by a '*' in between.
#' mobi_sm <- relationships(
#'   paths(to = "Satisfaction",
#'         from = c("Image", "Expectation", "Value",
#'                 "Image*Expectation", "Image*Value"))
#' )
#'
#' # Load data, assemble model, and estimate using semPLS
#' mobi <- mobi
#' seminr_model <- estimate_pls(mobi, mobi_mm, mobi_sm,inner_weights = path_factorial)
#'
#' @export
product_indicator <- function(iv, moderator, weights) {
  scaled_interaction <- function(data, measurement_model, structural_model, ints, inner_weights) {
    interaction_name <- paste(iv, moderator, sep = "*")
    iv1_items <- measurement_model[measurement_model[, "construct"] == iv, "measurement"]
    iv2_items <- measurement_model[measurement_model[, "construct"] == moderator, "measurement"]

    iv1_data <- as.data.frame(scale(data[iv1_items]))
    iv2_data <- as.data.frame(scale(data[iv2_items]))

    multiples_list <- lapply(iv1_data, mult, iv2_data)
    interaction_data <- do.call("cbind", multiples_list)
    colnames(interaction_data) <- as.vector(sapply(iv1_items, name_items, iv2_items))
    intxn_mm <- matrix(measure_interaction(interaction_name, interaction_data, weights), ncol = 3, byrow = TRUE)
    return(list(name = interaction_name,
                data = interaction_data,
                mm = intxn_mm))
  }
  class(scaled_interaction) <- append(class(scaled_interaction), c("construct", "scaled_interaction"))
  return(scaled_interaction)
}

#' \code{two_stage} creates an interaction measurement item by the two-stage approach.
#'
#' This function automatically generates an interaction measurement item for a PLS SEM using the two-stage approach.
#'
#' @param iv The independent variable that is subject to moderation.
#' @param moderator The moderator variable.
#' @param weights is the relationship between the items and the interaction terms. This can be
#' specified as \code{correlation_weights} or \code{mode_A} for correlation weights (Mode A) or as
#' \code{regression_weights} or \code{mode_B} for regression weights (Mode B). Default is correlation weights.
#'
#' @usage
#'  # two stage approach as per Henseler & Chin (2010):
#'  two_stage(iv, moderator, weights)
#'
#' @references Henseler & Chin (2010), A comparison of approaches for the analysis of interaction effects
#' between latent variables using partial least squares path modeling. Structural Equation Modeling, 17(1),82-109.
#'
#' @examples
#' data(mobi)
#'
#' # seminr syntax for creating measurement model
#' mobi_mm <- constructs(
#'   composite("Image",        multi_items("IMAG", 1:5)),
#'   composite("Expectation",  multi_items("CUEX", 1:3)),
#'   composite("Value",        multi_items("PERV", 1:2)),
#'   composite("Satisfaction", multi_items("CUSA", 1:3)),
#'   interaction_term(iv = "Image", moderator = "Expectation", method = two_stage)
#' )
#'
#' #  structural model: note that name of the interactions construct should be
#' #  the names of its two main constructs joined by a '*' in between.
#' mobi_sm <- relationships(
#'   paths(to = "Satisfaction",
#'         from = c("Image", "Expectation", "Value",
#'                  "Image*Expectation"))
#' )
#'
#' mobi_pls <- estimate_pls(mobi, mobi_mm, mobi_sm)
#' summary(mobi_pls)
#'
#' @export
two_stage <- function(iv, moderator, weights) {
  two_stage_interaction <- function(data, measurement_model, structural_model, ints, inner_weights) {
    interaction_name <- paste(iv, moderator, sep = "*")
    # remove interactions from structural model
    structural_model <- structural_model[ !grepl("\\*", structural_model[,"source"]), ]
    measurement_mode_scheme <- sapply(unique(c(structural_model[,1],structural_model[,2])), get_measure_mode, measurement_model, USE.NAMES = TRUE)
    first_stage <- seminr::simplePLS(obsData = data,
                                     smMatrix = structural_model,
                                     mmMatrix = measurement_model,
                                     inner_weights = inner_weights,
                                     measurement_mode_scheme = measurement_mode_scheme)

    interaction_term <- as.matrix(first_stage$construct_scores[, iv] * first_stage$construct_scores[, moderator], ncol = 1)[,, drop = FALSE]
    colnames(interaction_term) <- c(interaction_name)

    intxn_mm <- matrix(measure_interaction(interaction_name, interaction_term, weights), ncol = 3, byrow = TRUE)

    return(list(name = interaction_name,
                data = interaction_term[,1, drop = FALSE],
                mm = intxn_mm))
  }
  class(two_stage_interaction) <- append(class(two_stage_interaction), c("construct", "two_stage_interaction"))
  return(two_stage_interaction)
}

process_interactions <- function(measurement_model, data, structural_model, inner_weights) {
  ints <- measurement_model[(substr(names(measurement_model), nchar(names(measurement_model))-10, nchar(names(measurement_model))) == "interaction")]
  non_ints <- measurement_model[(substr(names(measurement_model), nchar(names(measurement_model))-10, nchar(names(measurement_model))) != "interaction")]
  measurement_model <- matrix(unlist(non_ints), ncol = 3, byrow = TRUE,
                              dimnames = list(NULL, c("construct", "measurement", "type")))


  if(length(ints)>0) {
    # update data with new interaction items
    names(ints) <- c()
    create_interaction <- function(intxn_function) { intxn_function(data, measurement_model, structural_model, ints, inner_weights) }
    intxns_list <- lapply(ints, create_interaction)

    get_data <- function(intxn) { intxn$data }
    interaction_data <- do.call("cbind", lapply(intxns_list, get_data))

    get_mm <- function(intxn) { intxn$mm }
    intxns_mm <- do.call("rbind", lapply(intxns_list, get_mm))
    # Append data with interaction data
    data <- cbind(data, interaction_data)

    # update measurement model with interaction constructs
    #intxns_mm <- matrix(unlist(lapply(intxns_list, measure_interaction)), ncol = 3, byrow = TRUE)
    # construct <- c(rbind(construct_name, item_names, composite_type))

    measurement_model <- rbind(measurement_model, intxns_mm)
  }
  return(list(data = data,
              measurement_model = measurement_model,
              ints = ints))
}
