################################################################################
## package 'secr'
## secr.fit.R
## moved from methods.R 2011-01-30
## 2011-10-20 generalized designD
## 2011-10-20 renamed 'grps' 'grouplevels'
## 2011-12-16 streamlined preparation of models
## 2012-01-22 purged phi/turnover
## 2012-01-31 experimental addition of parameter cut
## 2012-04-06 'fixed' bug fixed (see functions.r)
## 2012-07-24 unmash component of details
## 2013-04-11 hcov
## 2013-04-19 lambda0
## 2013-04-20 default mask type changed to trapbuffer
################################################################################

secr.fit <- function (capthist, model = list(D~1, g0~1, sigma~1), mask = NULL,
    buffer = NULL, CL = FALSE, detectfn = NULL, binomN = NULL, start = NULL,
    link = list(), fixed = list(), timecov = NULL, sessioncov = NULL, hcov = NULL,
    groups = NULL, dframe = NULL, details = list(), method = 'Newton-Raphson',
    verify = TRUE, biasLimit = 0.01, trace = NULL, ncores = 1, ...)

{
# Fit spatially explicit capture recapture model
#
# Arguments:
#
#  capthist   -  capture history object (includes traps object as an attribute)
#  model      -  formulae for real parameters in terms of effects and covariates
#  mask       -  habitat mask object
#  buffer     -  default buffer width should mask not be provided
#  CL         -  logical switch : conditional likelihood (T) or full likelihood (F)
#  detectfn   -  code for detection function 0 = halfnormal, 1 = hazard, 2 = exponential etc.
#  start      -  start values for maximization (numeric vector link scale);
#                if NULL then 'autoini' function is used
#  link       -  list of parameter-specific link function names 'log', 'logit', 'identity',
#                'sin', 'neglog'
#  fixed      -  list of fixed values for named parameters
#  timecov    -  data for time covariates if these are used in 'model'
#  sessioncov -  dataframe of session-level covariates
#  groups     -  vector of names to group fields in attr(capthist,'covariates') dataframe
#  dframe     -  optional data frame of design data for detection model (tricky & untested)
#  details    -  list with several additional settings, mostly of special interest
#  method     -  optimization method (indirectly chooses
#  verify     -  logical switch for pre-check of capthist and mask with verify()
#  trace      -  logical; if TRUE output each likelihood as it is calculated
#  ...        -  other arguments passed to nlm() or optim()


    if (!inherits(capthist, 'capthist'))
        stop ("requires 'capthist' object")
    if ((detector(traps(capthist))=='cue') & (is.null(groups) | !CL))
        stop ("cue detector requires CL = TRUE and groups")

    #################################################
    ## Remember start time and call

    ptm  <- proc.time()
    starttime <- format(Sys.time(), "%H:%M:%S %d %b %Y")

    cl   <- match.call(expand.dots = TRUE)

    if (ncores > 1) {
        if (require(parallel)) {
            clust <- makeCluster(ncores, methods = FALSE, useXDR = FALSE)
            clusterEvalQ(clust, library(secr))
        }
        else {
            stop("package parallel not installed")
        }
    }
    else
        clust <- NULL

    #################################################
    ## Default detection function

    if (is.null(detectfn)) {
        if (detector(traps(capthist)) %in% c('cue', 'signal')) {
            detectfn <- 10
            warning ("detectfn not specified; using signal strength (10)")
        }
        else if (detector(traps(capthist)) %in% c('signalnoise')) {
            detectfn <- 12
            warning ("detectfn not specified; using signal-noise (12)")
        }
        else {
            detectfn <- 0
        }
    }

    else {
        if (detector(traps(capthist)) == 'presence')
            detectfn <- valid.detectfn(detectfn, 0:8)
        else
            detectfn <- valid.detectfn(detectfn)
    }

    #################################################
    ## Use input 'details' to override various defaults

    defaultdetails <- list(distribution = 'poisson',
                           scalesigma = FALSE,
                           scaleg0 = FALSE,
                           hessian = 'auto',
                           trace = TRUE,
                           LLonly = FALSE,
                           centred = FALSE,
                           binomN = 1,
                           cutval = 0,
                           minprob = 1e-50,
                           tx = 'identity',
                           param = 0,
                           unmash = FALSE,
                           telemetrysigma = FALSE,
                           ignoreusage = FALSE,
                           debug = FALSE,
                           intwidth2 = 0.8
                           )

    if (detector(traps(capthist)) %in% .localstuff$countdetectors)
        defaultdetails$binomN <- 0   ## Poisson
    if (!is.null(attr(capthist,'cutval')))
        defaultdetails$cutval <- attr(capthist,'cutval')
    else if (ms(capthist) & !is.null(attr(capthist[[1]],'cutval')))   ## 2012-09-04
        defaultdetails$cutval <- attr(capthist[[1]],'cutval')
    if (is.logical(details$hessian))
        details$hessian <- ifelse(details$hessian, 'auto', 'none')
    details <- replace (defaultdetails, names(details), details)
    if (!is.null(trace)) details$trace <- trace
    if (!is.null(binomN)) {
        if (detector(traps(capthist)) == 'count') {
            if (tolower(binomN) == 'usage')
                binomN <- 1   ## code for 'binomial size from usage' 2012-12-22
            if (tolower(binomN) == 'poissonhazard')
                binomN <- -1  ## interpret g() as true detection function 2013-01-07
        }
        details$binomN <- binomN   ## 2011 01 28
    }
    if (details$LLonly)  details$trace <- FALSE
    if (detector(traps(capthist)) != 'multi') details$param <- 0

    ## 2011-02-06 to be quite clear -
    if (detector(traps(capthist)) %in% c(.localstuff$exclusivedetectors,
                                         'proximity','signal','signalnoise'))
        details$binomN <- 1;


    #################################################
    ## MS - indicator TRUE if multi-session (logical)
    ## sessionlevels - names of sessions (character)

    MS <- ms(capthist)
    sessionlevels <- session(capthist)

    if (is.null(sessionlevels)) sessionlevels <- '1'
    anycount <- any(detector(traps(capthist)) %in% .localstuff$countdetectors)
    anypoly  <- any(detector(traps(capthist)) %in% c('polygon',  'polygonX'))
    anytrans <- any(detector(traps(capthist)) %in% c('transect', 'transectX'))
    alltelem <- all(detector(traps(capthist)) %in% c('telemetry'))
    if (alltelem) CL <- TRUE

    if (MS) {
       if (any (sapply(traps(capthist), detector) == 'single'))
        warning ("multi-catch likelihood used for single-catch traps")
    }
    else {
       if (detector(traps(capthist)) == 'single')
        warning ("multi-catch likelihood used for single-catch traps")
    }

    #################################################
    ## Optional data check added 2009 09 19

    if (verify) {
        memo ('Checking data', details$trace)
        test <- verify(capthist, report = 1)
        if (test$errors)
            stop ("'verify' found errors in 'capthist' argument")

        if (!is.null(mask)) {
            if (MS & ms(mask)) {
                ## list of masks
                test <- lapply(mask, verify, report = 1)
                notOK <- any(unlist(test))
            }
            else notOK <- verify(mask, report = 1)$errors
            if (notOK)
                stop ("'verify' found errors in 'mask' argument")
        }
    }

    #################################################
    ## Ensure valid mask
    ## assume traps(capthist) will extract a list of trap layouts
    ## if multi-session (MS == TRUE)

    usebuffer <- is.null(mask)    ## flag for later check
    if (usebuffer) {
        if (is.null(buffer)) {
            buffer <- 100
            if (!(detector(traps(capthist))=='presence') & !alltelem)
                warning ("using default buffer width 100 m")
        }
        if (MS) mask <- lapply (traps(capthist), make.mask, buffer = buffer, type = "trapbuffer")
        else    mask <- make.mask(traps(capthist), buffer = buffer, type = "trapbuffer")
    }
    else {
      if (MS & !ms(mask)) {
          ## inefficiently replicate mask for each session!
          mask <- lapply(sessionlevels, function(x) mask)
          class (mask) <- c('list', 'mask')
          names(mask) <- sessionlevels
      }
    }

    nc <- ifelse (MS, sum(sapply(capthist, nrow)), nrow(capthist))
    if (nc < 1)
        warning (nc, " detection histories")

    #################################################
    ## orphan mark-resight code - not currently used

    if (MS) {
        q  <- attr(capthist[[1]],'q')
        Tm <- attr(capthist[[1]],'Tm')
    }
    else {
        q <- attr(capthist,'q')
        Tm <- attr(capthist,'Tm')
    }
    nonID <- !is.null(Tm)   ## were marked animals recorded if unidentified?
    if (!is.null(q) & CL)
        stop ("mark-resight incompatible with CL")

    #################################################
    ## optional centring of traps and mask 2010 04 27
    if (details$centred) {
        centre <- function (xy, dxy) {
            xy[,] <- sweep(xy, MARGIN = 2, FUN='-', STATS = dxy)
            xy
        }
        if (MS) {
            nsess <- length(traps(capthist))
            offsetxy <- lapply(traps(capthist), function(xy) apply(xy, 2, mean))
            for (i in 1:nsess) {
                temptraps <- centre(traps(capthist[[i]]), offsetxy[[i]])
                traps(capthist[[i]]) <- temptraps
                mask[[i]] <- centre(mask[[i]], offsetxy[[i]])
                attr(mask[[i]], 'meanSD')[1,1:2] <- attr(mask[[i]], 'meanSD')[1,1:2] -
                    offsetxy[[i]]
                attr(mask[[i]], 'boundingbox') <- centre(attr(mask[[i]], 'boundingbox'),
                    offsetxy[[i]])
            }
        }
        else {
            offsetxy <- apply(traps(capthist), 2, mean)
            traps(capthist) <- shift(traps(capthist), -offsetxy)
            mask <- shift.traps(mask, -offsetxy)
            attr(mask, 'meanSD')[1,1:2] <- attr(mask, 'meanSD')[1,1:2] - offsetxy
            attr(mask, 'boundingbox') <- centre(attr(mask, 'boundingbox'), offsetxy)
        }
    }

    #################################################
    ## Use input formula to override defaults

    if ('formula' %in% class(model)) model <- list(model)
    model <- stdform (model)  ## named, no LHS
    ## pmix, lambda0 added to defaultmodel 2013-04
    defaultmodel <- list(D=~1, g0=~1, lambda0=~1, sigma=~1, z=~1, w=~1,
         pID=~1, beta0=~1, beta1=~1, sdS=~1, b0=~1, b1=~1, pmix=~1)
    model <- replace (defaultmodel, names(model), model)

    ## modelled parameters
    pnames <- switch (detectfn+1,
        c('g0','sigma'),           # 0 halfnormal
        c('g0','sigma','z'),       # 1 hazard rate
        c('g0','sigma'),           # 2 exponential
        c('g0','sigma','z'),       # 3
        c('g0','sigma'),           # 4
        c('g0','sigma','w'),       # 5
        c('g0','sigma','w'),       # 6
        c('g0','sigma','z'),       # 7
        c('g0','sigma','z'),       # 8
        c('b0','b1'),              # 9
        c('beta0','beta1','sdS'),  # 10
        c('beta0','beta1','sdS'),  # 11
        c('beta0','beta1','sdS'),  # 12  cf parnames() in utility.R: muN, sdN?
        c('beta0','beta1','sdS'),  # 13  cf parnames() in utility.R: muN, sdN?
        c('lambda0','sigma'),      # 14 hazard halfnormal
        c('lambda0','sigma','z'),  # 15 hazard hazard rate
        c('lambda0','sigma'),      # 16 hazard exponential
        c('lambda0','sigma','w'),  # 17
        c('lambda0','sigma','z'))  # 18

    if (!CL) pnames <- c('D', pnames)
    if (!is.null(q) & nonID) {
	pnames <- c(pnames, 'pID')
        if (model$pID != ~1)
            stop ("'pID' must be constant in this implementation")
    }
    if (alltelem) {
        rnum <- match(c('D','g0'), pnames)
        pnames <- pnames[-rnum[!is.na(rnum)]]
    }
    fnames <- names(fixed)
    pnames <- pnames[!(pnames %in% fnames)]        ## drop fixed real parameters

    ############################################
    # Finite mixtures - 2009 12 10, 2011 12 16
    ############################################

    nmix <- get.nmix(model, capthist, hcov)
    if ((nmix>1) & !is.null(hcov) & !is.null(groups))
        stop ("hcov mixture model incompatible with groups")

    if ((nmix>1) & (nmix<4) & !('pmix' %in% fnames)) {
        model$pmix <- as.formula(paste('~h', nmix, sep=''))
        ## but do we allow users to specify model$pmix at all?
        if (!all(all.vars(model$pmix) %in% c('session','g','h2','h3')))
            stop ("formula for pmix may include only 'session', 'g' or '1'")
        pnames <- c(pnames, 'pmix')
    }
    else
        model$pmix <- NULL
    details$nmix <- nmix

    model[!(names(model) %in% pnames)] <- NULL     ## select real parameters
    vars <-  unlist(lapply(model, all.vars))

    ############################################
    ## Specialisations
    ############################################
    if (CL & !(is.null(groups) | (detector(traps(capthist))=='cue'))) {
        groups <- NULL
        warning ("groups not valid with CL; groups ignored")
    }
    if (CL && var.in.model('g', model))
        stop ("'g' is not a valid effect when 'CL = TRUE'")
    if ((length(model) == 0) & !is.null(fixed))
        stop ("all parameters fixed")     ## assume want only LL
    if (details$scalesigma) {
        if (CL)
            stop ("cannot use 'scalesigma' with 'CL'")
        if (!is.null(fixed$D))
            stop ("cannot use 'scalesigma' with fixed density")
        if (!(model$D == formula(~1) |
              model$D == formula(~session)))
            stop ("cannot use 'scalesigma' with inhomogenous density or groups")
        if (!is.null(groups))
            stop ("cannot use 'scalesigma' with groups")
    }
    if (details$scaleg0) {
        if (!is.null(groups))
            stop ('Cannot use scaleg0 with groups')
    }

    ############################################
    # Link functions (model-specific)
    ############################################
    defaultlink <- list(D='log', g0='logit', lambda0='log', sigma='log', z='log', w='log',
        pID='logit', beta0='identity', beta1='neglog', sdS='log', b0='log', b1='neglog',
        pmix='logit', cuerate='log', cut='identity')
    if (anycount) defaultlink$g0 <- 'log'
    link <- replace (defaultlink, names(link), link)
    link[!(names(link) %in% c(fnames,pnames))] <- NULL
    if (details$scaleg0) link$g0 <- 'log'  ## Force log link in this case as no longer 0-1
    if (!(detector(traps(capthist))=='cue')) link$cuerate <- NULL

    ##############################################
    # Prepare detection design matrices and lookup
    ##############################################

    memo ('Preparing detection design matrices', details$trace)
    design <- secr.design.MS (capthist, model, timecov, sessioncov, groups, hcov,
                              dframe)
    design0 <- secr.design.MS (capthist, model, timecov, sessioncov, groups, hcov,
                               dframe, naive = T, bygroup = !CL)

    ############################################
    # Prepare density design matrix
    ############################################
    D.modelled <- !CL & is.null(fixed$D)
    if (!D.modelled) {
       designD <- matrix(nrow = 0, ncol = 0)
       grouplevels <- 1    ## was NULL
       attr(designD, 'dimD') <- NA
       nDensityParameters <- integer(0)
    }
    else {
        grouplevels  <- group.levels(capthist,groups)
        if (!is.null(details$userDfn)) {
            ## may provide a function used by getD in functions.R
            ## userDfn(mask, beta[parindx$D], ngrp, nsession)
            designD <- details$userDfn
            if (!is.function(designD))
                stop ("details$userDfn should be a function")
            ## this form of call returns only coefficient names
            Dnames <- designD('parameters', mask)
        }
        else {
            memo ('Preparing density design matrix', details$trace)
            temp <- D.designdata( mask, model$D, grouplevels, sessionlevels, sessioncov)
            D.designmatrix <- model.matrix(model$D, temp)
            attr(D.designmatrix, 'dimD') <- attr(temp, 'dimD')
            Dnames <- colnames(D.designmatrix)
            designD <- D.designmatrix
        }
        nDensityParameters <- length(Dnames)
    }

    ############################################
    # Parameter mapping (general)
    ############################################
    np <- sapply(design$designMatrices, ncol)
    np <- c(D = nDensityParameters, np)
    NP <- sum(np)
    parindx <- split(1:NP, rep(1:length(np), np))
    names(parindx) <- names(np)
    if (!D.modelled) parindx$D <- NULL

    ############################################
    # Optionally start from previous fit
    ############################################

    if (inherits(start, 'secr')) {
        ## use 'mapbeta' from score.test.R
        start <- mapbeta(start$parindx, parindx, coef(start)$beta, NULL)
    }

    ############################################
    # send data to worker processes
    # do it once, not each eval
    ############################################
    if (ncores > 1) {
        clusterExport(clust, c("capthist", "mask", "groups",
            "design","design0", "detectfn"), envir = environment())
    }

    ############################################
    # Single evaluation option
    ############################################
    .localstuff$iter <- 0
    if (details$LLonly) {
      if (is.null(start))
          stop ("must provide transformed parameter values in 'start'")
      if (!is.null(q))
          stop ("not for mark-resight")

      LL <- - secr.loglikfn (beta = start,
                       parindx    = parindx,
                       link       = link,
                       fixedpar   = fixed,
                       designD    = designD,
                       design     = design,
                       design0    = design0,
                       capthist   = capthist,
                       mask       = mask,
                       detectfn   = detectfn,
                       CL         = CL,
                       hcov       = hcov,
                       groups     = groups,
                       details    = details,
                       logmult    = TRUE,     ## add if possible
                       ncores     = ncores,
                       clust      = clust
                       )

      return(c(logLik=LL))
    }
    ############################################
    # Start values (model-specific)
    # 'start' is vector of beta values (i.e. transformed)
    ############################################
    if (is.null(start)) {
        start3 <- list(D=NA, g0=NA, sigma=NA)
        ## not for signal attenuation
        if (!(detectfn %in% c(9,10,11,12,13)) && !anypoly && !anytrans) {
            memo('Finding initial parameter values...', details$trace)
            ## autoini uses default buffer dbar * 4
            if (MS)
                ## Using session 1, but this can be risky
                start3 <- autoini (capthist[[1]], mask[[1]],
                                   binomN = details$binomN,
                                   ignoreusage = details$ignoreusage)
            else
                start3 <- autoini (capthist, mask,
                                   binomN = details$binomN,
                                   ignoreusage = details$ignoreusage)

            if (any(is.na(unlist(start3)))) {
                warning ("'secr.fit' failed because initial values not found",
                         " (data sparse?); specify transformed values in 'start'")
                return (list(call=cl, fit=NULL))
            }
            if (details$unmash & !CL) {
                nmash <- attr(capthist[[1]], 'n.mash')
                if (!is.null(nmash)) {
                    n.clust <- length(nmash)
                    start3$D <- start3$D / n.clust
                }
            }
            if (details$scaleg0 & anycount)
                stop ("'scaleg0' not compatible with count detectors")
            ## next two stmts must be this order (g0 then sigma)
            if (details$scaleg0) start3$g0 <- start3$g0 * start3$sigma^2
            if (details$scalesigma) start3$sigma <- start3$sigma * start3$D^0.5

            memo(paste('Initial values ', paste(paste(c('D', 'g0', 'sigma'),
                '=', round(unlist(start3),5)),collapse=', ')),
                details$trace)
        }
        else warning ("using default starting values")

        #--------------------------------------------------------------
        # assemble start vector
        default <- list(
            D     = ifelse (is.na(start3$D), 1, start3$D),
            g0    = ifelse (is.na(start3$g0), 0.1, start3$g0),
            lambda0 = -log(1-ifelse (is.na(start3$g0), 0.1, start3$g0)),
            sigma = ifelse (is.na(start3$sigma), unlist(RPSV(capthist))[1], start3$sigma),
            z     = 5,
            w     = 10,
            pID   = 0.7,
            beta0 = details$cutval + 30,
            beta1 = -0.2,
            sdS   = 2,
            b0    = 2,        ## changed from 15 2010-11-01
            b1    = -0.1,
            pmix  = 0.75      ## superceded below
        )
        if (detectfn %in% c(6)) {
            default$w <- default$sigma
            default$sigma <- default$sigma/2
        }
        if (detectfn %in% c(7)) {
            default$z <- default$sigma/5
        }
        if (detectfn %in% c(8, 18)) {
            default$z <- 1    ## cumulative gamma
        }
        if (anypoly | anytrans) {
            if (MS) {
                tempcapthist <- capthist[[1]]
                tempmask <- mask[[1]]
            }
            else {
                tempcapthist <- capthist
                tempmask <- mask
            }
            default$D <- 2 * nrow(tempcapthist) / (nrow(tempmask)*attr(tempmask,'area'))
            default$g0 <- sum(tempcapthist) / nrow(tempcapthist) / ncol(tempcapthist)
            default$lambda0 <- -log(1-details$g0)
            if (details$binomN > 1)
                default$g0 <- default$g0 / details$binomN
            if ((details$binomN == 1) & (detector(traps(capthist)) %in% c('polygon','transect'))) {
                ## assume using usage for binomN
                usge <- usage(traps(capthist))
                default$g0 <- default$g0 / mean(usge[usge>0])
            }
            default$sigma <- RPSV(tempcapthist)
        }
        if (is.na(default$sigma)) default$sigma <- 20
        getdefault <- function (par) transform (default[[par]], link[[par]])

        start <- rep(0, NP)
        for ( i in 1:length(parindx) )
            start[parindx[[i]][1]] <- getdefault (names(model)[i])
        if ((details$nmix>1) & !('pmix' %in% fnames))
            ## new starting values 2013-04-20
            start[parindx[['pmix']]] <- clean.mlogit((1:nmix)-0.5)[-1]
        if (detector(traps(capthist))=='cue')
            start <- c(start, log(3))    ## cuerate
        if (detectfn %in% c(12,13))
            start <- c(start, 46,3)    ## muN, sdN

        # D/ngrp when figure out where to calculate this

        ## if (!(is.null(q) | !nonID) & is.null(fixed$pID))
        ##     start[parindx$pID[1]] <- getdefault('pID')

        # start vector completed
        #--------------------------------------------------------------
    }

    ############################################
    ## ad hoc fix for experimental parameters
    ############################################
    nmiscparm <- 0
    if (detector(traps(capthist)) %in% c('cue'))
        nmiscparm <- 1
    if (detector(traps(capthist)) %in% c('signalnoise'))
        nmiscparm <- 2
    NP <- NP + nmiscparm
    stopifnot (length(start) == NP)

    ############################################
    # Fixed beta parameters
    ############################################
    fb <- details$fixedbeta
    if (!is.null(fb)) {
        if (!(length(fb)== NP))
            stop ("invalid fixed beta - require NP-vector")
        if (sum(is.na(fb))==0)
            stop ("cannot fix all beta parameters")
        ## drop unwanted betas; remember later to adjust parameter count
        start <- start[is.na(fb)]
        NP <- length(start)
    }
    ############################################
    # Variable names (general)
    ############################################
    betanames <- unlist(sapply(design$designMatrices, colnames))
    names(betanames) <- NULL
    realnames <- names(model)
    if (D.modelled) betanames <- c(paste('D', Dnames, sep='.'), betanames)
    betanames <- sub('..(Intercept))','',betanames)

    ############################################
    # Variable names (model-specific)
    ############################################

    if (detector(traps(capthist))=='cue') {
        betanames <- c(betanames, 'cuerate')
        realnames <- c(realnames, 'cuerate')
    }
    if (detectfn %in% c(12,13)) {
        betanames <- c(betanames, 'muN', 'sdN')
        realnames <- c(realnames, 'muN', 'sdN')
    }
    ## allow for fixed beta parameters
    if (!is.null(details$fixedbeta))
        betanames <- betanames[is.na(details$fixedbeta)]
    betaw <- max(max(nchar(betanames)),8)   # for 'trace' formatting

    ############################################
    # Maximize likelihood
    ############################################

    memo('Maximizing likelihood...', details$trace)
    if (details$trace)
        cat('Eval     Loglik', formatC(betanames, format='f', width=betaw), '\n')

    loglikefn <- secr.loglikfn
    if (!is.null(q))
        stop ("mark-resight option not operative")
        ## loglikefn <- MRsecr.loglikfn

    ## arguments always passed to loglikefn
    secrargs <- list(
                     link       = link,
                     fixedpar   = fixed,
                     parindx    = parindx,
                     capthist   = capthist,
                     mask       = mask,
                     CL         = CL,
                     detectfn   = detectfn,
                     designD    = designD,
                     design     = design,
                     design0    = design0,
                     hcov       = hcov,
                     groups     = groups,
                     details    = details,
                     logmult    = TRUE,     # add if possible
                     ncores     = ncores,
                     clust      = clust,
                     betaw      = betaw)    # for trace format

    ############################################
    ## calls for specific maximisation methods
    lcmethod <- tolower(method)
##  if ((lcmethod %in% c('optimise')) & (NP == 1)) {
## 2013-04-21
    if (NP == 1) {
        lcmethod <- "optimise"
        signs <- c(-1,1) * sign(start)
        print(start)
        print( start*(1 + details$intwidth2 * signs))
        args <- list (f         = loglikefn,
            interval  = start * (1 + details$intwidth2 * signs))
        args <- c(args, secrargs)
        args <- replace (args, names(list(...)), list(...))
        this.fit <- try(do.call (optimise, args))
        if (inherits(this.fit, 'try-error'))
            warning ("univariate search for minimum failed")
        this.fit$par <- this.fit$minimum
        this.fit$value <- this.fit$objective
        if (details$hessian != "none")
            details$hessian <- "fdHess"
    }
    else if (lcmethod %in% c('newton-raphson')) {
        args <- list (p         = start,
                      f         = loglikefn,
                      hessian   = tolower(details$hessian)=='auto',
                      stepmax   = 10)
        args <- c(args, secrargs)
        args <- replace (args, names(list(...)), list(...))
        this.fit <- do.call (nlm, args)
        this.fit$par <- this.fit$estimate     # copy for uniformity
        this.fit$value <- this.fit$minimum    # copy for uniformity
        if (this.fit$code > 2)
            warning ("possible maximization error: nlm returned code ",
                this.fit$code, ". See ?nlm")
    }
    #-----------------------------------------------------------------
    else if (method %in% c("Nelder-Mead", "BFGS", "CG", "L-BFGS-B",
                             "SANN", "Brent")) {
        args <- list(par     = start,
                     fn      = loglikefn,
                     hessian = tolower(details$hessian)=='auto',
                     method  = method)
        args <- c(args, secrargs)
        args <- replace (args, names(list(...)), list(...))
        this.fit <- do.call (optim, args)
        if (this.fit$convergence != 0)
            warning ("probable maximization error: optim returned convergence ",
                this.fit$convergence, ". See ?optim")
    }
    #-----------------------------------------------------------------
    # Hessian-only 2013-02-23
    else if (lcmethod %in% 'none') {
        if (require (nlme)) {
            memo ('Computing Hessian with fdHess in nlme', details$trace)
            loglikfn <- function (beta) {
                do.call(secr.loglikfn, c(list(beta=beta), secrargs))
            }
            grad.Hess <- fdHess(start, fun = loglikfn, .relStep = 0.001, minAbsPar=0.1)
            this.fit <- list (value = 0, par = start,
                              gradient = grad.Hess$gradient,
                              hessian = grad.Hess$Hessian)
            biasLimit <- NA   ## no bias assessment
        }
        else {
            stop("package nlme not installed")
        }
    }
    else stop ("maximisation method", method, "not recognised")
    ############################################################################

    this.fit$method <- method         ## remember which method we used...
    covar <- NULL
    N <- NULL
    if (this.fit$value > 1e9) {     ## failed
        # this.fit$beta[] <- NA
        this.fit$par[] <- NA
    }
    else {

    ############################################
    ## Variance-covariance matrix
    ############################################

        if (tolower(details$hessian)=='fdhess') {
            if (require (nlme)) {
                memo ('Computing Hessian with fdHess in nlme', details$trace)
                loglikfn <- function (beta) {
                    do.call(secr.loglikfn, c(list(beta=beta), secrargs))
                }
                grad.Hess <- fdHess(this.fit$par, fun = loglikfn, .relStep = 0.001, minAbsPar=0.1)
                this.fit$hessian <- grad.Hess$Hessian
            }
            else  {
                stop("package nlme not installed")
            }
        }
        if (!is.null(this.fit$hessian)) {
            ## switched to MASS function ginv 2013-04-13
            ## covar <- try(solve(this.fit$hessian))
            covar <- try(ginv(this.fit$hessian))
            if (inherits(covar, "try-error")) {
                warning ("could not invert Hessian to compute ",
                         "variance-covariance matrix")
                covar <- matrix(nrow = NP, ncol = NP)
            }
            else if (any(diag(covar)<0)) {
                warning ("at least one variance calculation failed ")
            }
            dimnames(covar) <- list(betanames, betanames)
        }

        ## predicted D across mask
        if (!CL) {
            D <- getD (designD, this.fit$par, mask, parindx, link, fixed,
                       grouplevels, sessionlevels)
            N <- t(apply(D, 2:3, sum, drop = FALSE))
            cellarea <- if (ms(mask)) sapply(mask, attr, 'area')
                        else cellarea <- attr(mask,'area')
            N <- sweep(N, FUN = '*', MARGIN = 1, STATS = cellarea)
        }
    }

    ############################################
    ## form output list
    ############################################

    desc <- packageDescription("secr")  ## for version number
    output <- list (call = cl,
                  capthist = capthist,
                  mask = mask,
                  detectfn = detectfn,
                  CL = CL,
                  timecov = timecov,
                  sessioncov = sessioncov,
                  hcov = hcov,
                  groups = groups,
                  dframe = dframe,
                  design = design,
                  design0 = design0,
                  start = start,
                  link = link,
                  fixed = fixed,
                  parindx = parindx,
                  model = model,
                  details = details,
                  vars = vars,
                  betanames = betanames,
                  realnames = realnames,
                  fit = this.fit,
                  beta.vcv = covar,
                  N = N,
                  version = desc$Version,
                  starttime = starttime,
                  proctime = (proc.time() - ptm)[3]
             )

    class(output) <- 'secr'

    ############################################
    ## buffer bias check
    ## not for polygon & transect detectors
    ## adjust for user-spec biasLimit 2012-01-10
    ############################################

    validbiasLimit <- !is.null(biasLimit)
    validbiasLimit <- validbiasLimit & is.finite(biasLimit)
    validbiasLimit <- validbiasLimit & (biasLimit>0)
    if (usebuffer & (this.fit$value < 1e9) &
        (detector(traps(capthist)) %in% .localstuff$pointdetectors) &
        !(detector(traps(capthist)) %in% c('cue','unmarked','presence')) &
        is.null(attr(capthist,'xylist')) &
        validbiasLimit) {
        if (MS) {
            nsess <- length(capthist)
            bias <- numeric(nsess)
            for (i in 1:nsess) {
                temptrps <- traps(capthist)[[i]]
                if (details$ignoreusage)
                    usage(temptrps) <- NULL
                biastemp <- try( bias.D(buffer, temptrps,
                                        detectfn = output$detectfn,
                                        detectpar = detectpar(output)[[i]],
                                        noccasions = ncol(capthist[[i]]),
                                        binomN = details$binomN) )
                if (inherits(biastemp, 'try-error'))
                   warning('could not perform bias check')
                else
                    bias[i] <- biastemp$RB.D
            }
        }
        else {
            temptrps <- traps(capthist)
            if (details$ignoreusage)
                usage(temptrps) <- NULL
            bias <- try( bias.D(buffer, temptrps,
                                detectfn = output$detectfn,
                                detectpar = detectpar(output),
                                noccasions = ncol(capthist),
                                binomN = details$binomN) )
            if (inherits(bias, 'try-error')) {
                warning('could not perform bias check')
                bias <- 0  ## 2012-12-18 suppresses second message
            }
            else
                bias <- bias$RB.D
        }
        if (any(bias > biasLimit))
            warning ("predicted relative bias exceeds ", biasLimit, " with ",
                     "buffer = ", buffer)
    }

    ############################################

    memo(paste('Completed in ', round(output$proctime,2), ' seconds at ',
        format(Sys.time(), "%H:%M:%S %d %b %Y"),
        sep=''), details$trace)

    if (ncores > 1) {
        stopCluster(clust)
    }

    output

}
