% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/RcppExports.R
\name{dTpdWou2D}
\alias{dTpdWou2D}
\title{Approximation of the transition probability density of the WN diffusion in 2D}
\usage{
dTpdWou2D(x, x0, t, alpha, mu, sigma, rho = 0, maxK = 2L, expTrc = 30)
}
\arguments{
\item{x}{a matrix of dimension \code{c(n, 2)} containing angles. They all must be in \eqn{[\pi,\pi)} so that the truncated wrapping by \code{maxK} windings is able to capture periodicity.}

\item{x0}{a matrix of dimension \code{c(n, 2)} containing the starting angles. They all must be in \eqn{[\pi,\pi)}. If all \code{x0} are the same, a matrix of dimension \code{c(1, 2)} can be passed for better performance.}

\item{t}{a scalar containing the times separating \code{x} and \code{x0}.}

\item{alpha}{vector of length \code{3} parametrizing the \code{A} matrix as in \code{\link{alphaToA}}.}

\item{mu}{a vector of length \code{2} giving the mean.}

\item{sigma}{vector of length \code{2} containing the \strong{square root} of the diagonal of \eqn{\Sigma}, the diffusion matrix.}

\item{rho}{correlation coefficient of \eqn{\Sigma}.}

\item{maxK}{maximum absolute value of the windings considered in the computation of the WN.}

\item{expTrc}{truncation for exponential: \code{exp(x)} with \code{x <= -expTrc} is set to zero. Defaults to \code{30}.}
}
\value{
A vector of size \code{n} containing the tpd evaluated at \code{x}.
}
\description{
Computation of the transition probability density (tpd) for a WN diffusion (with diagonal diffusion matrix)
}
\details{
The function checks for positive definiteness. If violated, it resets \code{A} such that \code{solve(A) \%*\% Sigma} is positive definite.

See Section 3.3 in García-Portugués et al. (2019) for details. See \code{\link{dTpdWou}} for the general case (less efficient for 1D).
}
\examples{
set.seed(3455267)
alpha <- c(2, 1, -1)
sigma <- c(1.5, 2)
rho <- 0.9
Sigma <- diag(sigma^2)
Sigma[1, 2] <- Sigma[2, 1] <- rho * prod(sigma)
A <- alphaToA(alpha = alpha, sigma = sigma, rho = rho)
solve(A) \%*\% Sigma
mu <- c(pi, 0)
x <- t(euler2D(x0 = matrix(c(0, 0), nrow = 1), A = A, mu = mu,
               sigma = sigma, N = 500, delta = 0.1)[1, , ])
\donttest{
sum(sapply(1:49, function(i) log(dTpdWou(x = matrix(x[i + 1, ], ncol = 2),
                                         x0 = x[i, ], t = 1.5, A = A,
                                         Sigma = Sigma, mu = mu))))
}
sum(log(dTpdWou2D(x = matrix(x[2:50, ], ncol = 2),
                  x0 = matrix(x[1:49, ], ncol = 2), t = 1.5, alpha = alpha,
                  mu = mu, sigma = sigma, rho = rho)))
\donttest{
lgrid <- 56
grid <- seq(-pi, pi, l = lgrid + 1)[-(lgrid + 1)]
image(grid, grid, matrix(dTpdWou(x = as.matrix(expand.grid(grid, grid)),
                                 x0 = c(0, 0), t = 0.5, A = A,
                                 Sigma = Sigma, mu = mu),
                         nrow = 56, ncol = 56), zlim = c(0, 0.25),
      main = "dTpdWou")
image(grid, grid, matrix(dTpdWou2D(x = as.matrix(expand.grid(grid, grid)),
                                   x0 = matrix(0, nrow = 56^2, ncol = 2),
                                   t = 0.5, alpha = alpha, sigma = sigma,
                                   mu = mu),
                         nrow = 56, ncol = 56), zlim = c(0, 0.25),
      main = "dTpdWou2D")

x <- seq(-pi, pi, l = 100)
t <- 1
image(x, x, matrix(dTpdWou2D(x = as.matrix(expand.grid(x, x)),
                             x0 = matrix(rep(0, 100 * 2), nrow = 100 * 100,
                                         ncol = 2),
                             t = t, alpha = alpha, mu = mu, sigma = sigma,
                             maxK = 2, expTrc = 30),
                             nrow = 100, ncol = 100),
      zlim = c(0, 0.25))
points(stepAheadWn2D(x0 = rbind(c(0, 0)), delta = t / 500,
                     A = alphaToA(alpha = alpha, sigma = sigma), mu = mu,
                     sigma = sigma, N = 500, M = 1000, maxK = 2,
                     expTrc = 30))
}
}
\references{
García-Portugués, E., Sørensen, M., Mardia, K. V. and Hamelryck, T. (2019) Langevin diffusions on the torus: estimation and applications. \emph{Statistics and Computing}, 29(2):1--22. \url{https://doi.org/10.1007/s11222-017-9790-2}
}
