% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/unround.R
\name{unround}
\alias{unround}
\title{Reconstruct rounding bounds}
\usage{
unround(x, rounding = "up_or_down", threshold = 5, digits = NULL)
}
\arguments{
\item{x}{String or numeric. Rounded number. \code{x} must be a string unless
\code{digits} is specified (most likely by a function that uses \code{unround()} as a
helper).}

\item{rounding}{String. Rounding method presumably used to create \code{x}.
Default is \code{"up_or_down"}. For more, see section \code{Rounding}.}

\item{threshold}{Integer. Number from which to round up or down. Other
rounding methods are not affected. Default is \code{5}.}

\item{digits}{Integer. This argument is meant to make \code{unround()} more
efficient to use as a helper function so that it doesn't need to
redundantly count decimal places. Don't specify it otherwise. Default is
\code{NULL}, in which case decimal places really are counted internally and \code{x}
must be a string.}
}
\value{
A tibble with seven columns: \code{range}, \code{rounding}, \code{lower},
\code{incl_lower}, \code{x}, \code{incl_upper}, and \code{upper}. The \code{range} column is a handy
representation of the information stored in the columns from \code{lower} to
\code{upper}, in the same order.
}
\description{
\code{unround()} takes a rounded number and returns the range of the
original value: lower and upper bounds for the hypothetical earlier number
that was later rounded to the input number. It also displays a range with
inequation signs, showing whether the bounds are inclusive or not.

By default, the presumed rounding method is rounding up (or down) from 5.
See the \code{Rounding} section for other methods.
}
\details{
The function is vectorized over \code{x} and \code{rounding}. This can be
useful to unround multiple numbers at once, or to check how a single number
is unrounded with different assumed rounding methods.

If both vectors have a length greater than 1, it must be the same
length. However, this will pair numbers with rounding methods, which can be
confusing. It is recommended that at least one of these input vectors has
length 1.

Why does \code{x} need to be a string if \code{digits} is not specified? In that
case, \code{unround()} must count decimal places by itself. If \code{x} then was
numeric, it wouldn't have any trailing zeros because these get dropped from
numerics.

Trailing zeros are as important for reconstructing boundary values as any
other trailing digits would be. Strings don't drop trailing zeros, so they
are used instead.
}
\section{Rounding}{
 Depending on how \code{x} was rounded, the boundary values can
be inclusive or exclusive. The \code{incl_lower} and \code{incl_upper} columns in the
resulting tibble are \code{TRUE} in the first case and \code{FALSE} in the second.
The \code{range} column reflects this with equation and inequation signs.

However, these ranges are based on assumptions about the way \code{x} was
rounded. Set \code{rounding} to the rounding method that hypothetically lead to
\code{x}:\tabular{ll}{
   \strong{Value of \code{rounding}} \tab \strong{Corresponding range} \cr
   \code{"up_or_down"} (default) \tab \verb{lower <= x <= upper} \cr
   \code{"up"} \tab \verb{lower <= x < upper} \cr
   \code{"down"} \tab \verb{lower < x <= upper} \cr
   \code{"even"} \tab (no fix range) \cr
   \code{"ceiling"} \tab \code{lower < x = upper} \cr
   \code{"floor"} \tab \code{lower = x < upper} \cr
   \code{"trunc"} (positive \code{x}) \tab \code{lower = x < upper} \cr
   \code{"trunc"} (negative \code{x}) \tab \code{lower < x = upper} \cr
   \code{"trunc"} (zero \code{x}) \tab \verb{lower < x < upper} \cr
   \code{"anti_trunc"} (positive \code{x}) \tab \code{lower < x = upper} \cr
   \code{"anti_trunc"} (negative \code{x}) \tab \code{lower = x < upper} \cr
   \code{"anti_trunc"} (zero \code{x}) \tab (undefined; \code{NA}) \cr
}


Base R's own \code{round()} (R version >= 4.0.0), referenced by \code{rounding = "even"}, is reconstructed in the same way as \code{"up_or_down"}, but whether the
boundary values are inclusive or not is hard to predict. Therefore,
\code{unround()} checks if they are, and informs you about it.
}

\examples{
# By default, the function assumes that `x`
# was either rounded up or down:
unround(x = "2.7")

# If `x` was rounded up, run this:
unround(x = "2.7", rounding = "up")

# Likewise with rounding down...
unround(x = "2.7", rounding = "down")

# ...and with `base::round()` which, broadly
# speaking, rounds to the nearest even number:
unround(x = "2.7", rounding = "even")

# Multiple input number-strings return
# multiple rows in the output data frame:
unround(x = c(3.6, "5.20", 5.174))
}
\seealso{
For more about rounding \code{"up"}, \code{"down"}, or to \code{"even"}, see
documentation for \code{round_up()}.

For more about the less likely \code{rounding} methods, \code{"ceiling"}, \code{"floor"},
\code{"trunc"}, and \code{"anti_trunc"}, see documentation for \code{round_ceiling()}.
}
