% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/locate.R
\name{locate}
\alias{locate}
\alias{locate,samc,missing-method}
\alias{locate,samc,ANY-method}
\title{Get cell numbers}
\usage{
locate(samc, xy)

\S4method{locate}{samc,missing}(samc)

\S4method{locate}{samc,ANY}(samc, xy)
}
\arguments{
\item{samc}{A \code{\link{samc-class}} object}

\item{xy}{Any valid input to the y argument of the \code{\link[raster]{extract}} function in the raster package.}
}
\value{
A rasterlayer or a vector
}
\description{
Get cell numbers from raster data
}
\details{
This function is used to get cell numbers from raster data. The numbers used
for origin and destination values in many samc metrics refer to column/row
numbers of the P matrix. For a P matrix derived from raster data, these numbers
would normally line up with the cell numbers of the raster, but this is not
always true. This is the case when the raster contains NA data; the cells
associated with this data are excluded from the P matrix. This causes issues
trying to determine the cell numbers that should be used in analyses.

The \code{\link{locate}} function operates more-or-less like the
\code{\link[raster]{cellFromXY}} function in the raster package, but unlike
\code{\link[raster]{cellFromXY}}, locate properly accounts for NA cells
in identifying cell numbers from coordinate data.

This function can also be used if the samc object was created from matrix inputs
for the resistance, absorption, and fidelity parameters. In this case, the
values in the xy coordinate parameter can be column-row values with the caveat
that (1,1) is the bottom left corner.

The xy parameter can also be excluded. In this case, the function returns a
raster where the values of the cells contains the cell number.

Internally, this function relies on the \code{\link[raster]{extract}} function
from the raster package, and any valid input for the y argument of that function
is valid here.
}
\examples{
# "Load" the data. In this case we are using data built into the package.
# In practice, users will likely load raster data using the raster() function
# from the raster package.
res_data <- samc::ex_res_data
abs_data <- samc::ex_abs_data
occ_data <- samc::ex_occ_data


# Make sure our data meets the basic input requirements of the package using
# the check() function.
check(res_data, abs_data)
check(res_data, occ_data)


# Create a `samc-class` object with the resistance and absorption data using
# the samc() function. We use the recipricol of the arithmetic mean for
# calculating the transition matrix. Note, the input data here are matrices,
# not RasterLayers. If using RasterLayers, the latlon parameter must be set.
samc_obj <- samc(res_data, abs_data, tr_fun = function(x) 1/mean(x))


# Convert the occupancy data to probability of occurrence
occ_prob_data <- occ_data / sum(occ_data, na.rm = TRUE)


# Calculate short- and long-term metrics using the analytical functions
short_mort <- mortality(samc_obj, occ_prob_data, time = 50)
short_dist <- distribution(samc_obj, origin = 3, time = 50)
long_disp <- dispersal(samc_obj, occ_prob_data)
visit <- visitation(samc_obj, dest = 4)
surv <- survival(samc_obj)


# Use the map() function to turn vector results into RasterLayer objects.
short_mort_map <- map(samc_obj, short_mort)
short_dist_map <- map(samc_obj, short_dist)
long_disp_map <- map(samc_obj, long_disp)
visit_map <- map(samc_obj, visit)
surv_map <- map(samc_obj, surv)
}
