% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/tidy_cubist.R, R/tidy_xrf.R
\name{tidy.cubist}
\alias{tidy.cubist}
\alias{tidy.xrf}
\title{Turn regression rule models into tidy tibbles}
\usage{
\method{tidy}{cubist}(x, ...)

\method{tidy}{xrf}(x, penalty = NULL, unit = c("rules", "columns"), ...)
}
\arguments{
\item{x}{A \code{Cubist} or \code{xrf} object.}

\item{...}{Not currently used.}

\item{penalty}{A single numeric value for the \code{lambda} penalty value.}

\item{unit}{What data should be returned? For \code{unit = 'rules'}, each row
corresponds to a rule. For \code{unit = 'columns'}, each row is a predictor
column. The latter can be helpful when determining variable importance.}
}
\value{
The Cubist method has columns \code{committee}, \code{rule_num}, \code{rule}, \code{estimate},
and \code{statistics}. The latter two are nested tibbles. \code{estimate} contains
the parameter estimates for each term in the regression model and \code{statistics}
has statistics about the data selected by the rules and the model fit.

The \code{xrf} results has columns \code{rule_id}, \code{rule}, and \code{estimate}. The
\code{rule_id} column has the rule identifier (e.g., "r0_21") or the feature
column name when the column is added directly into the model. For multiclass
models, a \code{class} column is included.

In each case, the \code{rule} column has a character string with the rule
conditions. These can be converted to an R expression using
\code{\link[rlang:parse_expr]{rlang::parse_expr()}}.
}
\description{
Turn regression rule models into tidy tibbles
}
\examples{
library(dplyr)

data(ames, package = "modeldata")

ames <-
  ames \%>\%
  mutate(Sale_Price = log10(ames$Sale_Price),
         Gr_Liv_Area = log10(ames$Gr_Liv_Area))

# ------------------------------------------------------------------------------

\donttest{
cb_fit <-
  cubist_rules(committees = 10) \%>\%
  set_engine("Cubist") \%>\%
  fit(Sale_Price ~ Neighborhood + Longitude + Latitude + Gr_Liv_Area + Central_Air,
      data = ames)

cb_res <- tidy(cb_fit)
cb_res

cb_res$estimate[[1]]
cb_res$statistic[[1]]
}

# ------------------------------------------------------------------------------

\donttest{
library(recipes)

xrf_reg_mod <-
  rule_fit(trees = 10, penalty = .001) \%>\%
  set_engine("xrf") \%>\%
  set_mode("regression")

# Make dummy variables since xgboost will not
ames_rec <-
  recipe(Sale_Price ~ Neighborhood + Longitude + Latitude +
         Gr_Liv_Area + Central_Air,
         data = ames) \%>\%
  step_dummy(Neighborhood, Central_Air) \%>\%
  step_zv(all_predictors())

ames_processed <- prep(ames_rec) \%>\% bake(new_data = NULL)

set.seed(1)
xrf_reg_fit <-
  xrf_reg_mod \%>\%
  fit(Sale_Price ~ ., data = ames_processed)

xrf_rule_res <- tidy(xrf_reg_fit)
xrf_rule_res$rule[nrow(xrf_rule_res)] \%>\% rlang::parse_expr()

xrf_col_res <- tidy(xrf_reg_fit, unit = "columns")
xrf_col_res
}
}
