% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/stan_betareg.R, R/stan_betareg.fit.R
\name{stan_betareg}
\alias{stan_betareg}
\alias{stan_betareg.fit}
\title{Bayesian beta regression models via Stan}
\usage{
stan_betareg(formula, data, subset, na.action, weights, offset,
  link = c("logit", "probit", "cloglog", "cauchit", "log", "loglog"),
  link.phi = NULL, model = TRUE, y = TRUE, x = FALSE, ...,
  prior = normal(), prior_intercept = normal(), prior_z = normal(),
  prior_intercept_z = normal(), prior_phi = cauchy(0, 5),
  prior_PD = FALSE, algorithm = c("sampling", "optimizing", "meanfield",
  "fullrank"), adapt_delta = NULL, QR = FALSE)

stan_betareg.fit(x, y, z = NULL, weights = rep(1, NROW(x)),
  offset = rep(0, NROW(x)), link = c("logit", "probit", "cloglog",
  "cauchit", "log", "loglog"), link.phi = NULL, ..., prior = normal(),
  prior_intercept = normal(), prior_z = normal(),
  prior_intercept_z = normal(), prior_phi = cauchy(0, 5),
  prior_PD = FALSE, algorithm = c("sampling", "optimizing", "meanfield",
  "fullrank"), adapt_delta = NULL, QR = FALSE)
}
\arguments{
\item{formula, data, subset}{Same as \code{\link[betareg]{betareg}}.}

\item{na.action}{Same as \code{\link[betareg]{betareg}}, but
rarely specified.}

\item{link}{Character specification of the link function used in the model 
for mu (specified through \code{x}). Currently, "logit", "probit",
"cloglog", "cauchit", "log", and "loglog" are supported.}

\item{link.phi}{If applicable, character specification of the link function 
used in the model for \code{phi} (specified through \code{z}). Currently, 
"identity", "log" (default), and "sqrt" are supported. Since the "sqrt"
link function is known to be unstable, it is advisable to specify a
different link function (or to model \code{phi} as a scalar parameter
instead of via a linear predictor by excluding \code{z} from the
\code{formula} and excluding \code{link.phi}).}

\item{model, offset, weights}{Same as \code{\link[betareg]{betareg}}.}

\item{x, y}{In \code{stan_betareg}, logical scalars indicating whether to
return the design matrix and response vector. In \code{stan_betareg.fit},
a design matrix and response vector.}

\item{...}{Further arguments passed to the function in the \pkg{rstan} 
package (\code{\link[rstan]{sampling}}, \code{\link[rstan]{vb}}, or 
\code{\link[rstan]{optimizing}}), corresponding to the estimation method 
named by \code{algorithm}. For example, if \code{algorithm} is
\code{"sampling"} it is possibly to specify \code{iter}, \code{chains},
\code{cores}, \code{refresh}, etc.}

\item{prior}{The prior distribution for the regression coefficients. 
\code{prior} should be a call to one of the various functions provided by 
\pkg{rstanarm} for specifying priors. The subset of these functions that 
can be used for the prior on the coefficients can be grouped into several 
"families":

\tabular{ll}{
  \strong{Family} \tab \strong{Functions} \cr 
  \emph{Student t family} \tab \code{normal}, \code{student_t}, \code{cauchy} \cr 
  \emph{Hierarchical shrinkage family} \tab \code{hs}, \code{hs_plus} \cr 
  \emph{Laplace family} \tab \code{laplace}, \code{lasso} \cr
  \emph{Product normal family} \tab \code{product_normal} \cr
}

See the \link[=priors]{priors help page} for details on the families and 
how to specify the arguments for all of the functions in the table above.
To omit a prior ---i.e., to use a flat (improper) uniform prior---
\code{prior} can be set to \code{NULL}, although this is rarely a good
idea.

\strong{Note:} Unless \code{QR=TRUE}, if \code{prior} is from the Student t
family or Laplace family, and if the \code{autoscale} argument to the 
function used to specify the prior (e.g. \code{\link{normal}}) is left at 
its default and recommended value of \code{TRUE}, then the default or 
user-specified prior scale(s) may be adjusted internally based on the scales
of the predictors. See the \link[=priors]{priors help page} for details on
the rescaling and the \code{\link{prior_summary}} function for a summary of
the priors used for a particular model.}

\item{prior_intercept}{The prior distribution for the intercept. 
\code{prior_intercept} can be a call to \code{normal}, \code{student_t} or 
\code{cauchy}. See the \link[=priors]{priors help page} for details on 
these functions. To omit a prior on the intercept ---i.e., to use a flat
(improper) uniform prior--- \code{prior_intercept} can be set to
\code{NULL}.

\strong{Note:} If using a dense representation of the design matrix 
---i.e., if the \code{sparse} argument is left at its default value of
\code{FALSE}--- then the prior distribution for the intercept is set so it
applies to the value when all predictors are centered.}

\item{prior_z}{Prior distribution for the coefficients in the model for 
\code{phi} (if applicable). Same options as for \code{prior}.}

\item{prior_intercept_z}{Prior distribution for the intercept in the model 
for \code{phi} (if applicable). Same options as for \code{prior_intercept}.}

\item{prior_phi}{The prior distribution for \code{phi} if it is \emph{not} 
modeled as a function of predictors. If \code{z} variables are specified 
then \code{prior_phi} is ignored and \code{prior_intercept_z} and 
\code{prior_z} are used to specify the priors on the intercept and
coefficients in the model for \code{phi}. When applicable, \code{prior_phi}
can be a call to \code{exponential} to use an exponential distribution, or
one of \code{normal}, \code{student_t} or \code{cauchy} to use half-normal,
half-t, or half-Cauchy prior. See \code{\link{priors}} for details on these
functions. To omit a prior ---i.e., to use a flat (improper) uniform
prior--- set \code{prior_phi} to \code{NULL}.}

\item{prior_PD}{A logical scalar (defaulting to \code{FALSE}) indicating
whether to draw from the prior predictive distribution instead of
conditioning on the outcome.}

\item{algorithm}{A string (possibly abbreviated) indicating the 
estimation approach to use. Can be \code{"sampling"} for MCMC (the
default), \code{"optimizing"} for optimization, \code{"meanfield"} for
variational inference with independent normal distributions, or
\code{"fullrank"} for variational inference with a multivariate normal
distribution. See \code{\link{rstanarm-package}} for more details on the
estimation algorithms. NOTE: not all fitting functions support all four
algorithms.}

\item{adapt_delta}{Only relevant if \code{algorithm="sampling"}. See 
\code{\link{adapt_delta}} for details.}

\item{QR}{A logical scalar (defaulting to \code{FALSE}) but if \code{TRUE} 
applies a scaled \code{\link{qr}} decomposition to the design matrix, 
\eqn{X = Q^\ast R^\ast}{X = Q* R*}, where \eqn{Q^\ast = Q \sqrt{n-1}}{Q* =
Q (n-1)^0.5} and \eqn{R^\ast = \frac{1}{\sqrt{n-1}} R}{R* = (n-1)^(-0.5)
R}. The coefficients relative to \eqn{Q^\ast}{Q*} are obtained and then
premultiplied by the inverse of \eqn{R^{\ast}}{R*} to obtain coefficients
relative to the original predictors, \eqn{X}. These transformations do not
change the likelihood of the data but are recommended for computational
reasons when there are multiple predictors. However, because when \code{QR}
is \code{TRUE} the \code{prior} argument applies to the coefficients
relative to \eqn{Q^\ast}{Q*} (and those are not very interpretable) it is
hard to specify an informative prior. Setting \code{QR=TRUE} is therefore
only recommended if you do not have an informative prior for the regression
coefficients.}

\item{z}{For \code{stan_betareg.fit}, a regressor matrix for \code{phi}.
Defaults to an intercept only.}
}
\value{
A \link[=stanreg-objects]{stanreg} object is returned 
for \code{stan_betareg}.

A \link[=stanfit-class]{stanfit} object (or a slightly modified 
  stanfit object) is returned if \code{stan_betareg.fit} is called directly.
}
\description{
Beta regression modeling with optional prior distributions for the 
coefficients, intercept, and auxiliary parameter \code{phi} (if applicable).
}
\details{
The \code{stan_betareg} function is similar in syntax to 
  \code{\link[betareg]{betareg}} but rather than performing maximum 
  likelihood estimation, full Bayesian estimation is performed (if 
  \code{algorithm} is \code{"sampling"}) via MCMC. The Bayesian model adds 
  priors (independent by default) on the coefficients of the beta regression
  model. The \code{stan_betareg} function calls the workhorse
  \code{stan_betareg.fit} function, but it is also possible to call the
  latter directly.
}
\examples{
### Simulated data
N <- 200
x <- rnorm(N, 2, 1)
z <- rnorm(N, 2, 1)
mu <- binomial(link = "logit")$linkinv(1 + 0.2*x)
phi <- exp(1.5 + 0.4*z)
y <- rbeta(N, mu * phi, (1 - mu) * phi)
hist(y, col = "dark grey", border = FALSE, xlim = c(0,1))
fake_dat <- data.frame(y, x, z)

fit <- stan_betareg(y ~ x | z, data = fake_dat, 
                    link = "logit", link.phi = "log", 
                    chains = 1, iter = 250) # for speed
print(fit, digits = 2)
plot(fit)
pp_check(fit)
prior_summary(fit)

}
\references{
Ferrari, SLP and Cribari-Neto, F (2004). Beta regression for 
  modeling rates and proportions. \emph{Journal of Applied Statistics}.
  31(7), 799--815.
}
\seealso{
\code{\link{stanreg-methods}} and 
\code{\link[betareg]{betareg}}.

The vignette for \code{stan_betareg}.
}

